#include <assert.h>
#include "common/ros.h"
#include "common/array.h"
#include "common/network.h"
#include "common/unparser.h"

implementArray(robjs, read_info)

Ros::Ros(int predict) : rset(predict) {
}

Ros::~Ros(){
    rset.clear();
    rset.reclaim();
}

void Ros::clear() {
    rset.clear();
}

int Ros::size() {
    return rset.size();
}

static bool check_ros(robjs const *rsetp) {
    Oref prev = 0;
    int rsize = rsetp->size();
    for (int i = 0; i < rsize; i++) {
	if (!OREF_LT(prev, rsetp->slot(i)))
	    return FALSE;
	prev = rsetp->slot(i);
    }
    return TRUE;
}

bool Ros::encode(Network* net) const {
    ubits32 rsize = rset.size();
    th_assert(check_ros(&rset), "Ros has duplicates or is not sorted");
    if (!net->send_ubits32 (rsize)) return FALSE;
    read_info* r = rset.as_pointer();
    if (! net->send_buffer (r, rsize*sizeof(read_info))) return FALSE;
    return TRUE;
}

bool Ros::decode(Network* net) {
    ubits32 rsize;

    if (!net->recv_ubits32 (&rsize))  return FALSE;

    rset.clear();
    rset._enlarge_by(rsize);
    return (net->recv_buffer(rset.as_pointer(), rsize*sizeof(read_info)));
}

void Ros::sort() {
    int sz = rset.size();
    sort_orefs(rset.as_pointer(), sz);

    // Eliminate eventual duplicates assuming that rset is sorted.
    int from=0, to = 0;
    Oref prev = 0;
    
    // skip portion of array without duplicates.
    while (from < sz) {
       if (!OREF_LT(prev, rset[from])) {
	   // A duplicate. Make to point to the current element
	   // from goes ahead
            to = from++;
            break;
       } else {
            prev = rset[from++];
	    to++;
       }
    }
 
   // Invariant: from > to. 
   // If there was a duplicate we eliminate duplicates in the
   // rest of the array.
   while (from < sz) {
        if (!OREF_LT(prev, rset[from])) 
	    // A duplicate 
            from++;
        else {
            // A non-duplicate
            prev = rset[from]; 
            rset[to++] = rset[from++];
        }
   }

   rset.remove(sz-to);
}

void Ros::unparse (unparser* unp) {
    int rsize = rset.size();

    unp->print("Ros: Rsize=%-d\n", rsize);
    unp->indent(4);
    for (int j=0; j<rsize; j++) {
	unp->print("Oref = %d:%-d \n", 
		OREF_SEGMENT(rset[j]), OREF_INDEX(rset[j]));
    }
    unp->indent(-4);
}
