// \section{Read Object Set Interface}
//
// The "Ros" class keeps track of the information about objects read
// by a transaction. Objects that have been modified are not added here.
// They are present in the MOS set.  _The current interface and implementation
// assume that we have just a single OR_.

#ifndef _ROS_H
#define _ROS_H

#include "common/oref.h"
#include "common/bits.h"
#include "common/basic.h"
#include "common/array.h"

class unparser;
class Network;
typedef Oref read_info;
// Information about read objects sent to the OR. Oref of the read object.

declareArray(robjs, read_info)

// \subsection{The Interface}
class Ros {
  public:
    Ros(int predict);	// Creates empty set with a prediction of the Ros size
    ~Ros();		// Releases all storage
    
    void add_obj(Oref o);
    // requires: object o has just been read and
    //           o has not been added earlier (both for efficieny)
    // modifies: this
    // effects: Inserts the oref o in the read object set

    const Oref* refs();
    // effects: Returns the list of the orefs in the Ros
    //          The length of this array is Ros->size()

    int size();
    // Returns the size of ROS.

    void clear();
    // effects	Remove all entries.

    void sort();
    // effects: Sorts the elements of the array in increasing order of Orefs.
    //          and eliminates eventual duplicates.

// \subsection{Element Generator}
    class Elements {
      public:
        Elements(Ros* set);
        // requires     set is not modified for the lifetime of generator.
        // effects      Generate each element of set.
	//              Each element is generated  exactly once
        //              Elements may be generated in arbitrary order.

	bool get(Oref& oref);
	// modifies this, oref.
	// effects  If more objects are available, modify oref to
	//          contain the pointer to the next oref and
	//          return TRUE.  Else do not modify oref,
	//	    and return FALSE.
      private:
        // Generator state includes the info array, and an index into the
        // the array
        robjs* list;
        int index;
    };

// \subsection{Network Transmission Operations}
    bool encode(Network* net) const;
    bool decode(Network* net);
    // requires -- this to be an empty ROS.

    void unparse (unparser* unp);

// \subsection{Representation}
  private:
    // Ros should never be copied or assigned. So these the copy constructor
    // and assignment operator are being declared here as private but are not
    // implemented 

    Ros(const Ros& r);
    Ros& operator=(const Ros& r);


    robjs rset;
    friend Elements;
};

inline Ros::Elements::Elements(Ros* s) {
    index = 0;
    list = &s->rset;
}

inline bool Ros::Elements::get(Oref& oref) {
    if (index >= list->size()) return FALSE;
    oref     = list->slot(index);
    index++;
    return TRUE;
}

inline void Ros::add_obj(Oref o) {
    rset.append(o);
}

inline const Oref* Ros::refs() {
    return rset.as_pointer();
}

#endif /* _ROS_H */
