#ifndef _SAMPLES_H
#define _SAMPLES_H

// A "Samples" object tracks the mean and standard deviation of a number
// of integer samples.

#include <stdio.h>

class Samples {
  public:
    Samples();
    ~Samples();

    void add(unsigned long s);
    // modifies	"this"
    // effects	Add sample "s".

    void add_bunch(int num, double sum, double sum_square);
    // requires	"sum" and "sum_square" are respectively the sum and
    //		sum of squares of "num" externally generated samples.
    // modifies	"this"
    // effects	Adds them to samples.

    void reset();
    // modifies	"this"
    // effects	Remove all samples.

    unsigned long sum();
    // effects	Return sum of all samples.

    unsigned long count();
    // effects	Return number of samples.

    double mean();
    // requires	"count() >= 1"
    // effects	Return mean of all samples.

    double stddev();
    // requires	"count() >= 2"
    // effects	Return standard deviation of all samples.

    void report(char const* name, FILE* stream = stdout);
    // effects	Report current state to "stream" prefixed with "name".

    void delta(Samples* prev, char const* name, FILE* stream = stdout);
    // requires  prev contains some previous state of this
    // effects	 Reports on samples added since prev state
  private:
    unsigned long total;
    unsigned long num;
    double squared_total;
};

inline Samples::Samples() {
    reset();
}

inline Samples::~Samples() {
}

inline void Samples::add(unsigned long s) {
    total += s;
    squared_total += ((double) s) * ((double) s);
    num++;
}

inline void Samples::reset() {
    total = 0;
    num = 0;
    squared_total = 0.0;
}

inline unsigned long Samples::sum() {
    return total;
}

inline unsigned long Samples::count() {
    return num;
}

#endif /* _SAMPLES_H */
