/* Time open hash table. */

#include <stdio.h>
#include <string.h>
#include "common/array.h"
#include "common/hash.h"
#include "common/intcontrol.h"
//#include "common/map.h"
#include "common/Timer.h"

// All map definitions are set-up to conform to the following signature:
class Table {
  public:
    virtual ~Table();

    virtual void insert(int, int) = 0;
    virtual void replace(int, int) = 0;
    virtual int  contains(int) = 0;
    virtual int  fetch(int) = 0;
    virtual int  fetch(int, int&) = 0;
    virtual void remove(int) = 0;
};
Table::~Table() {}

// Different type definitions
//	CMap		Original chained map by Andrew
//	OMap		Open addressed map
//	TMap		Chained map built using the "Map" template.

// OMap declaration
#define HMAP     OMapRep
#define HKEY     int
#define HVAL     int
#define HCONTROL IntControl
#include "common/hmap.h"

#define HMAP     OMapRep
#define HKEY     int
#define HVAL     int
#define HCONTROL IntControl
#include "common/hmap_p.h"

class OMap : public Table {
  public:
    OMap();
    OMap(int sizehint);
    virtual ~OMap();

    virtual void insert(int, int);
    virtual void replace(int, int);
    virtual int  contains(int);
    virtual int  fetch(int);
    virtual int  fetch(int, int&);
    virtual void remove(int);
  private:
    OMapRep rep;
};

OMap::OMap() : rep() {}
OMap::OMap(int hint) : rep() { rep.predict(hint); }
OMap::~OMap() {}
void OMap::insert(int k, int v) { rep.insert(k, v); }
void OMap::replace(int k, int v) { rep.insert(k, v); }
int  OMap::contains(int k) { return rep.contains(k); }
int  OMap::fetch(int k) { int v; rep.find(k, v); return v; }
int  OMap::fetch(int k, int& v) { return rep.find(k, v); }
void OMap::remove(int k) { rep.remove(k); }

#if 0
// CMap declaration
static int cmap_hash(int x, void* data) { return x; }
static int cmap_compare(int a, int b, void* data) { return (a == b); }
DECLARE_hash(CMapRep,void,void,cmap_hash,cmap_compare,0,0);

class CMap : public Table {
  public:
    CMap();
    CMap(int sizehint);
    virtual ~CMap();

    virtual void insert(int, int);
    virtual void replace(int, int);
    virtual int  contains(int);
    virtual int  fetch(int);
    virtual int  fetch(int, int&);
    virtual void remove(int);
  private:
    CMapRep rep;
};

CMap::CMap() : rep(0) {}
CMap::CMap(int hint) : rep(hint) {}
CMap::~CMap() {}
void CMap::insert(int k, int v) { rep.add((void*) k, (void*) (v+1)); }
void CMap::replace(int k, int v) { rep.store((void*) k, (void*) (v+1)); }
int CMap::contains(int k) { return (rep.fetch((void*) k) ? 1 : 0); }
int CMap::fetch(int k) { return ((int) rep.fetch((void*) k)) - 1; }
int CMap::fetch(int k, int& v) {
    int result = (int) rep.fetch((void*) k);
    if (result) {
	v = result;
	return 1;
    }
    return 0;
}
void CMap::remove(int k) { rep.remove((void*) k); }
#endif

// TMap declaration
//typedef Map<int,int> TMapRep;


// Routines for measuring various things:
static void measure_growth(Table*, char const* label);
static void measure_replace(Table*, char const* label);
static void measure_lookup(Table*, char const* label);
static void measure_remove(Table*, char const* label);

// Number of iterations
static const int count = 10000;

// Routine to report elapsed time per iteration
static void report(char const* title, Timer const& t) {
    printf("%-20s %2.8f seconds\n",
	   title, t.elapsed()/count);
}

int
main() {
    // Open addressing map
    OMap* omap;
    omap = new OMap;
    measure_growth(omap, "omap grow");
    delete omap;
    omap = new OMap(count);
    measure_growth(omap, "omap grow/predicted");
    delete omap;
    omap = new OMap(count);
    measure_replace(omap, "omap replace");
    delete omap;
    omap = new OMap(count);
    measure_lookup(omap, "omap lookup");
    delete omap;
    omap = new OMap(count);
    measure_remove(omap, "omap remove");
    delete omap;

#if 0
    // Chained hash table
    CMap* cmap;
    cmap = new CMap;
    measure_growth(cmap, "cmap grow");
    delete cmap;
    cmap = new CMap(count);
    measure_growth(cmap, "cmap grow/predicted");
    delete cmap;
    cmap = new CMap(count);
    measure_replace(cmap, "cmap replace");
    delete cmap;
    cmap = new CMap(count);
    measure_lookup(cmap, "cmap lookup");
    delete cmap;
    cmap = new CMap(count);
    measure_remove(cmap, "cmap remove");
    delete cmap;
#endif

    return 0;
}

// Fill table with 0..count-1
static void fill(Table* table) {
    for (int i = 0; i < count; i++)
	table->insert(i, i);
}

static void measure_growth(Table* table, char const* label) {
    Timer t;
    t.start();
    fill(table);
    t.stop();

    report(label, t);
}

static void measure_replace(Table* table, char const* label) {
    fill(table);

    Timer t;
    t.start();
    for (int i = 0; i < count; i++) {
	table->replace(i, i);
    }
    t.stop();

    report(label, t);
}

// To prevent compilers from optimizing it all away
static int lookup_result = 0;

static void measure_lookup(Table* table, char const* label) {
    fill(table);

    Timer t;
    int result = 0;
    t.start();
    for (int i = 0; i < count; i++) {
	result ^= table->contains(i);
    }
    t.stop();
    lookup_result ^= result;

    report(label, t);
}

static void measure_remove(Table* table, char const* label) {
    fill(table);

    Timer t;
    t.start();
    for (int i = 0; i < count; i++) {
	table->remove(i);
    }
    t.stop();

    report(label, t);
}
