/*
\section{Timestamp Interface}

The timestamp abstraction is used to generate local timestamps.  These
timestamps have to be augmented with a globlly unique FE or "OR_id" to
make them unique.  _This is a stack abstraction._
*/

#ifndef _TSTAMP_H
#define _TSTAMP_H

extern "C" {
#include <sys/time.h>
}

#include "common/basic.h"
#include "common/bits.h"

class unparser;
class Device;

// \subsection{The Interface}
class Tstamp {
  public:
    Tstamp();
    // effects - Uninitialized timestamp.

    Tstamp(ubits32 seconds, ubits32 microseconds);
    // effects - create a timestamp that has the number of seconds
    // and microseconds since a particular standard time.
  
    Tstamp (Tstamp const&);
    // Copy constructor.

    void operator=(Tstamp const& t);
    // Assignment operator. Copies the contents of t into this.

    bool operator<(Tstamp const& t)  const;
    // effects --- returns true iff this "<" t.
    
    bool operator==(Tstamp const& t)  const;
    // effects --- returns true iff this "==" t.
    
    bool operator!=(Tstamp const& t)  const;
    // effects --- returns true iff this "!=" t.

    bool operator>(Tstamp const& t) const;
    // effects --- returns true iff this ">" t.

    Tstamp operator+(bits32 increment) const;
    // effects --- incrments the timestamp by "increment" microseconds.

    Tstamp next() const;
    // effects --- Increments this and returns it.

// \subsection{Device Transmission Operations}
    bool encode(Device* net) const;
    bool decode(Device*);

    void unparse (unparser* unp);

/*
\subsection{Representation}

The basic idea is is to keep the local time with an accuracy of about
a millisecond. "struct timeval" lets you do better than that.  We keep 
a 32 bit counter that is meant to be set to the number of microseconds 
elapsed, but may be incremented (through "next") to guarantee monotonicity.
*/    
  private:
    bits32 seconds;	// No. of seconds. Same as tv_sec in
    bits32 counter;	// Counter to keep the ts monotonically increasing
};

inline void Tstamp::operator=(Tstamp const& t) {
    seconds = t.seconds;
    counter = t.counter;
}

inline bool Tstamp::operator==(Tstamp const& t) const{
    return (seconds == t.seconds  && counter == t.counter);
}

inline bool Tstamp::operator!=(Tstamp const& t) const{
    return (seconds != t.seconds  || counter != t.counter);
}

inline bool Tstamp::operator<(Tstamp const& t) const{
    if (seconds != t.seconds) return (seconds < t.seconds);
    return (counter < t.counter);
}

inline bool Tstamp::operator>(Tstamp const& t) const{
    if (seconds != t.seconds) return (seconds > t.seconds);
    return (counter > t.counter);
}

#endif /* _TSTAMP_H */

