/* Copyright Barbara Liskov, MIT 1996 */

#include "common/fe_or_msg.h"
#include "common/network.h"
#include "common/other_unix.h"
#include "common/th_assert.h"

#include "config/vdefs/EDGE_MARKING.h"
#include "config/vdefs/PAGING.h"
#include "config/vdefs/INDIRECTION.h"

#include "types/class.h" 

#include "runtime/disphdr.h" 
#include "runtime/obj_class.h" 
#include "runtime/stats.h"
#include "runtime/surr_xref.h"

#include "fe_config.h" 

#include "cache.h"
#include "cacheval.h"
#include "gc.h"
#include "net.h"
#include "swiz.h"

#include "gc_register.h"

#include <assert.h>
#include <string.h>
#include "indirect_table.h"
#include "common/mdebug.h"

static cacheval* cache;         /* The cache */
static cacheval* locache;       /* Low bound of cache */
static cacheval* hicache;       /* High bound of cache */
static cacheval* nextfree;      /* The first free quadword in the cache */

#if INDIRECTION
cacheval* indirect_table; // Keeps track of swizzled references
int itable_cur_size;
int itable_max_size;
#endif

#if PAGING
#include "common/page.h"
Page* pcache;               /* The paged cache */
int numpages;               /* size of paged cache */
#endif

GC* gc;                  /* The garbage collector */

extern "C" {
void* sbrk(ssize_t);
}

bool cache_init() {
  /* Needed: mechanism to limit cache to size of physical memory
     (so that we don't page excessively in our "cache").
     
     */

  /* Die if the following aren't true */

  assert(sizeof(bits16) == sizeof(ubits16));
  assert(sizeof(bits32) == sizeof(ubits32));
  assert((2*sizeof(ubits32)) == sizeof(ubits64));
  assert((2*sizeof(ubits16)) == sizeof(ubits32));

  assert(sizeof(OR_uid) == sizeof(ubits64));
  assert(sizeof(Oref) == sizeof(bits32));
  assert((8 * sizeof(char)) == sizeof(ubits64));
  assert(sizeof(cacheval) == sizeof(ubits64));

  if ((FEConf->cachesize <= 0) || 
      (FEConf->avg_obj <= 0) ||
      (cache != 0))
    return FALSE;

  int slots = FEConf->cachesize / 8;
  cache = new cacheval[slots];
#if PAGING
  numpages = (FEConf->cachesize / (PAGE_SLOTS * sizeof(OR_slot))) + 1;
  pcache = new Page[numpages];
#endif
  locache = cache;
  hicache = locache + slots;
  nextfree = locache;
  init_swiz();
  init_net();
  gc = new GC(locache, hicache);
#if INDIRECTION
  int itable_max_size = slots / 32; // Assuming an average object size of 32
  // XXX Currently code will break if there are more objects than
  // this. This needs to be corrected
  indirect_table = new cacheval[itable_max_size];
  itable_cur_size = 0;
#endif
  return TRUE;
}

#if INDIRECTION
// This code is currently broken
obj get_indirect_storable_pointer(obj o) {
    // Requires: Object dispatch header has been installed

    // Return the pointer to the entry in the indirect table
    // Since this is a proper pointer. Assuming no invalidations and
    // no surrogates, the object must be present in the cache

    // Assert that this is a normal object and is not marked


//     th_assert(!IS_MARKED_POINTER(o), "Marked pointer used for indirection");
//     th_assert(cache_obj_plausible((obj)o), "Object pointer is not in heap");
//     core c = BUMP(core, o, *o);
//     bool new_entry = !c->inf || !c->inf->indirect_pointer;
//     if (!c->inf) {
// 	th_assert (itable_cur_size < itable_max_size - 1, "ITABLE FULL");
// 	c->inf = new_info();
//     }
//     th_assert(cache_obj_plausible((obj)c->inf),"Info pointer not in heap");
//     if (new_entry) {
// 	indirect_table[itable_cur_size].vm_addr = c;
// 	c->inf->indirect_pointer = (char*)&indirect_table[itable_cur_size++]; 
//     }
//     return (obj) &c->inf->indirect_pointer;
}
#endif


#define ZERO_FILL_EVERYTHING FALSE

cacheval* cache_prim_alloc(size_t size) {
#if ZERO_FILL_EVERYTHING
  return gc->alloc(size);
#else
  return gc->raw_alloc(size);
#endif
}

obj cache_oalloc (size_t size) {
  return (obj)gc->alloc(size);
}

void gc_register_root(obj* rootloc) {
   gc->register_root(rootloc);
}

void gc_register_meta_root(obj* rootloc) {
   gc->register_meta_root(rootloc);
}

bool cache_obj_plausible(obj possible) {
  return gc->legal_pointer(possible);
}

obj cache_create_surrogate (Xref x, objtype t) {
  /* requires - xref x refers to an object of type t.
     effects - creates a surrogate of the appropriate size and
               initializes its header.
     errors - none.  A nonsense xref passed to this routine, or
              an inappropriate type, will manifest itself as a 
              problem when the surrogate is used.
  */
  STATS(stats->surrogate_creations++)
  size_t size = (hdr_size(t) * 8) + sizeof(Xref);
  obj result = (obj) cache_oalloc(size);
  inst_surr_hdr(result, t);
  memcpy(SURR_XREF(result), &x, sizeof(Xref));
  return result;
}

obj cache_fetch (surr s, prefetch_hint* h) {
#if EDGE_MARKING || PAGING
  th_fail("Tried to fetch surrogate in system with marked pointers");
  return 0;
#else
  Xref *x = SURR_XREF(s);

  return cache_wait_for(*x, h);
#endif
}

obj cache_fetch_pointer (obj s, prefetch_hint* h) {
#if EDGE_MARKING
    th_assert(IS_MARKED_POINTER(s), "Tried to fetch using an unmarked pointer");
    Xref x = get_pointer_xref(s);
    obj lookedup;
    bool found = swiz->find(x, lookedup);
    if (found == FALSE) {
	return cache_wait_for(x, h);
    }
    else {
	return lookedup;
    }
#else
    th_fail("Tried to fetch with marked pointer in system with surrogates");
    return 0;
#endif
}

void cache_children_init (children_gen g, obj o) {
  g->o = o;
  g->index = 0;
}

bool cache_children_done (children_gen g) {
  /* XXX implement */
  assert(FALSE);
  return TRUE;
}

obj cache_children_next (children_gen g) {
  /* XXX implement */
  assert(FALSE);
  return 0;
}

void make_into_surr (obj o, objtype t, Xref x) {
    // requires --- o to be a persistent object
    // Most probably this is not the procedure you are looking for.
    // Install the surrogate headers on the space pointed to by
    // o. Uses x to change the swizzle table entry

    inst_surr_hdr(OBJ_START(o), t);
    memcpy(SURR_XREF(o), &x, sizeof(Xref));
}

bool cache_obj_invalidate (obj o) {
  /* requires --- o is a pointer to a surrogate or non-surrogate
     persistent object. Actually shrink the object into a surrogate 
     Returns TRUE if invalided the object. False means that the object
     was an empty surrogate
     */

    Xref x;
    core c = cache_real_obj(o, &x);
    if (!c) return FALSE;
    // c now points to an actual object
    th_assert(!IS_NULL_XREF(c->xref), "Xref of persistent object is NULL");
    objtype t =  class_as_objtype(get_obj_class(o));
    make_into_surr ((obj)c, t, c->xref);
    return TRUE;
}

core cache_real_obj (obj o, Xref* xref) {

#if EDGE_MARKING
    // We are not handling edge marking currently XXX
    // if (IS_MARKED_POINTER(o))
    //return 0;
    th_fail("cache_real_obj has to be written for edge marking");
#else
    while (IS_SURR(o)) {
	if (!IS_FULL_SURR(o)){
	    *xref = *SURR_XREF(o);
	    //  Look up in the swizzle table if the object exists in the cache
	    o = swiz_get_object(*xref);
	    if (!o) return NULL;
	    // Found an object in the swizzle table. Continue and check if it
	    // is a surrogate or not
	    continue;
	}
	o = *SURR_FORWARD(o);
    }
    th_assert(o, "Surrogate object pointed to NULL");
#endif
    core c = BUMP(core, o, *o);
    *xref = c->xref;
    return c;
}

bool cache_finished_op() {
  /* XXX should actually check to see if the cache is nearly
     full.  This version is just for testing the gc. */
    return gc->may_collect();
}

void cache_force_gc() {
  gc->collect();
}

obj cache_forwarded(obj possible) {
  /* Return the new address of an object, or 0 if the object has no
     new forwarding address. */
  return gc->forwarded(possible);
}

int cache_occupied_slots() {
  /* Return the number of bytes in a gc halfspace, i.e. half the cache. */
  return gc->occupied_slots();
}

#if PAGING

int cache_find_empty_page(int segnum) {
   /* To do: implement a real cache manager */
   return segnum;
}

#endif /* PAGING */
