/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _FE_CACHE_H
#define _FE_CACHE_H

#ifdef __cplusplus
extern "C" {
#endif

#include "common/oref.h"
#include "common/obj_bitfield.h"
#include "common/basic.h"
#include "common/bits.h"
#include "common/xref.h"

#include "runtime/obj.h"
#include "runtime/surr.h"

#include "cache_fetch.h"
#include "config/vdefs/INSANELY_AGRESSIVE_GC.h"

/*
* The Cache hides details of object fetching, swizzling, and storage
* management at the front end.  There is a single cache per FE.
*/

bool cache_init();
    /* effects - create a cache from configuration information "FEConf", 
                 returns "TRUE".
       errors  - returns "FALSE" if config makes no sense, if the cache
                 can't be created, or if the cache has already been created.
    */


/* cache_fetch is defined in cache_fetch.h */


extern obj cache_oalloc (size_t s);
    /* effects - allocates "s" bytes of zero-filled storage for a new object.
                 The cache is allowed to put additional information
                 before or after the allocated region, and to
                 allocate in arbitrary areas; accordingly, the caller
                 cannot assume that successive calls will produce 
                 contiguous allocation.  
                 
       errors - returns zero if it is not possible to allocate the object
                as requested.
    */

void cache_force_gc();

   /* requires - no current FE operation (so that the execution stack is
                 empty).
      effects - force a garbage collection to occur, snapping 
                all filled surrogates. This means that objects may move 
		in the cache, so any direct pointers may be invalid.  Handles 
                and similar roots are updated to remain accurate.
   */

bool cache_finished_op ();

   /* requires - no current FE operation (so that the execution stack is
                 empty).
      effects - notifies the cache that an operation has finished, so
                that the execution stack is empty (making garbage collection
                easier).  The cache may, but is not required to, 
                perform storage reclamation of some sort before
                returning.  

                This operation is intended to be cheap enough 
                that it should be called at the end of every operation.
 
                It normally returns FALSE.  When it returns TRUE,
                it means that the garbage collector is thrashing or
                has locked up.

                If a garbage collection does occur, objects may move in the 
                cache, so any direct pointers may be invalid.  Handles and 
                similar roots are updated to remain accurate.
   */

void cache_register_gc_root(obj *rootloc);

  /* effects - registers "rootloc" as a cell containing a root "obj".
               The gc will subsequently use the referenced "obj" as a root
               for collections, and will fix up "rootloc" so that it continues
               to point to that "obj" even if it moves in the cache.  This
               means that from the caller's point of view, "rootloc" may
               change unpredictably whenever the gc can run. 

     errors - it is an unchecked runtime error to register a cell
              more than once if that cell may be unregistered.
  */

void cache_unregister_gc_root(obj *rootloc);

  /* requires - "rootloc" has been registered exactly once.

     effects - unregisters "rootloc".  Note that this does not ensure that
               the referenced "obj" is not a root: it may still behave
               as a root if it has been registered from another location.
  */

bool cache_obj_plausible(obj possible);

  /* effects - returns "TRUE" iff "possible" is in a plausible part of
               the cache.  Should be used mostly for assertions.  
  */

core cache_real_obj (obj o, Xref* xref);
/* effects: determines o's xref (which could be the null xref) and puts it
            into "xref". If o is a non-persistent object, returns a pointer to
	    the core
	    If o is a persistent object, it  looks up the swizzle table and
	    if the actual object exists in the cache, returns a pointer to
	    the object, else returns NULL.
*/

bool cache_obj_invalidate (obj o);
    /* effects - Converts "o" to to a surrogate. Returns true if 
                 it did some work, i.e. "o" wasn't already
                 an empty surrogate 
    */


/* 

The cache provides a stack generator for the objects immediately 
reachable from an object o. Duplicates may appear in the generated 
collection. 

*/

typedef struct{obj o; 
               int index;} *children_gen;

void cache_children_init (children_gen g, obj o);
    /* effects - initializes a "children_gen" "g" to generate the objects
                 immediately reachable from "obj" "o".
    */

bool cache_children_done (children_gen g);
    /* effects - returns "TRUE" if all of the children have been generated.
    */

obj cache_children_next (children_gen g);
    /* effects - returns the next child of the object.
       errors - returns zero if there are no more children.
    */

obj cache_forwarded (obj possible);
    /* requires - called only at end of gc.
       effects - returns new address of "possible", or 0 if "possible"
                 is no longer reachable.  Used to update data structures
                 that point into the heap but are not roots.
    */

int cache_occupied_slots();

bool obj_in_heap(obj x);
/* effects Returns TRUE if the object is in the normal frontend heap */

void cache_correct_meta_heap();
/* Correct objects in the meta heap that have pointers to objects
   in the normal heap. If an object o is pointed to by a meta object and
   o is immutable, o is allocated on the meta heap (if o is mutable, it is an
   error).
*/

void cache_check_state();

#ifdef __cplusplus
}
#endif

#endif /* _FE_CACHE_H */
