/* Copyright Barbara Liskov, MIT 1996 */

// This is a fairly conventional stop-and-copy two-space collector,
// using Cheney breadth-first traversal.  It must run only when the
// FE call stack is empty (i.e. between client operations) since it has
// no means of finding roots in stack frames.

// The gc only understands ``real'' objects, so the storage allocated
// should only be used in the appropriate way.  

// If the cache is using surrogates, (i.e. EDGE_MARKING is FALSE,
// so that USE_SURRS is TRUE), the gc 
// expects that every pointer it follows can be interpreted as
// an "obj" ("DV *").  If the methods field is not actually
// a "DV", bad things will happen.  When pointer-marking is used
// instead of surrogates (i.e. EDGE_MARKING is TRUE) then the gc
// tests pointers before using them.  

// The bitfield and size must be accurate or the gc will get confused.  The
// xref field can be NULL instead of being the xref of an object.

// Allocation happens in fromspace.  During a gc, objects are copied to
// tospace.  When there are no more uncopied reachable objects in fromspace,
// a flip occurs, reversing the roles of fromspace and tospace, and the
// garbage collector finishes.

// The garbage collector uses the following non-obvious tricks:

// 1. (Only when there are surrogates, i.e. EDGE_MARKING is FALSE)
//    The low-order bit of a surrogate's method pointer is used
//    as a flag indicating that the surrogate has been copied into 
//    tospace.  We know that all legal addresses have a zero low-order
//    bit, so if an object has one in its low-order bit then it's
//    either a surrogate or things are really screwed up.  If the
//    object really is a surrogate (subtract 1 from "address" and check 
//    for non-zero "t" field in "DV" pointed to by resulting pointer),
//    then the forwarding pointer is in the place where a surrogate usually
//    contains the xref of the absent object.  Note that in general,
//    using the method field of a marked surrogate requires unmarking the
//    surrogate first and then re-marking it afterward.

// 2. (Always) For all non-surrogate objects, the xref field is used for
//    the forwarding pointer.  A bit in the object determines whether the
//    object has been copied or not. An object in fromspace with this bit
//    set indicates that the object  has been copied.

// config includes
#include "config/vdefs/QUIT_ON_THRASH.h"
#include "config/vdefs/SHRINK_OBJECTS.h"
#include "config/vdefs/GC_PROFILE.h"
#include "config/vdefs/GC_EXPOSE.h"
#include "config/vdefs/COUNT_ALLOCS.h"
#include "config/vdefs/GC_SHRINK_BY_SPACE.h"
#include "config/vdefs/GC_SPACE_RECLAIMED.h"
#include "config/vdefs/GC_COLLECTION_TIME.h"
#include "config/vdefs/DO_NOT_RUN_GC.h"
#include "config/vdefs/GC_STOMP_FROMSPACE.h"

// common includes
#include "common/th_assert.h"
#include "common/Timer.h"

// fe includes
#include "fe_config.h"

#include "client/handle_gc.h"
#include "client/meth_handle_tab_iter.h"


#include "runtime/obj_class.h"
#include "runtime/surr.h"
#include "runtime/surr_xref.h"
#include "runtime/transinfo.h"

#include "types/class_class.h"
#include "types/type.h"
#include "types/class_instn_class.h"
#include "types/string_class.h"

// cache includes
#include "gc.h"
#include "swiz.h"

// system includes
#include <sys/types.h>
#include <string.h>

#include "common/mdebug.h"

// Simplified tweaking of optional code.  
// GC_PROFILE turns off everything (but you still have to define
// NDEBUG to turn off assertions).

#if GC_PROFILE
#define GC_STATS FALSE
#define GC_DEBUG FALSE
#define GC_CHECK FALSE
#define SHOW_GC FALSE
#define GC_REASON FALSE
#endif

// GC_EXPOSE turns on everything except the variables gc_debug and gc_vc.

#if GC_EXPOSE
#define GC_STATS TRUE
#define GC_DEBUG FALSE
#define GC_CHECK FALSE
#define GC_VISIBLE_CHECK FALSE
#define SHOW_GC TRUE
#define GC_REASON FALSE
#endif

// Print scads of debugging information?
// Both GC_DEBUG and gc_debug must be TRUE to print.
#ifndef GC_DEBUG
#define GC_DEBUG FALSE
#endif

#if GC_DEBUG
static bool gc_debug = FALSE;
#define DEBUG(x) (gc_debug?(x):0);
#else
#define DEBUG(x) 
#endif /* GC_DEBUG */

// Check objects just before copy and 
// copied data structures just before flip?
#ifndef GC_CHECK
#define GC_CHECK FALSE
#endif

// Print scads of information about check?
// GC_CHECK, GC_VISIBLE_CHECK, and gc_vc must all be TRUE to print.
#if GC_CHECK
#ifndef GC_VISIBLE_CHECK
#define GC_VISIBLE_CHECK TRUE
#endif /* GC_VISIBLE_CHECK defined? */
#endif /* GC_CHECK */

#if GC_VISIBLE_CHECK
static bool gc_vc = FALSE;
#define VCHECK(x) (gc_vc?(x):0);
#else
#define VCHECK(x)
#endif 

// Provide indication that a GC took place?
#ifndef SHOW_GC
#define SHOW_GC FALSE
#endif

// Explain why GC is running?
#ifndef GC_REASON
#define GC_REASON FALSE
#endif

#if GC_DEBUG || GC_VISIBLE_CHECK || SHOW_GC || GC_STATS || GC_REASON || COUNT_ALLOCS
#include <stdio.h>
#endif

#define USE_SURRS !EDGE_MARKING


#include "types/any.h"

// Forward declaration of private routines

#if GC_DEBUG
void print_classname(obj x) {
  objtype c = class_as_objtype(get_any_class(obj_as_any(x)));
  string s = c->name;
  fprintf(stderr, "obj %lx class: %s\n", x, string_charp(s));
}
static void debug_here() {int x = 42;}
#endif

#define LEGAL_PERCENTAGE(x) (((x) >= 0) && ((x) <= 100))

// Into which space does pointer x point?
#define FROMSPACE(x) BETWEEN((cacheval *)x, from_base, from_limit)
#define TOSPACE(x) BETWEEN((cacheval *)x, to_base, to_limit)


// A forwarding pointer is a pointer in fromspace that points 
// into tospace.
#define FORWARDING(x) TOSPACE(x)

// The forwarding pointer clobbers the old "xref" field.
#define GET_FORWARD(x) (*((obj *) &OBJ_XREF(x)))
#define SET_FORWARD(x, n)  *((obj *) &OBJ_XREF(x)) = n;

#define CVSIZE sizeof(cacheval)

// Address aligned with cachevals?
#define ALIGNED(x) (((long int)(x) % CVSIZE) == 0)

// increment an obj by a specified number of slots
#define INCR_BY_SLOTS(o, n) \
                (obj)(((cacheval *)(o)) + (n))

// get the nth slot of object o (returns an obj *)
#define NTH_SLOT(o, n) (obj *)(INCR_BY_SLOTS(OBJ_FIELDS(o), n))

// check whether low bit is set in address
#define IS_LOW_BIT_ON(x) (((long int)(x) % 2) != 0)

// set low bit of address (make it unaligned)
#define SET_LOW_BIT(x) ((long int)x | 0x1)

// clear low bit of address 
#define CLEAR_LOW_BIT(x) ((long int)x & ((long int) -2))

// (Un)mark a surrogate as copied
#define MARK_SURR(x) (*x) = (DV)(SET_LOW_BIT((*x)));
#define UNMARK_SURR(x) (*x) = (DV)(CLEAR_LOW_BIT((*x)));

// A guess about how much above the heap the code goes.  Only matters for
// sanity checks on code addresses in dispatch vectors.
#define TOP_OF_CODE     (top + (10 * 1024 * 1024))  

#define ABOVE_HEAP(x) ((((void *)(x)) > ((void *)top)) && \
                       (((void *)(x)) < ((void *)TOP_OF_CODE)))

#define BELOW_HEAP(x) (((x) > 0) && (((void *)(x)) < ((void *)bottom)))

#define OUTSIDE_HEAP(x) (BELOW_HEAP(x) || ABOVE_HEAP(x))

#define LEGAL_CODE_ADDRESS(x) OUTSIDE_HEAP(x)

#define REASONABLE_DV(x) \
  th_assert(((x)->t == 0) || !IN_HEAP((x)->t), \
	    "Strange value for t in dispatch vector"); \
  th_assert(((x)->boffset <= 0) && ((x)->boffset >= -32), \
	    "Strange value for boffset in dispatch vector"); \
  th_assert(((x)->foffset >= 0) && ((x)->foffset <= 256), \
            "Strange value for foffset in dispatch vector"); \
  th_assert(((x)->offset >= 0) && ((x)->offset <= 256), \
            "Strange value for offset in dispatch vector"); \
  th_assert(LEGAL_CODE_ADDRESS((x)->get_address), \
            "Strange address for get_address in dispatch vector"); \
  th_assert(LEGAL_CODE_ADDRESS((x)->get_class), \
            "Strange address for get_class in dispatch vector");

declareArray(Objs,obj)

implementArray(ObjArray, obj *)
implementArray(Objs,obj)
implementArray(IterArray, obj_iter)
implementArray(RegArray, Region)

declareArray(VictimList, obj)
implementArray(VictimList, obj)

#if GC_STATS
static int gcCount = 0;
static int copied_objs = 0;
static int copied_slots = 0;
static int alloc_since_gc = 0;
static int alloc_slots_since_gc = 0;
#endif

#if COUNT_ALLOCS
static int allocs = 0;
static int alloc_slots = 0;
#endif 

#if USE_SURRS
static obj advance_by_surr_size(obj x);
static obj advance_unswz(obj x);
#endif

cacheval gcNil;
static bool debug_gc_trans = FALSE; // For debugging the transaction mgmt

void cache_check_state() {
    gc->check_state();
}
void cache_correct_meta_heap() {
    gc->check_heap(FALSE, TRUE, 0, TRUE);
    gc->check_heap(FALSE, FALSE, 0, FALSE);
}

int meta_calculated_size = 0; // Keeps track of the actual space allocated
// while traversing the meta objects

void GC::check_object(obj x, int i, bool normal, bool mark,
		      int print_level, bool allow_copy) {
#define PRINT_CHECK(level, x) {if (print_level >= level) x;}

    if (IS_SURR(x)) {
	// Meta objects cannot be surrogates
	if (!normal)
	    fprintf(stderr, "Problem: Object %d (%p) is a surrogate\n", i, x);
	return;
    }
    if (IS_UNSWIZZLED(x))
	return;
    core c = BUMP(core, x, *x);
    if (mark && IS_TRAVERSE_PRIM(c->stamp))
	return;
    if (!mark && IS_TRAVERSE_PRIM(c->stamp) == 0)
	return;
    c->stamp = mark? MARK_TRAVERSE_PRIM(c->stamp):UNMARK_TRAVERSE_PRIM(c->stamp);

    if (!normal) {
	int osize = OBJ_SIZE_BYTES(cao(c));
	meta_calculated_size += osize;
    }
    char const *s = "";
    type tag_ = (type)get_obj_class(x);
    string tname = type_name(tag_);
    if (!tname)
	PRINT_CHECK(1,
		    fprintf(stderr, "Object %d (%p) has no type name\n", i, x))
    else
	s = string_charp(tname);

    if (normal && !obj_in_heap(x))
	fprintf(stderr, "Problem: Object number %d (%p) Type = %s, %p), not in heap\n", i, x, s);
    if (!normal && obj_in_heap(x))
	fprintf(stderr, "Problem: Object number %d (%p) Type = %s, %p), in heap\n", i, x, s);


    if (obj_in_heap((obj)c->methods)) {
	fprintf(stderr, "Methods of Object number %d = %p in the heap\n", i, x);
    }
    Obj_bitfield bf = OBJ_BITFIELD(x);
    
    int slots = OBJ_NUM_SLOTS(x);
    fevalue *fields = OBJ_FIELDS(x);
    if (bf == OBJ_BF_ALLDATA)
	return;
    for (int j = 0; j < slots; j++) {
	if (bf == OBJ_BF_ALLREFS || OBJ_BF_ISREF(bf, j)) {
	    obj px = fields[j].o;
	    if (!px) {
		PRINT_CHECK(1, fprintf(stderr, "Error: Slot %d of object %d (%p with type %s) is NULL\n", j, i, x, s));
		continue;
	    }
	    if (IS_SURR(px)) continue;
	    type pointed = (type)get_obj_class(px);
	    string pointed_tname = type_name(pointed);
	    if (!pointed_tname)
		PRINT_CHECK(1, fprintf(stderr, "Error: Slot %d of object %d (%p with type %s) has no type name\n", j, i, x, s));

	    bool inheap = obj_in_heap(px);
	    if (normal && !inheap)
		fprintf(stderr, "Problem: Slot number %d of object (No. = %d, "
			"Type = %s, %p), not in heap\n", j, i, s, x); 
	    if (!normal && inheap) {
		if (allow_copy && pointed == (type)String) {
		// Move the object to the meta area if it is a string
		    bool saved = normal_heap;
		    normal_heap = FALSE;
		    bool saved_meta = allow_meta;
		    allow_meta = TRUE;
		    fields[j].o = (obj) string_copy((string) px);
		    normal_heap = saved;
		    allow_meta = saved_meta;
		    px = fields[j].o;
		} else {
		    fprintf(stderr, "Problem: Slot number %d of object "
			    "(No. = %d, Type = %s, %p), in heap\n",
			    j, i, s, x); 
		}
	    }
	    check_object(px, i, obj_in_heap(x), mark, print_level, allow_copy);
	}
    }
}

void GC::check_heap(bool normal, bool mark, int print_level, bool allow_copy) {
    int size;
    ObjArray* r;

    meta_calculated_size = 0;
    r = normal? roots: roots_meta;
    size = r->size();
    fprintf(stderr, "Heap check started (Normal = %d)\n", normal);
    for (int i = 0; i < size; i++) {
	obj x = *(r->slot(i));
	check_object(x, i, normal, mark, print_level, allow_copy);
    }
}

static void check_handle_table(bool mark, int print_level) {
    for (HandleIter hi = HandleIter(); hi.ok(); hi.next()) {
	obj o = hi.get_obj();
	gc->check_object(o, -1, obj_in_heap(o), mark, print_level, FALSE);
    }
}

static void check_rw_set(bool mark, int print_level) {
    Transinfo::RW_Iter iter(Fe_Trans, RW_ITER_ALL_SETS);

    obj o;
    Xref xref;
    while (iter.get(o, xref)) {
        if (!o) continue;
	gc->check_object(o, -1, obj_in_heap(o), mark, print_level, FALSE);
    }
}

void GC::check_state() {
    // Check the state of the meta objects and other objects
    static bool first = TRUE;

    check_heap(TRUE, TRUE, 0, FALSE);
    check_heap(TRUE, FALSE, 0, FALSE);
    check_heap(FALSE, TRUE, 0, first);
    check_heap(FALSE, FALSE, 0, FALSE);

    check_handle_table(TRUE, 0);
    check_handle_table(FALSE, 0);

    swiz_check_table(TRUE, 0);
    swiz_check_table(FALSE, 0);

    check_rw_set(TRUE, 0);
    check_rw_set(FALSE, 0);

    first = FALSE;
}

GC::GC(cacheval *low, cacheval *high) {
   // Check architectural assumptions

   // Ensure that we have two regions the same size
   ptrdiff_t size = high - low;
   if ((size % 2) != 0) {
     high -= 1;
     size -= 1;
   }
   // Set initialization value
   gcNil.vm_addr = 0x0;

   // determine whether we should shrink every time gc is done.
   // This will happen if the environment variable SHRINK_EVERY_GC is set
   char *res = getenv("SHRINK_EVERY_GC");
   debug_gc_trans = res != 0;

   // Set up pointers
   bottom = low;
   top = high;
   from_free = low;
   from_base = low;
   from_limit = low + ((size / 2));  
   to_base = low + (size / 2);
   to_limit = low + size;
   from_is_low = TRUE;
   last_free = from_free;
   th_assert(to_limit == top, "Problem with dividing cache into two spaces");

   // Set up arrays to hold info about roots and forbidden regions
   roots = new ObjArray();
   roots_meta = new ObjArray();
   root_iters = new IterArray();
   forbidden = new RegArray();

   // Set up rescue mechanism
   space_determines_rescue = GC_SHRINK_BY_SPACE;

   // Set up mechanism for using space threshold
   free_space_threshold = GC_SPACE_RECLAIMED;
   th_assert(free_space_threshold >= 0, "Negative fraction");
   th_assert(free_space_threshold <= 1, "Can't reclaim more than entire space");

   // Set up thrashing detection
   gc_start_times = new tms;
   times(gc_start_times);
   gc_end_times = new tms;
   times(gc_end_times);
   last_gc_time = 0;
   gc_time_threshold = GC_COLLECTION_TIME;
   th_assert(gc_time_threshold >= 0, "Negative fraction for time threshold");

   ok_to_alloc = TRUE;
 }

GC::~GC() {
   // call destructors?
   ok_to_alloc = FALSE;
   delete roots;
   delete roots_meta;
   delete root_iters;
   delete forbidden;
 }

void GC::register_meta_root(obj* rootloc) {
    th_assert(!IN_HEAP(*rootloc), "Registerred meta object is in the fromheap");
    roots_meta->append(rootloc);
}

void GC::register_root(obj* rootloc) {
    th_assert(IN_HEAP(*rootloc), "Registerred root not in heap");
#if GC_CHECK
   int numroots = roots->size();
   for (int i = 0; i < numroots; i++) {
     if (rootloc == roots->slot(i))
       fprintf(stderr, 
               "New root (0x%lx) duplicates existing root %d\n",
               rootloc, i);
   }
#endif
   roots->append(rootloc);
}

bool GC::legal_pointer(obj possible) {
   bool right_place = IN_FROMHEAP(possible);
   bool aligned = ALIGNED(possible);
   return (right_place && aligned);
 }

void GC::flip() {
#if GC_CHECK
   check_copied();
#endif   
   cacheval *swap;
   swap = from_base;
   from_base = to_base;
   to_base = swap;
   swap = from_limit;
   from_limit = to_limit;
   to_limit = swap;
   swap = from_free;
   from_free = to_free;
   to_free = swap;
   from_is_low = !from_is_low;
   th_assert((from_is_low?((from_base == bottom) && (to_limit == top)):
	              ((to_base == bottom) && (from_limit == top))),
	     "Flipped spaces do not occupy entire cache");
   th_assert(((from_base < from_limit) && (to_base < to_limit)),
	     "Flipped spaces are upside down");
 }

 static inline int bytes2slots(int bytes) {
  return (bytes + 7) / 8;
 }


cacheval* GC::alloc(int size) {

   th_assert(ok_to_alloc, "Tried to alloc during GC");
   cacheval* result;
   int slots = bytes2slots(size);
   result = raw_alloc_slots(slots);
   init_storage(result, slots);
   return result;
 }


void GC::init_storage(cacheval *start, int slots) {
   th_assert(gcNil.vm_addr == 0x0L, "Unexpected value for gcNil");
   for (int i = 0; i < slots; i++) {
     // This is redundant but helps cxx do the right thing
     gcNil.vm_addr = 0x0L; 
     // Actually zero the location (Must be 0 for handle_ and xref field)
     (*(start+i)).vm_addr = gcNil.vm_addr;
   }
 }

cacheval* GC::raw_alloc(int size) {
   th_assert(ok_to_alloc, "Tried to alloc during GC");
   int slots = bytes2slots(size);
   return raw_alloc_slots(slots);
 }

#if COUNT_ALLOCS
// Use hash table to map object-size -> number-of-objects

#include "common/hashfuncs.h"
#include "common/openhashmap.h"

declareOpenHashMap(alloctable,int,int,hash_int,comp_int)
implementOpenHashMap(alloctable,int,int,hash_int,comp_int)
static alloctable alloct;

void print_alloctable() {
    int cumsize = 0;
    int cumcount = 0;
    for (alloctable::Bindings b = &alloct; b.ok(); b.next()) {
	int slotsize = b.key();
	int count = b.val();
	int totalsize = slotsize*count*sizeof(cacheval);
	cumsize += totalsize;
	cumcount += count;
	fprintf(stderr, "%6d word objects = %6d, total size = %7d bytes\n", 
		slotsize, count, totalsize);
    }
    fprintf(stderr, "Number of objects = %d\n", cumcount);
    fprintf(stderr, "Total allocations = %d bytes\n", cumsize);
}
#endif /* COUNT_ALLOCS */

cacheval* GC::raw_alloc_slots(int slots) {

   cacheval* newfree = from_free + slots;
   cacheval* result;
   if (newfree > from_limit) {  
     th_fail("Ran out of storage during an operation "
              "(limitation of current gc implementation)");
     return 0;
   }
   else {
       result = from_free;
       from_free = newfree;
#if GC_STATS
       alloc_since_gc += 1;
       alloc_slots_since_gc += slots;
#endif 
#if COUNT_ALLOCS
       int oldcount = 0;
       alloct.fetch(slots, oldcount);
       alloct.store(slots, oldcount+1);
       allocs += 1;
       alloc_slots += slots;
#endif
   }
   return result;   
}

static bool gc_is_a_good_idea(ptrdiff_t total, 
                              ptrdiff_t remain, 
                              ptrdiff_t recent_alloc,
                              ptrdiff_t alloc) {
  bool cant_sustain_allocs = ((2*recent_alloc) >= remain) &&
      ((4 * remain) < total);
  bool almost_full = ((20 * remain) < total);
#if GC_REASON
  if (cant_sustain_allocs || almost_full)
    fprintf(stderr, "Decided to GC: ");
  if (cant_sustain_allocs)
    fprintf(stderr, "can't sustain allocation rate\n");
  else if (almost_full)
    fprintf(stderr, "cache is almost full\n");
  else if (almost_full && cant_sustain_allocs)
    fprintf(stderr, "fast allocations and nearly-full cache\n");
  fflush(stderr);
#endif
  return (cant_sustain_allocs || almost_full);
}

bool GC::may_collect() {
#if COUNT_ALLOCS
  if (FEConf->debug_level >= 2) {
      fprintf(stderr, "Op allocated %d objects (%d total slots)\n", 
	      allocs, alloc_slots);
  }
  allocs = 0;
  alloc_slots = 0;
#endif
  ptrdiff_t remain = from_limit - from_free;
  ptrdiff_t total = from_limit - from_base;
  ptrdiff_t recent_alloc = from_free - last_free;
  ptrdiff_t alloc = from_free - lastGC_free;
  if (gc_is_a_good_idea(total, remain, recent_alloc, alloc)) {
    collect();
  }
  last_free = from_free;
  th_assert((from_limit - from_free) > 50, "Failed to free enough storage");
  return FALSE;  
}

static clock_t clockdiff (struct tms *later, struct tms *earlier) {
  clock_t usr = later->tms_utime - earlier->tms_utime;
  clock_t sys = later->tms_stime - earlier->tms_stime;

//  XXX The following assertion would be nice but unfortunately it's 
//  wrong when the FE is forked.  
//  If the newly-forked FE is idle until gc time, its
//  user and system times will be less than those values were when they
//  were set in the parent process.  One solution would be to ensure that
//  gc_end_times is only set *after* the fork, but unfortunately this
//  code has no clue about forking.  Another possibility is to create
//  a fork-aware timing abstraction instead of using struct tms directly.

//  th_assert((usr >= 0) && (sys >= 0), "Negative timer");

  return (usr + sys);
}


Timer gc_timer, shrink_timer, actual_gc_timer, rest_gc_timer;

int gc_occurred = 0;
long ngc = 0;
float used_space_before_gc = 0;
float used_space_after_gc = 0;

void GC::collect() {

#if DO_NOT_RUN_GC
    return;
#endif

  if (last_free == from_free)
    return;                   // no allocation, so no point in GC
  ok_to_alloc = FALSE;

  // update gc statistics.
  gc_timer.start();
  used_space_before_gc = 
    (used_space_before_gc*ngc + (from_free - from_base))/(ngc+1);
  ngc++;
  

#ifdef COLLECT_BEFORE_SHRINK
  clear_swiztab();
  internal_collect();
#endif
  ok_to_alloc = TRUE;
  if (space_determines_rescue) {
    int space = from_limit - from_base;
    int freespace = from_limit - from_free;
    float free_fraction = (float)freespace / (float)space;
    // For debugging the shrinking of objects with transaction mgmt
    // Shrink every time gc is done
    if (free_fraction <= free_space_threshold || debug_gc_trans) {
      rescuer();
      ok_to_alloc = FALSE;
      clear_swiztab();
      internal_collect();
      ok_to_alloc = TRUE;
    }
  }  
  used_space_after_gc = 
    (used_space_after_gc*(ngc-1) + (from_free - from_base))/ngc;

  gc_occurred = 1;

  gc_timer.stop();
#if CHECK_META_AND_NORMAL
  check_state();
#endif
  if (FEConf->debug_level) 
      fprintf(stderr, "GC+Shrinking accumulated time  %f\n", 
						gc_timer.elapsed());
}
    
void GC::internal_collect() {
#if SHOW_GC
   fprintf(stderr, "GC start\n");
#endif
#if GC_STATS
   gcCount += 1;
   copied_objs = 0;
   copied_slots = 0;
#endif 
  actual_gc_timer.start();
  start_gc();
  loop {
    obj *curr = uncopied();
    if (curr == 0)
      break;
    *curr = copy_object(*curr);
  }
  actual_gc_timer.stop();
  rest_gc_timer.start();
  Fe_Trans->correct_non_persistent_objects();

  flip();
  end_gc();
  rest_gc_timer.stop();
#if SHOW_GC
   fprintf(stderr, "GC end\n");
#endif
#if GC_STOMP_FROMSPACE
   // Stomp out thw old fromspace with a special value (for debugging)

   ptrdiff_t size = top - bottom;  // Total size of cache
   if ((size % 2) != 0) {
       size -= 1;
   }
   cacheval* ptr = to_base;
   for (int i = 0; i < size/2; i++) {
       ptr[i].pair.value1 = 0x85858585;
       ptr[i].pair.value2 = 0x85858585;
   }
#endif
}

void GC::start_gc() {
  times(gc_start_times);
  clock_t worktime = clockdiff(gc_start_times, gc_end_times);
  if (!space_determines_rescue) {
    float gc_fraction = (float)last_gc_time/(float)worktime;
    if (gc_fraction >= gc_time_threshold)
      rescuer();
  }
#if SHOW_GC || GC_DEBUG
  fprintf(stderr, "last gc time: %d, worktime: %d\n", last_gc_time, worktime);
#endif

  to_free = to_base;

  // roots from handle table
  for (HandleIter hi = HandleIter(); hi.ok(); hi.next()) {
    GC::add_handle_root(hi);
  }
  // roots from future table
  for (FutureIter fi = FutureIter(); fi.ok(); fi.next()) {
    GC::add_future_root(fi);
  }

  // Roots from the read/write set
  GC::scan_rw_set();

  // put in root objects from array
  int numroots = roots->size();
  for (int i = 0; i < numroots; i++) {
#if 0
    for (int j = 0; j < numroots; j++) {
      if ((i != j) && (roots->slot(i) == roots->slot(j)))
        fprintf(stderr, 
                "Roots %d and %d both contain %lx\n", i, j, roots->slot(i));
    }
#endif
    obj *r = (*roots)[i];
    DEBUG(fprintf(stderr, "Root %d: 0x%lx\n", i, *r))
    th_assert(IN_HEAP(*r), "Object not in heap");
    if (!is_copied(*r)) {
      *r = copy_object(*r);  
    }
    else {
      if (FROMSPACE(*r))
        *r = forward_object(*r);
    }
  }
  // set up curr_obj etc for first object
  first_object();

  // XXX run iterators, add those root objects 

  // XXX find roots from stack if invoked during operation
}

// XXX We need to change this so only the written objects
// are treated as roots.  At that point, we will have to
// change the swizzle table cleanup routines to mark the
// read objects as "shrunk" and treat them specially to
// have the concurrency control stuff still work out
// correctly.
void GC::scan_rw_set() {
    Transinfo::RW_Iter iter(Fe_Trans, RW_ITER_GC);

    obj o;
    Xref xref;
    while (iter.get(o, xref)) {
        if (!o) continue;
	if (is_copied(o) && FROMSPACE(o))
	    iter.replace(forward_object(o));
	else if (!is_copied(o))
	    iter.replace(copy_object(o));
    }
}

void GC::add_handle_root(HandleIter hi) {
    // This method should not call hi.next()
    handle h = hi.get_handle();
    th_assert((h > 0), "Iterator yielded an invalid handle");
    obj o = hi.get_obj();
    // th_assert(IN_HEAP(o), "Object not in heap");
    DEBUG(fprintf(stderr, "Handle %d = 0x%lx\n", h, o));
    if (is_copied(o) && FROMSPACE(o))
	hi.replace(h, forward_object(o));
    else if (!is_copied(o))
	hi.replace(h, copy_object(o));   
}

void GC::add_future_root(FutureIter fi) {
    // This method should not call fi.next()
    future f = fi.get_future();
    th_assert((f < 0), "Iterator yielded an invalid future");
    obj o = fi.get_obj();
    // th_assert(IN_HEAP(o), "Object not in heap");
    DEBUG(fprintf(stderr, "Future %d = 0x%lx\n", f, o));
    if (is_copied(o) && FROMSPACE(o))
	fi.replace(f, forward_object(o));
    else if (!is_copied(o))
	fi.replace(f, copy_object(o));   
}

void GC::end_gc() {
  lastGC_free = from_free;  // must happen after flip
  last_free = from_free; 
#if GC_STATS
  fprintf(stderr, "GC %d statistics:\n", gcCount);
  fprintf(stderr, "Allocated %d objects (%d total slots) before GC\n", 
	  alloc_since_gc, alloc_slots_since_gc);
  fprintf(stderr, "Copied %d live objects (%d total slots) during GC\n",
	  copied_objs, copied_slots);
  alloc_since_gc = 0;
  alloc_slots_since_gc = 0;
#endif
  times(gc_end_times);
  last_gc_time = clockdiff(gc_end_times, gc_start_times);
}

#if USE_SURRS

// This routine does not exist when EDGE_MARKING is TRUE
static bool is_copied_surrogate(obj x) {
 if (IS_LOW_BIT_ON((*x))) {
#ifndef NDEBUG
    UNMARK_SURR(x)
    if (!IS_SURR(x)) 
      th_fail("Method address of object has low bit set, not legal address");
    MARK_SURR(x)
#endif
    return TRUE;
  } else
    return FALSE;
}
#endif

obj * GC::uncopied () {
  loop {
    obj *possible = next_ref();
    if (possible == 0)
      return 0;
    if (!is_copied(*possible))
      return possible;
    else {
      th_assert(IN_FROMHEAP(*possible), 
		"Ref returned by next_ref points into tospace");
#if USE_SURRS
      if (is_copied_surrogate(*possible))
        *possible = get_forward_from_surrogate(*possible);
      else {
#endif
        obj forward = GET_FORWARD(*possible);
        th_assert(FORWARDING(forward), 
                  "Forwarding pointer doesn't point into tospace");
        *possible = forward;
#if USE_SURRS
      }
#endif
    }
  }
}

bool GC::ok_to_use(obj *x) {
// Make sure that next_ref never yields nil or a marked pointer 

#if EDGE_MARKING
  return ((*x) && PRESENT(*x));
#else
  // Sanity check
  // th_assert(((*x == 0) || IN_HEAP(*x)), "Wandered out of heap during gc");
  return ((*x) && TRUE);   /* could just return *x but that's sleazy */
#endif
}

obj * GC::next_ref() {
  loop {
    // Try to find a reference in the current object
   
    int numslots = OBJ_NUM_SLOTS(curr_obj);
    obj *result;
    if (OBJ_BF_SPECIAL(curr_bf)) {
      if ((curr_bf == OBJ_BF_ALLREFS) && (curr_slot < (numslots - 1))) {
	curr_slot += 1;
        result = NTH_SLOT(curr_obj, curr_slot);
        if (ok_to_use(result)) 
          return result;
      }
      else if (curr_bf == OBJ_BF_LONG_BF) {
	th_fail("Long bitfields not yet implemented");
      }
    }
    else {
      for (int i = curr_slot + 1; i < numslots; i++) {
	if (OBJ_BF_ISREF(curr_bf, i)) {
	  curr_slot = i;
	  result = NTH_SLOT(curr_obj, curr_slot);
          if (ok_to_use(result))
            return result;
	}
	else {
#if GC_DEBUG	  
	  th_assert((!IN_FROMHEAP(NTH_SLOT(curr_obj, i))),
		    "Non-ref field looks like ref (may be OK)");
#endif
	}
      }
    }

    // Return a pointer to the class field in the dispatch vector.
    // (Required to secure class-objects in use.)
    // However, not all dispatch vectors of an object are fixed.
    // The various dispatch vectors get fixed when the class-object
    // is copied (see "copy_real_object").
    //     if (!curr_class_marked) {
    //       curr_class_marked = TRUE;
    //       DV dv = *curr_obj;
    //       dv->c = dv->get_class(curr_obj);
    //       th_assert(dv->c, "Class field in dispatch vector is NULL");
    //       obj *result = (obj *)&(dv->c);
    //       if (ok_to_use(result) && IN_FROMHEAP(*result)) 
    // 	return result;
    //     }

    // Give up and get another object
    
    if (next_object() == 0)
      return 0;
  }
}

void GC::first_object() {
  curr_obj = (obj)to_base;
#if USE_SURRS
  //XXXX
  // We know that copied empty surrogates and unswizzled objects 
  // contain no pointers, so we skip over them to find the first real object
  while (IS_SPECIAL(curr_obj)){
    if (IS_SURR(curr_obj)) {
      DEBUG(fprintf(stderr, "Skipping surrogate at %lx\n", curr_obj))
      curr_obj = advance_by_surr_size(curr_obj);
    } else {
      th_assert(IS_UNSWIZZLED(curr_obj), "Unknown type of object");
      curr_obj = advance_unswz(curr_obj);
    }
  }
#endif /* USE_SURRS */
  curr_slot = -1;
  // curr_class_marked = FALSE;
  curr_bf = OBJ_BITFIELD(curr_obj);
}

obj GC::next_object() {
  // sets curr_obj, curr_bf, curr_slot

  th_assert((((cacheval *)curr_obj) < to_free), 
	    "Wandered into unallocated storage during gc");

  cacheval *fields = (cacheval *)OBJ_FIELDS(curr_obj);
  int num_slots = OBJ_NUM_SLOTS(curr_obj);
  cacheval *end_of_fields = fields + num_slots;
  curr_obj = (obj)end_of_fields;
    
  th_assert(((cacheval *)curr_obj <= to_free), 
	    "Copied object beyond upper limit of tospace");

  if ((cacheval *)curr_obj == to_free)
    return 0;

#if USE_SURRS

  // Should never see a full surrogate in this code (it should have
  // been snapped out when objects were being copied into tospace).

  while (IS_SPECIAL(curr_obj)){
    if (IS_SURR(curr_obj)) {
      th_assert(!IS_FULL_SURR(curr_obj), "Full surrogate was not snapped out");
      curr_obj = advance_by_surr_size(curr_obj);
    } else {
      th_assert(IS_UNSWIZZLED(curr_obj), "Unknown type of object");
      curr_obj = advance_unswz(curr_obj);
    }
    if ((cacheval *)curr_obj == to_free)
      return 0;
  }

#endif /* USE_SURRS */

  // At this point curr_obj is a non-surrogate

  curr_bf = OBJ_BITFIELD(curr_obj);
  curr_slot = -1;
  // curr_class_marked = FALSE;
  return curr_obj;
}

bool GC::is_copied_real(obj x) {
#if EDGE_MARKING
  th_assert(PRESENT(x), "Tried to copy marked pointer in gc");
#else
  th_assert(!IS_SURR(x), "Tested surrogate as if it were real object");
#endif
  bool tospace = TOSPACE(x);
  core c = BUMP(core, x, *x);
  return (tospace || IS_COPIED_CORE(c));
}

bool GC::is_copied(obj x) {
#if EDGE_MARKING
  if (IS_MARKED_POINTER(x))
    return TRUE;
#else 
  if (is_copied_surrogate(x))
    return TRUE;
  if (IS_SURR(x))
    return FALSE;
#endif
  return is_copied_real(x);
}

obj GC::copy_normal_object(obj fromspace_obj) {
  // requires: not previously copied.  
  // If we snap one or more surrogates out, we must recheck the object
  // to determine whether it has been copied already.  

  bool snapped = FALSE;  // Do we need to recheck object?

  th_assert(IN_FROMHEAP(fromspace_obj),
	    "GC attempted to copy an object not in fromspace");
  th_assert(TOSPACE(to_free),
	    "GC allocation pointer is not in tospace");
  th_assert(ALIGNED(fromspace_obj),
	    "GC attempted to copy a non-aligned object");
  th_assert(ALIGNED(to_free),
	    "Allocation pointer has become misaligned");
  th_assert(!is_copied(fromspace_obj), 
	    "GC attempted to copy an already-copied object");

#if USE_SURRS
  bool surr_flag = FALSE;
  if (IS_SURR(fromspace_obj)) {
    // snap out any filled surrogates.  Go through this loop
    // once for each full surrogate until encountering
    // a copied surrogate, an empty surrogate, or a real object.
    loop {
      // Must test for copied surrogate first because
      // the mark for a copied surrogate screws up the method pointer
      if (is_copied_surrogate(fromspace_obj)) {
        return get_forward_from_surrogate(fromspace_obj);
      }

      // At this point we know that the method pointer should be OK
      else {
        bool s = IS_SURR(fromspace_obj);
        if (s && !IS_FULL_SURR(fromspace_obj)) {
          // empty, uncopied surrogate
          surr_flag = TRUE;
          break;
        }
        else if (!s) {
          // a real object, may have been copied
          if (is_copied_real(fromspace_obj)) {
            return GET_FORWARD(fromspace_obj);
          }      
          else {
            surr_flag = FALSE;
            break;
          }
        }
        else if (s && IS_FULL_SURR(fromspace_obj)) {
          // follow forward pointer (implicitly snapping out surrogates)
          DEBUG(fprintf(stderr, 
                        "Snapping out full surrogate %lx\n", 
                        fromspace_obj))
          obj forward = *SURR_FORWARD(fromspace_obj);
          th_assert(IN_FROMHEAP(forward),
	    "GC attempted to copy an object not in fromspace");
          fromspace_obj = forward;
        }
        else th_fail("Unexpected case when snapping surrogates");
      }
    }
  }

// At this point fromspace_obj is something that needs to be copied.
// surr_flag is TRUE if that something is an empty surrogate.

  // copy an uncopied surrogate
  if (surr_flag)
    return copy_empty_surrogate(fromspace_obj);
#endif /* USE_SURRS */

  // copy an uncopied object
  obj tospace_obj = copy_real_object(fromspace_obj);

  // put the forwarding address into the old object
  if (IN_HEAP(fromspace_obj))
      SET_FORWARD(fromspace_obj, tospace_obj);

  return tospace_obj;
}

#if USE_SURRS
// This routine does not exist if EDGE_MARKING is defined
obj GC::copy_empty_surrogate(obj from_surr) {
  th_assert(!is_copied_surrogate(from_surr), 
            "GC attempted to copy an already-copied surrogate");
#if GC_CHECK
  REASONABLE_DV(*from_surr)
#endif
  cacheval *surrstart = (cacheval *)OBJ_START(from_surr);
  cacheval *surrfinish = (cacheval *)(SURR_XREF(from_surr) + 1);
  size_t bytes = (size_t)((char *)surrfinish - (char *)surrstart);
  obj to_surr = (obj)to_free;
  memcpy(to_free, surrstart, bytes);
#if GC_STATS
  copied_objs += 1;
  copied_slots += (bytes / sizeof(cacheval));
#endif      
  to_free += (bytes / sizeof(cacheval));
  cacheval cv;
  cv.fev.o = to_surr;
  *SURR_XREF(from_surr) = cv.xref;
  // Must get all relevant info from from_surr *before* marking it,
  // since the mark messes up using it as a pointer
  obj surrtype = SURR_TYPE_OBJ(from_surr);
  MARK_SURR(from_surr)
  DEBUG(fprintf(stderr, 
                "Copying surrogate %lx (start: %lx finish: %lx) -> %lx\n", 
	        from_surr, surrstart, surrfinish, to_surr))
  th_assert(!IS_SURR(surrtype),
            "Type object is a surrogate");
  th_assert(!IN_HEAP(surrtype),
            "Surrogate's type object is in the heap");
#if CHECK_META_AND_NORMAL
  //check_object(surrtype, -1, FALSE, TRUE, 0, FALSE);
  // check_object(surrtype, -1, FALSE, FALSE, 0, FALSE);
#endif
  SURR_TYPE(to_surr) = (objtype)surrtype;
  enter_nobj_info(to_surr, *SURR_XREF(to_surr));
  return to_surr;
}
#endif /* USE_SURRS */

obj GC::copy_real_object(obj fromspace_obj) {

#if GG_CHECK
  REASONABLE_DV(fromspace_obj)
#endif

  // copy the object itself to tospace
  int is_unswz = IS_UNSWIZZLED(fromspace_obj);

  cacheval *objstart = (cacheval *)OBJ_START(fromspace_obj);
  int num_slots;
  cacheval *fields;

  if (is_unswz) {
     
    core c = BUMP(core, fromspace_obj, *fromspace_obj);
    OR_obj *o = (OR_obj *) (((cacheval *)c)+1 + FE_obj_headers - OR_obj_headers);
    num_slots = OR_OBJ_SIZE(o);
    fields = (cacheval *)o->slot;
  } else {
    fields = (cacheval *)OBJ_FIELDS(fromspace_obj);
    num_slots = OBJ_NUM_SLOTS(fromspace_obj);
  }
  int delta = (char*)objstart - (char*)fromspace_obj;

#if GC_CHECK && !LAZY_SWIZZLING
  Obj_bitfield bf = OBJ_BITFIELD(fromspace_obj);
  if (!(OBJ_BF_SPECIAL && (bf == OBJ_BF_ALLDATA))) {
    for (int q = 0; q < num_slots; q++) {
      if (OBJ_BF_SPECIAL(bf)) {
        if (bf == OBJ_BF_ALLREFS) {
          th_assert((IN_FROMHEAP(NTH_SLOT(fromspace_obj, q))),
                    "Object in fromspace contains non-fromspace pointers");
        }
        else
          th_fail("GC doesn't understand long bitfields");
      }
      else if (OBJ_BF_ISREF(bf, q)) {
          th_assert((IN_FROMHEAP(NTH_SLOT(fromspace_obj, q))),
                    "Object in fromspace contains non-fromspace pointers");
      }
    }
  }
#endif

  cacheval *objfinish = (cacheval *)(fields + num_slots);
  size_t bytes = (size_t)((char *)objfinish - (char *)objstart);
  obj tospace_obj = (obj)to_free;
  memcpy(to_free, objstart, bytes);

#if GC_STATS
  copied_objs += 1;
  copied_slots += (bytes / sizeof(cacheval));
#endif      

  to_free += (bytes / sizeof(cacheval));
  
  DEBUG(fprintf(stderr, "Copying object %lx\n"
	                "(start: %lx fieldstart: %lx fields: %d finish: %lx)\n"
	                "-> %lx\n", 
	                fromspace_obj, 
	                objstart, fields, num_slots, objfinish, 
  	                tospace_obj))

  // XXX The code below assumes that:
  //  1. the tospace-obj points to the top of the class object.
  //  2. the .c field in the dispatch vectors points to the same class object
  //     -- this is not true of dispatch vectors in classes like IntAny, 
  //     where the .c field points to Int (etc.). 
  //     Any such descrepancies must be fixed in patch_up() called after GC.

  // If the object is a class object, patch up the class
  // pointers in its dispatch vectors.

#ifndef NDEBUG
    if (!is_unswz) {
         obj core_tospace_obj = BUMP(obj, tospace_obj, *tospace_obj);
	 th_assert(!isa_class(core_tospace_obj), "Copied object is a class");
	 th_assert(!isa_objtype(core_tospace_obj), "Copied obj is an objtype");
    }
#endif
  // Mark the fromspace object to have been copied
  core c = BUMP(core, fromspace_obj, *fromspace_obj);
  MARK_COPIED_CORE(c);
  obj newtospace_obj = (obj) ((char*)tospace_obj - (char*)delta);
  c = BUMP(core, newtospace_obj, *newtospace_obj);
  if (IS_PERSISTENT(c))
	enter_nobj_info(newtospace_obj, c->xref);
  return newtospace_obj;
}

obj GC::forward_object(obj fromspace_obj) {
  // requires: previously copied.  

  // Since the copier must have already snapped the link, we know
  // that any real object at the end of the surrogate chain must have
  // been copied; however, we don't know where it was copied to 
  // unless we follow the chain down to the real object again.
						
  th_assert(IN_FROMHEAP(fromspace_obj),
	    "GC attempted to forward using an object not in fromspace");
  th_assert(TOSPACE(to_free),
	    "GC allocation pointer is not in tospace");
  th_assert(ALIGNED(fromspace_obj),
	    "GC attempted to forward using a non-aligned object");
  th_assert(ALIGNED(to_free),
	    "Allocation pointer has become misaligned");
  th_assert(is_copied(fromspace_obj), 
	    "GC attempted to forward using an uncopied object");

#if USE_SURRS
  if (is_copied_surrogate(fromspace_obj)) {
    return get_forward_from_surrogate(fromspace_obj);
  }

  if (IS_SURR(fromspace_obj)) {
    // snap out any filled surrogates.  Go through this loop
    // once for each full surrogate until encountering
    // a copied surrogate or a (copied) real object.
    loop {
      // Must test for copied surrogate first because
      // the mark for a copied surrogate screws up the method pointer
      if (is_copied_surrogate(fromspace_obj)) {
        return get_forward_from_surrogate(fromspace_obj);
      }

      // At this point we know that the method pointer should be OK
      else {
        bool s = IS_SURR(fromspace_obj);
        if (s && !IS_FULL_SURR(fromspace_obj)) {
          // empty, uncopied surrogate
          th_fail("Should not have uncopied surrogate in forward chain");
        }
        else if (!s) {
          // a real object, may have been copied
          if (is_copied_real(fromspace_obj)) {
            return GET_FORWARD(fromspace_obj);
          }      
          else {
            th_fail("Should not have uncopied object in forward chain");
          }
        }
        else if (s && IS_FULL_SURR(fromspace_obj)) {
          // follow forward pointer (implicitly snapping out surrogates)
          DEBUG(fprintf(stderr, 
                        "Snapping out full surrogate %lx\n", 
                        fromspace_obj))
          fromspace_obj = *SURR_FORWARD(fromspace_obj);
        }
        else th_fail("Unexpected case when snapping surrogates");
      }
    }
  }
#endif /* USE_SURRS */

  th_assert(is_copied_real(fromspace_obj),
            "Cannot forward an uncopied object");
  return GET_FORWARD(fromspace_obj);
}

#if USE_SURRS
// This routine does not exist if EDGE_MARKING is TRUE
static inline obj advance_by_surr_size(obj x) {
  return INCR_BY_SLOTS(x, SURR_SLOTS);
}

static inline obj advance_unswz(obj x) {
  core c = BUMP(core, x, *x);
  OR_obj *o = (OR_obj *) (((cacheval *)c)+1 + FE_obj_headers - OR_obj_headers);
  int num_slots = OR_OBJ_SIZE(o);
  return INCR_BY_SLOTS((cacheval *)o->slot, num_slots);
}

#endif /* USE_SURRS */

static inline obj advance_obj(obj x) {
  int num_slots = OBJ_NUM_SLOTS(x);
  return INCR_BY_SLOTS(OBJ_FIELDS(x), num_slots);
}

void GC::check_copied() {
#if GC_CHECK
  // check integrity of tospace just before flip
  obj o = (obj)to_base;
  while (o < (obj)to_free) {

    // Still sane pointer?
    th_assert((ALIGNED(o) && TOSPACE(o)), 
              "Object pointer not aligned in tospace");

#if USE_SURRS
    // Surrogate?
    if (IS_SURR(o)) {
      REASONABLE_DV(*o)
      VCHECK(fprintf(stderr, "%lx surrogate\n", o))
      o = advance_by_surr_size(o);
    }

    else {
#endif /* USE_SURRS */
      REASONABLE_DV(*o);
      int num_slots = OBJ_NUM_SLOTS(o);
      VCHECK(fprintf(stderr, "%lx object with %d fields\n", o, num_slots));
      o = advance_obj(o);
    }
#if USE_SURRS
    }
#endif /* USE_SURRS */
  }
  fflush(stderr);

  // check on root objects from array
  int numroots = roots->size();
  VCHECK(fprintf(stderr, "checking roots from array: "))
  for (int i = 0; i < numroots; i++) {
    VCHECK(fprintf(stderr, "%d ", i))
    obj *r = (*roots)[i];
    th_assert((TOSPACE(*r)), "Uncopied root");
    core c = BUMP(core, r, *r);
    th_assert((c->handle_ < 100000), "Unlikely handle number");
  }
  VCHECK(fprintf(stderr, "\n"))
#endif /* GC_CHECK */
}

obj GC::forwarded_normal_object(obj o) {
  th_assert(!ok_to_alloc, 
            "Looked for a forwarding pointer when no gc took place");
  th_assert(IN_FROMHEAP(o), "Trying to forward with forwarded object");

#if USE_SURRS
  DV dv = (DV)CLEAR_LOW_BIT(*o);
  // Test if the object is a surrogate but with the unmarked
  // DV. Should be kept consistent with IS_SURR in runtime/surr.h
  if (dv->t != 0 && dv->foffset != 1) {
      if (IS_LOW_BIT_ON(*o)) 
          // If the low bit is set the surrogate o was copied
          // the forwarding pointer is in the xref field of the
          // This code should be kept consistent with SURR_XREF
          // in runtime/surr_xref.h
          return *((obj *)((obj)((char *)o + dv->offset) + 1));
      else 
          return 0;
  } else {
#endif
    obj forward = GET_FORWARD(o);
    if (FORWARDING(forward))
      return forward;
    else
      return 0;
#if USE_SURRS
  }
#endif
}

#if USE_SURRS
// This routine does not exist when EDGE_MARKING is TRUE
obj GC::get_forward_from_surrogate(obj surr) {
  th_assert(is_copied_surrogate(surr), 
            "Tried to get forwarding pointer from uncopied object");
  UNMARK_SURR(surr)
  obj result = *((obj *)SURR_XREF(surr));
  MARK_SURR(surr)
  th_assert(FORWARDING(result), 
		"Forwarding pointer does not point into tospace");
  return result;
}
#endif /* USE_SURRS */

void GC::rescuer() {
#if QUIT_ON_THRASH
  th_fail("Thrashing");
#endif /* QUIT_ON_THRASH */
#if SHRINK_OBJECTS
  shrink_timer.start();
  shrink_objects();
  shrink_timer.stop();
#endif /* SHRINK_OBJECTS */
}

int GC::occupied_slots() {
  return (int)(from_free - from_base);
}

bool obj_in_heap (obj x) {
  return gc->in_heap(x);
}
