/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _GC_H
#define _GC_H

// common includes
#include "common/basic.h"
#include "common/array.h"

// fe includes 
#include "runtime/obj.h"
#include "client/handle_gc.h"

// cache includes 
#include "cacheval.h"
#include "region.h"

// system includes
#include <sys/times.h>

typedef obj (*obj_iter)();
class Objs;

declareArray(ObjArray, obj *)
declareArray(RegArray, Region)
declareArray(IterArray, obj_iter)

class GC {
 public:

    GC(cacheval *low, cacheval *high);
    // effects -- creates new gc'ed region of cache.

    ~GC();
    // effects -- reclaim gc resources.  This does not free
    //            the corresponding region of cache.

    cacheval* alloc(int bytes);
    // effects -- allocates a contiguous chunk of storage if possible,
    //            filled with gcNil to ensure correct gc behavior.

    cacheval* raw_alloc(int bytes);    
    // effects -- allocate storage without putting in gcNil.  Use with 
    //            caution.

    void init_storage(cacheval*, int bytes);  
    // effects -- put gcNils into storage.  For use with "raw_alloc" and
    //            to be used with similar caution.

    void collect();
    // effects -- reclaim cache storage. 

    bool may_collect();
    // effects -- advise that this may be a good time to reclaim cache storage.
    //            Returns "TRUE" if the gc is thrashing, "FALSE" otherwise.

    bool in_tospace(obj x) {
        return ((((cacheval *)x) >= to_base) && (((cacheval *)x) < to_limit));
    }
    // effects -- returns "TRUE" if the object is in tospace


    bool legal_pointer(obj possible);
    // effects -- returns "TRUE" if the object is in the allocated region
    //            of fromspace.

    void register_root(obj* rootloc);
    // effects -- register an address containing an "obj".  The "obj" stored
    //            in that location must be treated as a root for gc, and 
    //            the location must be updated whenever the gc moves the
    //            object.

    void register_meta_root(obj* rootloc);
    // effects -- Same as register_root except that rootloc is in the meta heap

    obj forwarded(obj possible);
    // requires -- must be called only at the end of a garbage collection (as
    //             a callback when called by the gc).
    // effects -- if the object "possible" is live, returns its new address.
    //            Otherwise returns 0.  Note that the returned address
    //            might be the same as the old address.

    void check_state();
    // effects: Checks that the meta heap and normal heap do not have pointers
    //          going across the two heaps

    bool in_heap(obj x);
    // effects: Returns true if x is in the theta heap

    int occupied_slots();
    // effects -- return the number of slots currently allocated.

    void check_object(obj x, int i, bool normal, bool mark, int
			     print_level, bool allow_copy);
    // effects: Checks if the objects x and objects accessible from x are in
    //          the correct heap. If normal is TRUE, then all these objects
    //          must be in the normal heap, else in the meta heap. Mark
    //          indicates whether to mark or unmark the object. Allow copy
    //          indicates whether immobjects objects from the normal heap
    //          can be copied from the meta to normal heap

    void check_heap(bool normal, bool mark, int print_level, bool allow_copy);
    // Check the given heap (normal or meta). See specs of check_object for
    // more details

    cacheval* raw_alloc_slots(int); 
    // effects actually allocate an object with the given number of slots.

 private:
    // Whole region
    cacheval* bottom;
    cacheval* top;
    bool ok_to_alloc;

    // Semispaces (fromspace and tospace)
    cacheval* from_base;         
    cacheval* from_free;
    cacheval* from_limit;
    cacheval* to_base;
    cacheval* to_free;
    cacheval* to_limit;
    bool from_is_low;

    // What was the free pointer the last time we checked for a GC?
    cacheval* last_free;

    // What was the free pointer the last time we finished a GC?
    cacheval* lastGC_free;

    // Roots
    ObjArray* roots;
    ObjArray* roots_meta;
    IterArray* root_iters;

    // special value
    cacheval gcNil;

    // Forbidden areas
    RegArray* forbidden;

    // Auxiliary data structures for tracking progress of gc
    obj curr_obj;
    bool curr_class_marked; // if the class of curr_obj has been looked into
    Obj_bitfield curr_bf;
    int curr_slot;
    bool in_object;

    // There are two mutually exclusive mechanisms for determining when
    // the gc is wasting its time and should invoke its rescue mechanism.

    // One mechanism looks at space reclaimed and calls the rescuer
    // if a collection doesn't free up enough space (as a percentage of the
    // cache).  The other mechanism checks the time spent in gc and
    // calls the rescuer when that exceeds a certain percentage of the
    // inter-gc time.

    // Choose mechanism 
    bool space_determines_rescue;

    // Data for detecting inadequate space reclaimed
    float free_space_threshold;  // as fraction of fromspace

    // Data for detecting excessive time spent in gc
    struct tms *gc_end_times;
    struct tms *gc_start_times;
    clock_t last_gc_time;
    float gc_time_threshold;      // as fraction of inter-gc time

    // Internal methods

    obj forwarded_normal_object(obj o);
    // requires: o is an object in the normal heap (+ requires of GC::forwarded)
    // effects: Same as GC::forwarded 

    void rescuer();               // invoked when the gc is in trouble
    void internal_collect();      // does the actual gc with no check
                                  // for whether it should run or not
    void flip();                  // reverse to and from spaces
    void start_gc();              // set up data structures to do gc
    obj * uncopied();             // ref to next obj to copy 
                                  // (returns 0 when no more)
    void first_object();          // set up first object in tospace
    obj next_object();            // return next obj in tospace 
                                  // (0 when no more)
    obj * next_ref();             // ref to next obj in current object 
				  // *ref must be in fromspace
                                  // (0 when no more)
    void mark_object(obj);        // mark fromspace object as copied
    bool is_copied(obj);          // is object marked as copied?
    bool is_copied_real(obj);     // is real object marked as copied?
    obj copy_object(obj);         // copy object into tospace, 
                                  // return new location
    obj copy_normal_object(obj);  // Like copy_object but requires obj to be a
                                  // normal object, 
    obj copy_real_object(obj);    // actually copy object header and fields
    obj copy_empty_surrogate(obj);// copy surrogate
    info copy_info(obj,obj);      // copy an info object
    obj forward_object(obj);      // find address of copied object
    void add_handle_root(HandleIter); // set up roots from handle table
    void add_future_root(FutureIter); // set up roots from future table
    void scan_rw_set();		  // Scan roots from read/write set for xaction
    void scan_method_handle_table(); // Scan roots from method handle table
    obj get_forward_from_surrogate(obj); // get forwarding pointer
    void check_copied();          // check integrity of copied data
    void end_gc();                // clean up
    bool ok_to_use(obj*);         // filter out certain irrelevant pointers
    
    };

#define BETWEEN(x, l, h) (((x) >= l) && ((x) < h))

// Legal heap pointers?
#define IN_FROMHEAP(x) BETWEEN((cacheval *)x, from_base, from_free)
#define IN_TOHEAP(x) BETWEEN((cacheval *)x, to_base, to_free)
#define IN_HEAP(x) (BETWEEN((cacheval*) x, bottom, top))
// Controls whether the normal and meta heap should be checked after GC
#define CHECK_META_AND_NORMAL FALSE

inline bool GC::in_heap(obj x) {
   return IN_HEAP(x);
}

inline obj GC::forwarded(obj o) {
    if (in_heap(o))
	return forwarded_normal_object(o);
    return o;
}

inline obj GC::copy_object(obj fromspace_obj) {
  if (in_heap(fromspace_obj))
      return copy_normal_object(fromspace_obj);
#if CHECK_META_AND_NORMAL
      check_object(fromspace_obj, -1, FALSE, TRUE, 0, FALSE);
    check_object(fromspace_obj, -1, FALSE, FALSE, 0, FALSE);
#endif
    return fromspace_obj;
}

extern int gc_occurred;
extern GC *gc;

#endif /* _GC_H */

