/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _NET_H
#define _NET_H

/* 
  net.h represents the part of the cache subsystem that interacts with
  the ORs across the network.  Only the cache subsystem should use 
  this interface.
*/

#include "common/xref.h"
#include "common/or_stat.h"
#include "common/bits.h"
#include "common/or_num.h"
#include "common/fe_or_msg.h"
#include "common/other_unix.h"

#include "fe/runtime/obj.h"
#include "fe/runtime/commit.h"
#include "fe/runtime/invalidation.h"

#ifdef __cplusplus
extern "C" {
#endif


class prefetch_hint;
class Network;
// Specifies whether the FE should block on the OR/FE boundary or not
enum Block_Option {
    FE_BLOCK,
    FE_NO_BLOCK
};

// Struct for each OR on which the FE should wait and the messages that it
// should block on
struct OR_desc {
    OR_num or;             // OR to wait from
    ubits32* msgtype; // Messages to block for (disjunction)
    int size; // Size of the above array
};


void init_net();
  /* effects - Set up net data structures.  
               Must be called before first call to any other routine
               defined in this file.
  */

OR_num init_locator(struct sockaddr_in *OR_spec);
  /* effects - Start up OR locator.  Returns OR # of OR at network address
               and port given in OR_spec.
               Must be called before first call to any other routine
               defined in this file except for init_net.
     errors  - Return 0 if locator can't be started, or if given OR can't be
               located.
  */

obj cache_wait_for (Xref x, prefetch_hint* h);
  /* effects - send request to appropriate OR and wait for 
               object(s) to come back.  Return the FE version of 
               the object corresponding to "x".

	       The returned object should be suitable for use by
	       the Theta runtime.  This means that under node marking,
	       the object should be fully swizzled, and that under
	       edge marking, the references in the object should be
	       in a format "suitable" for the runtime system.

               The called routine is guaranteed that the caller will
               not be doing anything else until this call returns.
               Accordingly, this routine may do various other pieces
               of work while waiting for the object to come from the OR,
               and it may process prefetched objects before returning
               the requested object.

     errors - return 0 if the requested object can't be fetched.
  */

extern obj cache_fetch_root (OR_num or_num, prefetch_hint* h);
    /* effects - returns the fully-swizzled root object for the 
                 specified OR, fetching it from that OR if necessary.
                 Hint "h" works as with "cache_fetch".
       errors - returns zero if there is no OR with the given number.
    */

#ifdef __cplusplus
}
#endif

class Network;	       

extern Network* cache_get_OR_conn (OR_num or_num);
  /* effects - return a non-zero Network* connected to the specified OR.
     errors - return 0 if the connection can't be created or connected.
  */

bool read_from_net(OR_desc* ods, int orsetsize, Block_Option option,
		   bool* aborted, bool allow_longjmp, void* env,
		   Msg_Wait_Type wt);
    // modifies aborted
    // effects: Keeps reading data from the servers till there is no more
    //          available. Sets the "message_from_server" global variable
    //          to FALSE on exit (while maintaining the invariant that it
    //          is TRUE when there is a message from a server).
    //          If option is FE_NO_BLOCK, then returns when no more data is
    //          available from the servers.
    //          If option is FE_BLOCK, it waits for messages specified in
    //          ods to be received (on all the ORs given in ods)
    //          env is used by various dispatcher routines for handling an
    //          incoming message

    //          If allow_longjmp is TRUE and the transaction aborts, this
    //          procedure does not return. Otherwise, if the transaction
    //          aborts, it sets *aborted to TRUE 
    // returns  (if it returns) TRUE if no failure occured while reading
    //          from the network 


extern bool get_or_stats(OR_num or_num, or_stat& stat);
// effects: Get the statistics from or_num into stat

extern or_stat initial_stat;

extern bool message_from_server;
// This variable is set to TRUE whenever the FE receives an asynchronous
// message from a server. 
// The semantics of theis variable are:
// If there is a message from the server, then this variable is TRUE

#endif /* _NET_H */
