/* Copyright Barbara Liskov, MIT 1996 */

#include "common/basic.h"
#include "common/xref.h"
#include "common/th_assert.h"
#include "common/array.h"
#include "common/fast_alloc.h"
#include "common/fail.h"
#include "common/openhashmap.h"
#include "common/prefetch.h"
#include "common/Timer.h"


// fe includes 
#include "runtime/disphdr.h"
#include "runtime/transinfo.h"
#include "runtime/obj.h"
#include "runtime/stats.h"
#include "runtime/surr_xref.h"
#include "boot/wellknown.h"
#include "types/string_class.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "fe_config.h"
#include "common/iter.h"
#include "common/link.h"

// fe/cache includes 
#include "swiztab.t"
#include "cache.h"
#include "gc.h"
#include "cache_internal.h"
#include "swiz.h"
#include "net.h"
#include "indirect_table.h"


// vdefs includes 
#include "config/vdefs/DEBUG_SWIZ.h"
#include "config/vdefs/DEBUG_PREFETCH.h"
#include "config/vdefs/DEBUG_SHRINK.h"
#include "config/vdefs/LAZY_SWIZZLING.h"
#include "config/vdefs/ADAPT_PREFETCH.h"
#include "config/vdefs/EDGE_MARKING.h"
#include "config/vdefs/SHRINK_FRACTION.h"
#include "config/vdefs/SHRINK_SUBPAGES.h"
#include "config/vdefs/PAGING.h"
#include "config/vdefs/NDEBUG.h"


//  system includes 
#include <stdlib.h>
#include <sys/uio.h>

#if DEBUG_SWIZ
#include <stdio.h>
static void debug_here() {int x = 42;}
#endif

extern "C" {
void *memcpy(void*, const void*, size_t);
}


#include "common/mdebug.h"


#if ADAPT_PREFETCH
// Global variable that holds the statistic on the number of objects used.
// Defined in net.cc
extern long p_nused;
#endif


// The swizzle table maps xrefs to virtual memory addresses for
// objects that are present in the FE cache.

SwizTab  swiz(1);           // swizzle table (xref -> VM addr) 

// Keeps track of number of persistent objects that have been shrunk
// in the current shrinking phase (and read/not read by current transaction)
static int shrunk_read_persistent;
static int shrunk_unread_persistent;


//
// Fifo of subpages used to do clustered cache evictions.
//
SubpageFIFO subpage_fifo;


void init_swiz() {
    swiz.predict(FEConf->cachesize/(4*FEConf->avg_obj));
}


/*

  SWIZZLING ONE REFERENCE

  This is the code where the real work of swizzling takes place, 
  one reference at a time.  There is a different procedure for
  each of the implemented swizzling techniques.

*/

#if EDGE_MARKING
static inline void edge_swiz_one_ref(cacheval *ref, int or_num) {
  /* effects - mutate the oref *ref so that it's a marked pointer
               containing a full xref.
  */
    // XXX This reference should be added to the outlist for OR #or_num
    //     for indirect protection.
    // I believe that this function does no work ?? -- Atul
    Xref x = ref->xref;
    set_pointer_xref(ref, x);
}
#endif // EDGE_MARKING

#if !(EDGE_MARKING)

static void helper_surr_swiz_one_ref(obj &o, objtype t, Xref x) {
  /* effects - create surrogate for object o, and return it in o.
  */
  int dhsize = hdr_size(t);
  o = (obj) gc->raw_alloc_slots(dhsize+1);
  if (dhsize == 1) 
    oac(o)->methods = *t->surr_DH;
  else 
    memcpy(o, t->surr_DH, dhsize * sizeof(DV));
  *SURR_XREF(o) = x;
}
  


static inline void surr_swiz_one_ref(cacheval *ref, int or_num, objtype t) {
  /* effects - create surrogate for the object indicated, if necessary.
               Mutate *ref to point to the cache representation of the
               object (surrogate or non-surrogate).
  */
    obj *obj_info = swiz.storeOrFind(ref->xref);     
    // XXX This reference should be added to the outlist for OR #or_num
    //     for indirect protection.
 
    if (*obj_info == 0) 
      // Install new surrogate for the object.
      helper_surr_swiz_one_ref(*obj_info, t, ref->xref);
    

    ref->vm_addr = (void *)*obj_info;
}
#endif

static inline void cache_swizzle_one_ref(cacheval *ref, int or_num, objtype t) {
    /* requires - *ref is an oref that has not been translated 
                  to a memory pointer.  The or_num indicates the OR
	          from which the oref comes.
       effects - mutates *ref to be a memory pointer to either the 
                 referenced object or a surrogate for the referenced
	         object (the surrogate is created if necessary).
       errors - none.  A nonsense field passed into this procedure
                will manifest itself as a problem at some subsequent
                fetch going through the surrogate created by this procedure.
  */
    STATS(stats->swizzled_refs++)
#if EDGE_MARKING
    edge_swiz_one_ref(ref, or_num);
#else // default node marking
    surr_swiz_one_ref(ref, or_num, t);
#endif 
}

/* 

  SWIZZLING OBJECTS

  This is the code that deals with multi-field entities,

*/

void cache_prim_swizzle (obj unswiz, int or_num, OR_obj *or_rep, class_ c) {
    /* effects - fully swizzle the object unswiz, in place. 
       errors - return 0 if the object can't be swizzled. 

       The pointer "unswiz" actually points into the nulls left at the
       beginning of the object for the future dispatch header, while 
       "or_rep" points to the beginning of the embedded "OR_obj".
       "c" gives the class of the object (note that the class field in the
       "OR_obj" might have been overwritten, hence the class cannot be
       derived from there).
       */
    
    or_rep = (OR_obj*) (((cacheval *) or_rep) + FE_obj_headers - OR_obj_headers);
    // We are incrmenting the rep by the extra slots in an FE object
    Obj_bitfield bf = OR_OBJ_BITFIELD(or_rep);
    
    if (bf != OBJ_BF_ALLDATA) {
      // If object only has non-ref fields don't swizzle.
      th_assert(bf != OBJ_BF_LONG_BF, 
		"Swizzling of long objects not implemented");
      th_assert(bf == OBJ_BF_ALLREFS || !OBJ_BF_SPECIAL(bf),
		"Unrecognized special bitfield");
      
      // Swizzle all the references in the object.
      cacheval *fieldstart = (cacheval *)or_rep->slot;
      int numslots = OR_OBJ_SIZE(or_rep);
      objtype *slottypes = computeSlottypes(c, numslots, bf);
      int i;
      
      if (bf == OBJ_BF_ALLREFS) {
	for (i = 0; i < numslots; i++) 
	   cache_swizzle_one_ref(fieldstart + i, or_num, *slottypes);
      } else {
        for (i = 0; i < numslots; i++) {
	    if (OBJ_BF_ISREF(bf, i)) {
               th_assert(slottypes[i], "Unitialized slottypes map\n"); 
	       cache_swizzle_one_ref(fieldstart + i, or_num, slottypes[i]);
            }
	}
      }
    }

    core x = (core)(unswiz + c->dhsize - 1);
    th_assert(c->dhsize > 0, "Nonsense size for dispatch header\n");
    if (c->dhsize == 1) x->methods = *c->dh;        
    else memcpy(unswiz, c->dh, c->dhsize * sizeof(DV));   
}



  
void enter_nobj_info (obj nobj, Xref x) {
    /* effects - create entry in swizzle table for new object */   
    obj new_info = get_storable_pointer(nobj);
    swiz.store(x, new_info);
}


#define CHUNK2OR_OBJ(x) ((OR_obj*)((cacheval*)(x) + DEFAULT_NULLSPACE))
#define OR_OBJ2CHUNK(x) ((cacheval*)(x) - DEFAULT_NULLSPACE)


/*
  An obj_pointers structure contains the various pointers
  to different versions of the object used during swizzling.

  "chunk" is the storage as fetched from the OR, including leading
  nullspace.

  "or_rep" is the storage as fetched from the OR, without leading
   nullspace.  

   The remaining fields currently point to a newly-allocated object in
   the heap, since we don't fetch directly into the heap.  However,
   we could fetch directly into the heap as long as we continue to
   use a copying collector.
   
   "fe_rep" is the storage as allocated.
   
   "fe_core" points to the core (the last method pointer before the 
    non-method-pointer fields).
    
    "OR_fields" points to the first (unswizzled) field of the object.

*/

typedef struct obj_pointers_s {cacheval *chunk;  // includes nullspace
			       OR_obj *or_rep;   // no nullspace
			       obj fe_rep;       // alloc'ed in cache
			       core fe_core;     // core 
			       OR_obj *or_fields; // unswizzled fields
                               } *obj_pointers;


// Statistic on the number of objects fetched that was already 
// cached.
int nduplicates = 0;

// Statistic on the number of bytes fetched
long nbytes_fetched = 0;

// space to put duplicate objects.
static cacheval ignore[DEFAULT_MAXSLOTS_OBJ];

// Iovecs array.
static iovec iov[UIO_MAXIOV];

static inline 
void cache_initialize_fetched(obj first, int num, int or_num) {
  class_code code = 0; 
  class_ cl;
  int hsize;	
  Xref x;
  x.or = or_num;
  obj fe_rep = first;
  
  for (int i = 0; i < num; i++) {
    OR_obj* or_rep = (OR_obj *) (fe_rep+1+ (FE_obj_headers - OR_obj_headers));
    OR_obj* or_part_above = (OR_obj*) (fe_rep + 1);

    if (OR_OBJ_CLASS(or_rep) != code) {
      // The class of this object is different from the class of
      // the previous one.
      code = OR_OBJ_CLASS(or_rep);
      cl = class_from_code(code);
      th_assert(cl != 0, "Couldn't find class");
      hsize = hdr_size(&(cl->hdr.inh));
    }

    // Initialize the object assuming we have only one word of dispatch
    // header.
    th_assert(hsize == 1, "Multiple words of dipatch header in object\n");
    x = *(Xref *)fe_rep;

#if LAZY_SWIZZLING
    inst_unsw_hdr(fe_rep, cl);
#else
    cache_prim_swizzle(fe_rep, or_num, or_part_above, cl);
#endif
    // Initialize the xref, stamp and handle  fields
    core c  = (core) fe_rep;
    c->stamp = 0;
    c->handle_ = 0;
    c->xref = x;
    // c->size = OR_OBJ_SIZE(or_rep);
    // c->bitfields = OR_OBJ_BITFIELD(or_rep);

    // Move fe_rep to point to the next object.
    fe_rep = (obj) (or_rep->slot + OR_OBJ_SIZE(or_rep));
  }
}

extern Timer fe_net_fetch_time;

void cache_enter_fetched_objects(Network *net, int num, OR_num or_num,
						 or_objdesc* desc) {
     th_assert(num < UIO_MAXIOV, "Cannot handle so many objects per fetch\n");
  
     // Find out where we are going to put the objects in the cache.
     obj first = (obj) gc->raw_alloc_slots(1);
      
     // Compute the space we need and compute iovecs for fetched 
     // objects.
     int count = 0;
     int non_duplicates = 0;
     Xref x;
     x.or = or_num;
  
     // During this loop no code can allocate in the GCed heap
     // This is checked by the assertion after the rest of the space is
     // allocated.
     for (int i = 0; i < num; i++, desc) {
       x.oref = desc[i].o;
 
       // This code does not have to worry about the new value being
       // different from the old one, etc because the concurrency control
       // code (invalidations) should take care of this issue. 
       obj *present_obj = swiz.storeOrFind(x);
       if (*present_obj && !IS_SURR(*present_obj)) {
	 // If there was a real object in swizzle table then ignore
	 // the newly-fetched object. Fill iovec for duplicate 
         // pointing to the ignore array.
	 iov[i].iov_base = (caddr_t) ignore;
	 iov[i].iov_len = desc[i].objsize * FE_SLOT_SIZE;
         th_assert(desc[i].objsize < DEFAULT_MAXSLOTS_OBJ,
		   "Fetched object is bigger than limit\n");
       } else {
	 non_duplicates++;
 
	 // If there was an empty surrogate in the swizzle table fill it.
	 if (*present_obj) {
	   STATS(stats->surrogate_fills++);
	   surr_fill((surr)*present_obj, first+count);
	 }
  
	 // Update object's entry in the swizzle table.
	 *present_obj = first+count;
 
	 // Fill iovec for non-duplicate assuming one word of dispatch
	 // header.
	 iov[i].iov_base = (caddr_t) (*present_obj
				      +1+(FE_obj_headers - OR_obj_headers));
	 iov[i].iov_len = desc[i].objsize * FE_SLOT_SIZE;
	 count = count + desc[i].objsize + 1
	     + (FE_obj_headers - OR_obj_headers);;
        
         // Save xref in first word of dispatch header. TODO: Could
         // avoid this cache miss if I can assume orefs are contiguous
         // most of the time, in this case could have small structure
         // with intervals of contiguous orefs.
         *(Xref *)(*present_obj) = x;
       }
     }
    
     // Allocate the rest of the slots we need.
     obj rest = (obj) gc->raw_alloc_slots(count-1);
     th_assert(first+1 == rest, "Failed to allocate contiguous cache space\n");

     // Read in the actual contents directly into the allocated cache space.
     if (!net->read_vector(iov, num)) 
       fail("Failed to read fetched objects");

     // It is difficult to place this timer because it will include the 
     // initial cache registering cost. TODO: get a timer to measure
     // that cost, but thast timer must be conditionally compiled
     // in when stats are required.
     fe_net_fetch_time.stop();

     nduplicates += num - non_duplicates;
     nbytes_fetched += (count-non_duplicates) * FE_SLOT_SIZE;

     cache_initialize_fetched(first, non_duplicates, or_num);
}




void cache_swizzle_object(int or_num, obj o) {
    th_assert(is_unswizzled(o), "swizzling an already swizzled object");
    th_assert(o == OBJ_START(o), "passed an interior pointer");

    core c = BUMP(core, o, *o); 
    class_ cl = (class_)(*o)->t;

#if ADAPT_PREFETCH
    // Update the statistic on the number of objects used.
    p_nused++;
#endif

    cache_prim_swizzle(o, or_num, (OR_obj *) (((cacheval *)c)+1), cl);
}




void clear_swiztab(void) {
    swiz.clear();
}

obj swiz_get_snap(Xref x) {
    // effects - returns the address of a non-surrogate object, or
    //           0 if the only version in the cache is a surrogate.

    obj latest;
    bool found = swiz.fast_find(x, latest);
    th_assert(found, "Nothing in swizzle table for a surrogate");
    if (!IS_SURR(latest))
	return latest;
    else
	return 0;
}

obj swiz_get_gc_object(Xref x) {
  obj lookedup;
  bool found = swiz.fast_find(x, lookedup);
  if (found == TRUE)
      return lookedup;
  else
      return 0;
}


obj swiz_get_object(Xref x) {
    // Similar to swiz_get_snap but without the requires clause
  obj lookedup = swiz_get_gc_object(x);
  if (lookedup && !IS_SURR(lookedup))
      return lookedup;
  return NULL;
}


//  SHRINKING
//  This code implements the mechanisms for deciding which objects should
//  be shrunk.

static inline bool toss_coin() {
    // When SHRINK_FRACTION is 0.0, this always returns false;
    // when SHRINK_FRACTION is 1.0, this always returns true.

    // skew range by 1
    int val = rand() + 1;
    float fract = (float) val / ((float) (RAND_MAX + 1));
    th_assert((fract > 0.0) && (fract <= 1.0), "Illegal fraction in toss_coin");
    return (fract <= SHRINK_FRACTION);
}

static inline void do_shrink(obj a, Xref key) {
    // requires - "a" is a core and is legal to shrink.
    //            "key" is its xref.
    // effects - shrinks "a" to a surrogate containing key

    core c = oac(a);
    th_assert(IS_PERSISTENT(c), "Tried to shrink non-persistent object");
#ifndef NDEBUG
    if (IS_READ_CORE(c) != 0)
	shrunk_read_persistent++;
    else
	shrunk_unread_persistent++;
#endif
    
    objtype t;
    if (IS_UNSWIZZLED(a))
       t = (*a)->t;
    else {
       th_assert((*a)->c, "Object class not initialized in DV");
       t = class_as_objtype((*a)->c);
    }

    inst_surr_hdr(OBJ_START(a), t);
    *SURR_XREF(a) = key;
    
    th_assert(IS_SURR(a), "Failed to shrink object");
    th_assert(XREF_EQUAL(*SURR_XREF(a), key), 
	      "Failed to put xref into surrogate");
    th_assert(cache_obj_plausible(a), "Shrank outside current space");
}

inline bool can_shrink(core ac) {
    // effects	Returns true iff it is legal to shrink "ac"

    //  Do not even THINK ABOUT shrinking type/class objects
    th_assert(!isa_objtype(cao(ac)), "shrinking a type object");

    // It is a bad idea to shrink written objects.
    if (IS_WRITTEN_CORE(ac) != 0) return FALSE;
    return TRUE;
}


static inline void shrink_subpages(int target_shrink, int &cur_shrink) {
  int subpages = subpage_fifo.size();
  while (subpages--) {
    Subpage *s = subpage_fifo.remove_first();
    obj swiz_entry;
    int num = s->num;
    int mod_num = 0;
    Xref xref;
    xref.or = s->or_num;
    or_obj_descriptors *descs = s->descs;

    for (int i = 0; i < num; i++) {	
      // Shrink objects in subpage that are still
      // cached.
      xref.oref = descs->slot(i).o;
      if (swiz.fast_find(xref, swiz_entry) && !IS_SURR(swiz_entry)) {
	int size = descs->slot(i).objsize;
	core ac = oac(swiz_entry); 	
	if (!IS_WRITTEN_CORE(ac)) {
	  cur_shrink += size;
	  do_shrink(swiz_entry, xref);
	} else {
	  // If objects are modified can't shrink them
	  // therefore compact them in subpage s.
	  descs->slot(mod_num).o = xref.oref;
	  descs->slot(mod_num).objsize = size;
	  mod_num++;
	}
      }
    }
    
    if (mod_num) {
      // Page had modified objects.
      s->num = mod_num;
      subpage_fifo.insert(s);
    } else 
	// Page did not have modified objects.
	delete s;
    
    if (cur_shrink >= target_shrink)
      // Enough space was freed stop shrinking objects.
      break;
  }
}


void shrink_objects() {
    // effects - shrink objects in accord with the compiled-in policy.

  int target_shrinking = SHRINK_FRACTION * (float)cache_occupied_slots();
  int current_shrinking = 0;
  shrunk_read_persistent = shrunk_unread_persistent = 0;

#if SHRINK_SUBPAGES
  shrink_subpages(target_shrinking, current_shrinking);
#else
  SwizTabGen bindings = swiz.mappings();
  
  XrefKey xrefk;
  while (bindings.get(xrefk)) {
    obj swiz_entry = bindings.value();
    Xref xref = xrefk.val; 
    if (!IS_SURR(swiz_entry)) {
      core ac = oac(swiz_entry);  // We know "a" is a core, so this is OK
      th_assert(cache_obj_plausible(swiz_entry), "Shrinking outside current space");
#if !LRU 
      // Use toss_coin to randomly choose whether
      // this object will be shrunk.
      if (can_shrink(ac) && toss_coin())
	do_shrink(swiz_entry, xref);
    }
#else   // LRU
    // Use CLOCK policy to choose objects to shrink
    // while iterating through swizzle table.

    // The original code used to check if the shrinking target has not
    // been achieved then try shrinking again (LRU code)
    // This is not being done now. Should it be done ?? XXX -- Atul
    if (can_shrink(ac)) {
      if (!IS_USED_CORE(ac) {
	// Not used recently, shrink
	current_shrinking += (ac->size) * FE_SLOT_SIZE;
	do_shrink(swiz_entry, xref);
      }
	else {
	  // recently used, don't shrink but turn off mark
	  UNMARK_AS_USED(ac);
	}
    }
#endif // LRU
#endif //!SHRINK_SUBPAGES


     if (FEConf->debug_level) {
       fprintf(stderr, "Shrunk %d read, %d unread objects\n",
	       shrunk_read_persistent, shrunk_unread_persistent);
       fprintf(stderr, "current_shrinking=%d target_shrinking=%d\n",
	       current_shrinking, target_shrinking);
     }
}

void swiz_check_table(bool mark, int print_level) {

  SwizTabGen bindings = swiz.fast_mappings();
  
  XrefKey xrefk;
  while (bindings.get(xrefk)) {
    obj o = bindings.fast_value();
    Xref xref = xrefk.val;
    gc->check_object(o, -1, TRUE, mark, print_level, FALSE);
  }
}

// extern "C" void dumpHashStatistics() {
//     fprintf(stderr, "Efficiency: %f\n", swiz->estimateEfficiency());
//     fprintf(stderr, "Clumping: %f\n", swiz->estimateClumping());
// }

// ------------------------------------------------------------------------
//                    PAGING  This code is experimental
// ------------------------------------------------------------------------


#if PAGING

static inline class_ class_from_orobj(OR_obj *or_rep) {
    /* effects - uses class code in OR object representation "*or_rep" to look
                 up the class implementing that object.
    */
    class_code code = OR_OBJ_CLASS(or_rep);
    class_ c = class_from_code(code);
    th_assert(c != 0, "Couldn't find class");
    return c;
}


static bool swiz_lookup(int or, int pagenum, int index, obj fe_rep) {
    Xref xref;
    xref.or = or;
    OREF_SET(xref.oref,pagenum,index);
    obj existing;
    if (swiz.find(xref, existing) == 0) {
	// New entry in swizzle table
	obj swiz_entry;
	swiz_entry = fe_rep;   // XXX which of multiple method pointers?
	swiz.store(xref, swiz_entry);
	return FALSE;
    }
    // Existing entry in swizzle table
    if (IS_SURR(existing)) {
	// Empty surrogate -- fill it.
	STATS(stats->surrogate_fills++);
#if DEBUG_SWIZ
	fprintf(stderr, "filled surrogate %lx\n -> %lx\n", 
		existing, fe_rep);
#endif
	surr old = (surr)existing;
	surr_fill(old, fe_rep);
	existing = fe_rep;
	swiz.store(xref, existing);
    }
    // Otherwise : A duplicate object, leave swiz alone
}

static core core_from_obj(obj fe_rep, class_ cl) {
    /* effects - returns a core, appropriately offset from "fe_rep",
                 based on the information in "cl".
    */
    int hsize = hdr_size(&(cl->hdr.inh));
    core fe_core = (core)(((char*)fe_rep) + ((hsize - 1) * FE_SLOT_SIZE));
    return fe_core;
}


static void swiz_one_obj(obj fe_rep, OR_obj* unswiz, 
                         int or, int pagenum, int index) {
    // XXX Only works for header size == 1
    th_assert(((OR_slot*)unswiz - (OR_slot*)fe_rep) == 1, 
	      "unexpected header size");
    class_ cl = class_from_orobj(unswiz);
   
    if (swiz_lookup(or, pagenum, index, fe_rep)) {
#if DEBUG_PREFETCH
	fprintf(stderr, "Fetched %d.%d \n", pagenum, index);
#endif
	// swizzle immediately
	cache_prim_swizzle(fe_rep, or, unswiz, cl);
    }
    else {
#if DEBUG_PREFETCH
	fprintf(stderr, "Prefetched %d.%d \n", pagenum, index);
#endif
#if LAZY_SWIZZLING
	//  install special header that will swizzle object on use
	inst_unsw_hdr(fe_rep, cl);
#else
	// swizzle immediately
	cache_prim_swizzle(fe_rep, or, unswiz, cl);
#endif
    }
    // Done here since inst_obj_hdr clears the xref field
    core fe_core = core_from_obj(fe_rep, cl);
    Xref x;
    x.or = or; OREF_SET(x.oref,pagenum,index);
    fe_core->xref = x;
    th_assert(!IS_NULL_XREF(x), "Received NULL xref for fetched object");
}

obj cache_swizzle_page(int pagenum, Xref askedfor) {
    Page *p = pcache[pagenum];

    // Assume objects start at beginning of page
    OR_slot *current = (OR_slot*)p;
    OR_slot *limit = current + PAGE_SLOTS;
    int index = 0;
    obj result = 0;

    th_assert(OREF_SEGMENT(askedfor.oref) == pagenum, "Processing wrong page");

    // Assume objects are densely packed on page, in index order
    loop {
	// As long as the next object has a plausible size, assume it's OK.
	// Pages should be zero-filled before being packed.  Xref in front
	// of each object.
	Xref *next_x = (Xref*)current;
	th_assert(askedfor.or == next_x->or, "Unexpected OR");
	th_assert(OREF_SEGMENT(askedfor.oref) == OREF_SEGMENT(next_x->oref), 
		  "Unexpected segment");
	th_assert(index == OREF_INDEX(next_x->oref), "Unexpected index");
	obj fe_rep = (obj)(current);
	OR_obj *unswiz = (OR_obj*)(current + 1 + FE_obj_headers - OR_obj_headers);
	int size = OR_obj_full_size(unswiz);
	int max_size = limit - current;
	// "if size is more than zero and less than off the end of buffer
	// then swizzle the object and advance current"
	if ((size > 0) && (size <= max_size)) {
	    if (index == OREF_INDEX(askedfor.oref)) {
		swiz_one_obj(fe_rep, unswiz, askedfor.or, pagenum, index);
		result = fe_rep;
	    }
	    else {
#if LAZY_SWIZZLING
		lazy_swiz(unswiz, pagenum, index);
#else
		swiz_one_obj(fe_rep, unswiz, askedfor.or, pagenum, index);
#endif
	    }
	    current += size;
	    index++;
	}
	else
	    break;
    }
    return result;
}
#endif // PAGING
