/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _SWIZ_T
#define _SWIZ_T 1


//
// The swizzle table maps Xrefs to virtual memory addresses (obj) for
// objects or surrogates that are present in the FE cache.
// This file should only be included in swiz.cc. It contains the swizzle 
// table implementation. This implementation is a specialization of
// common/bhash.h and common/buckets.h. 
//

#include "common/map.h"
#include "common/xrefkey.h"
#include "common/bhash.t"
#include "common/buckets.t"
  

//
// SwizBuckets specializes the generic Buckets template 
// defined in common/buckets.t to <XrefKey, obj>
// pairs. There is one pair inlined in the bucket to get
// better cache behavior. It is an implementation of
// a pairset as descibed in common/bhash.h
//
class SwizBuckets : public Buckets<XrefKey, obj> {
friend class SwizBucketsGen;
public:
   SwizBuckets() { object = 0;}
   void operator=(SwizBuckets const &b);
   bool find(XrefKey k, obj &o) const;
   obj fetch(XrefKey k) const; 
   bool store(XrefKey k, obj o);
   bool remove(XrefKey k, obj &o);
   obj remove_fast(XrefKey k);
   HashGenerator<XrefKey, obj> *mappings();
   void clear_mappings(void);
   obj* storeOrFind(XrefKey k);

protected:
   obj* _storeOrFind(XrefKey k);

   // The first pair is inlined in the table.
   XrefKey key;
   obj object;
};

class SwizBucketsGen : public BucketsGenerator<XrefKey, obj> {
public:
   SwizBucketsGen(SwizBuckets &bu) : BucketsGenerator<XrefKey, obj>(bu) {
     where = 0;
     b = &bu;
   }
   void changeBucket(SwizBuckets &buckets); 
   bool get(XrefKey &k);
   obj &value() const;
   void remove();

protected:
   void remove_inlined();

   int where;
   SwizBuckets *b;
};


inline void SwizBucketsGen::changeBucket(SwizBuckets &buckets) {
     pairs = 0;
     last = &buckets.pairs;
     where = 0;
     b = &buckets;
}

inline bool SwizBucketsGen::get(XrefKey &k) {
    if (++where == 1) {
	if (b->object) {
	    k = b->key;
	    return TRUE;
        } else
            return FALSE;
    } else
       return BucketsGenerator<XrefKey, obj>::get(k);
}

inline obj &SwizBucketsGen::value() const {
	return (where == 1) ? b->object : pairs->value;
}

inline void SwizBucketsGen::remove() {
    if (where == 1) {
	if (b->pairs == 0) 
	   b->object = 0;	
        else
           remove_inlined();
    } else
       BucketsGenerator<XrefKey, obj>::remove();
}
	
void SwizBucketsGen::remove_inlined() {
        // The inlined <key,value> pair is being removed
        // and there is at least another pair in the
        // overflow chain. Just copy the first pair of
        // the overflow chain into the table and delete
        // from the chain.
        BucketsImpl<XrefKey, obj> *p = b->pairs;
        b->key = p->key;
        b->object = p->value;
        b->pairs = p->next;
        changeBucket(*b);
        delete p;
}


inline void SwizBuckets::clear_mappings(void) {
  object = 0; 
  Buckets<XrefKey, obj>::clear_mappings();
}


inline void SwizBuckets::operator=(SwizBuckets const &b) {
     Buckets<XrefKey, obj>::operator=(b);
     key = b.key; object = b.object;
}


inline bool  SwizBuckets::find(XrefKey k, obj &o) const {
     if (object) {
        if (k == key) {
          o = object;
          return TRUE;
        } else 
            return Buckets<XrefKey, obj>::find(k, o);
     } else
        return FALSE;
}


inline obj SwizBuckets::fetch(XrefKey k) const {
     if (object && k == key)
        return object;
     else
        return Buckets<XrefKey, obj>::fetch(k);
}


inline bool SwizBuckets::store(XrefKey k, obj o) {
     if (object == 0) {
       key = k; object = o;
       return FALSE;
     } else
        if (key == k) {
          object = o;
          return TRUE;
        } else
          return Buckets<XrefKey, obj>::store(k, o);
}


inline obj  *SwizBuckets::storeOrFind(XrefKey k) {
     if (object == 0 || key == k) {
       key = k; 
       return &object;
     } else
          return _storeOrFind(k);
}

obj  *SwizBuckets::_storeOrFind(XrefKey k) {
    BucketsImpl<XrefKey, obj> *b = pairs;
    while (b) {
        if (b->key == k) 
            return &(b->value);
        b = b->next;
    }
    add(k, 0);
    return &(pairs->value);
}


bool SwizBuckets::remove(XrefKey k, obj &o) {
  if (object) {
        if (k == key) {
          o = object;
          if (pairs) {
            BucketsImpl<XrefKey, obj> *p = pairs;
            key = p->key;
            object = p->value;
            pairs = p->next;
            delete p;
          } else
            object = 0;
          return TRUE;
        } else
          return Buckets<XrefKey, obj>::remove(k, o);
     } else
        return FALSE;
}

inline obj SwizBuckets::remove_fast(XrefKey k) {
  obj o;
  remove(k, o);
  return o;
}


HashGenerator<XrefKey, obj> *SwizBuckets::mappings() {
	return new SwizBucketsGen(*this);
}

class SwizTabGen;

class SwizTab : public BHash<XrefKey, obj, SwizBuckets> {
public:
  SwizTab(int size) : BHash<XrefKey, obj, SwizBuckets>(size) {}
  SwizTabGen fast_mappings(); 
  int tslots(void) {return numSlots;}
  SwizBuckets *tbuckets(void) {return buckets;}
  
  // If there is no mapping for k in the table then it allocates a 
  // mapping between k and 0. In either case it returns a pointer to
  // the value associated to k. If the pointer points to 0 then it
  // should be updated to a non-zero value before the next use of the
  // swizzle table.
  obj *storeOrFind(XrefKey k) {
    return buckets[do_hash(k)].storeOrFind(k);
  }
  
  bool fast_find(XrefKey k, obj &v) const {
    return buckets[do_hash(k)].find(k, v);
  }
  
  // Clear all mappings from swizzle table.
  void clear(void) {
    for (int i=0; i < numSlots; i++) 
      buckets[i].clear_mappings();
  }
};
  

class SwizTabGen {
public:
   SwizTabGen(SwizTab &tab) : t(&tab), sg(*tab.tbuckets()) { slot = 0;}
   bool get(XrefKey &k);
   void fast_remove() { sg.remove();}
   obj &fast_value() {return sg.value();}
protected:
   SwizBucketsGen sg;
   SwizTab *t;
   int slot;
};


SwizTabGen SwizTab::fast_mappings() {	
	return SwizTabGen(*this);
}

bool SwizTabGen::get(XrefKey &k) {
    if (sg.get(k)) return TRUE;
    
    int tslots =  t->tslots();
    SwizBuckets *tbuckets = t->tbuckets();
     
    do {
        slot++;
        if (slot == tslots) 
            return FALSE;
        sg.changeBucket(tbuckets[slot]);
    } while (!sg.get(k));
    return TRUE;
}

  

// Force template instantiation for swizzle table.
#if __DECCXX
#pragma define_template Buckets<XrefKey, obj>
#pragma define_template BHash<XrefKey, obj, SwizBuckets >
#pragma define_template BucketsImpl<XrefKey, obj>
#pragma define_template BHashGenerator<XrefKey, obj, SwizBuckets >
#pragma define_template BucketsGenerator<XrefKey, obj>
#endif



#endif //_SWIZ_T






