// Copyright 1995 Barbara Liskov

#include "server.h"
#include "invoke.h"
#include "handle.h"
#include "future.h"
#include "runtime/alloc.h"
#include "fe_config.h"
#include "gen-include.h" 
#include "gen-C-veneer.h"
#include "dump.h"
#include "runtime/stats.h"

#include "types/vec_instns.h"
#include "runtime/except.h"
#include "runtime/disphdr.h"
#include "runtime/obj_class.h"
#include "runtime/value.h"
#include "runtime/surr.h"
#include "runtime/commit.h"

#include "boot/wellknown.h"
#include "boot/specs.h"

#include "config/vdefs/SHM.h"

#if !SHM

#include "binary_interface.h"
#include "common/xref.h"
#include "common/bits.h"
#include "common/basic.h"
#include "common/th_assert.h"

#include "cache/cache.h"
#include "cache/gc_init.h"

#include "types/any.h"
#include "types/array.h"
#include "types/class.h"
#include "types/exception.h"
#include "types/method.h"
#include "types/string_class.h"
#include "types/type.h"
#include "types/vec.h"
#include "types/vec_instns.h"
#include "types/objtype_class.h"

extern "C" {
#include "fe_expr.h"
#include "types/list.h"
#include "boot/lex_input.h"
#include "types/char.h"
}

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "init_fe.h"
#define STDBUFSIZE 200
extern int feyy_lineno;

static void invoke_cmd(connection c, char const inp[], bool futures_specified);
static void wellknown_cmd(connection c, char const inp[]);
static void input_cmd(connection c, char const line[]);
static void print_cmd(connection c, char const inp[]);
static void commit_cmd(connection c, char const inp[]);
static void invalidate_obj_cmd(connection c, char const inp[]);
static void info_obj_cmd(connection c, char const inp[]);
static void stringlen_cmd(connection c, char const inp[]);
static void print_results(connection c, method m, rt_result result, 
			  bool futures_specified, int result_futures[], int retc);
static void print_abort(connection); 

#if COLLECT_STATS
static void stats_cmd(connection c);
static void zero_stats_cmd(connection c);
#endif 

static const char whitespace[] = " \t\n\r\f";
static struct closure binary_cl = { (ifunc)binaryDispatch, 0 };
bool   typecheck_args(const method m, const vec args, int *mismatch);

int max_string_display_length = 200;
/* Longest result string to display directly in ASCII interface */

static void print(connection c, char const msg[])
{
  connection_fwrite(c, msg, strlen(msg));
}

static bool match(char const *buf, char const *token)
{
    int l = strlen(token);
    return (0 == strncmp(buf, token, l) &&
	    strchr(whitespace, buf[l]));
}

static char help_message[] = 
      "The following commands are supported in ASCII mode\n"
      "  help, ? : This message\n"
      "  quit, q : Close connection\n"
      "  shutdown : shut down FE\n"
      "  invoke <self> [<type>.]<method> <arg1> .. <argn> : evaluate an expr\n"
      "  input  <string> : return a handle to <string>\n"
      "  wellknown <objname> : supply handle for object `objname'\n"
      "  print <object> : display the given object of a builtin type\n"
      "  stringlen <integer> : set the maximum length of strings to "
	  "display directly\n"
      "  binary : Enter binary mode\n"
      "  checkpoint, cp : commit changes, start a new transaction\n"
      "  abort : abort current transaction\n"
#if COLLECT_STATS
      "  stats : show current counts for various statistics\n"
      "  zero : set counts to zero\n"
#endif /* COLLECT_STATS */
#if 0
      "  dispatch <obj> : send future input to \"dispatch\" method of <obj>\n"
#endif
      "\n"
      "  Multiple commands may be separated by semicolons.\n"
      "\n"
      "  No other help available yet.\n";

static void help(connection c, char const topic[])
{
    connection_printf(c,  "D %d bytes\n", strlen(help_message));
    connection_printf(c,  "%s\n", help_message);
}

/* Find the _next token_: return a pointer to the first non-whitespace
   character following any amount of whitespace, some non-whitespace,
   then some whitespace. */
char const *nt(char const *cmd)
{
    int i = strspn(cmd, whitespace);
    i += strcspn(cmd + i, whitespace);
    i += strspn(cmd + i, whitespace);
    return cmd + i;
}


#if 0
static int strnlen(char const *buf, int max)
{
    char const *c = buf;
    while (*c++ && max--) /* SKIP */;
    return (c - 1) - buf;
}
#endif

static void not_supported(connection c, char const cmd[])
{
    char buf[STDBUFSIZE];
    sprintf(buf, "2 \"%.150s\" is not yet supported\n", cmd);
    print(c, buf);
}

static void normalDispatch(connection c, obj env, FILE *f)
{
  char linebuf[BUFSIZ], *line = linebuf;
  bool gc_thrash = FALSE;
  
  /* Note: There is no while loop here since main.cc controls the
     looping. This is more inefficient than binary_interface.cc but that is
     acceptable
  */
  int to_abort = CATCH_ABORT();
  if (to_abort) {
      /* A longjmp has been made. Send abort exception to client */
      print_abort(c);
      return; /* So that the real setjmp gets called now */
      }
  do {
    fscanf(f, "%[^\n]", line);

    if (gc_thrash) { print(c, "8 the garbage collector is thrashing\n"); }

    line += strspn(line, whitespace);
    if (!*line) continue;	     	/* skip empty commands */

    STATS(stats->ops++)

    if (match(line, "help") ||
	match(line, "?")) { help(c, nt(line));  }
    else if (match(line, "quit") ||
	match(line, "q")) { connection_close(c);  }
    else if (match(line, "binary")) {
      fprintf(stderr, "3 Entering binary mode.\n"); 
      connection_setDisp(c, binary_cl);
    }
    else if (match(line, "invoke") ||match(line, "i")) {
	invoke_cmd(c, nt(line), FALSE);  
      }
    else if (match(line, "pinvoke") || 
	     match(line, "pi")) {
      invoke_cmd(c, nt(line), TRUE);  }		    
    else if (match(line, "wellknown") || match(line, "wk"))
      { wellknown_cmd(c, nt(line));  }
    else if (match(line, "input")) { input_cmd(c, nt(line));  }
    else if (match(line, "print") ||
	match(line, "p")) { print_cmd(c, nt(line));  }
    else if (match(line, "abort")) { not_supported(c, "abort");  }
    else if (match(line, "dispatch")) { not_supported(c, "dispatch");  }
    else if (match(line, "checkpoint") ||
	match(line, "cp")) { commit_cmd(c, nt(line));  }
    else if (match(line, "stringlen") ||
	match(line, "sl")) { stringlen_cmd(c, nt(line));  }
    else if (match(line, "uncache") || match(line, "u"))
	{invalidate_obj_cmd(c, nt(line));  }
    else if (match(line, "info")) {info_obj_cmd(c, nt(line));  }
#if COLLECT_STATS
    else if (match(line, "stats")) {stats_cmd(c); }
    else if (match(line, "zero")) {zero_stats_cmd(c); }
#endif
    else if (match(line, "shutdown")) {
      connection_broadcast(string_new("Shutdown request has been honored\n"));
      exit(EXIT_SUCCESS);
    }
    else {
      print(c, "1 Unrecognized command\n");
    }
    th_assert(!exc, "Unexpected exception raised");
#if INSANELY_AGGRESSIVE_GC
    cache_force_gc();
#else
    gc_thrash = cache_finished_op();
#endif
  } while (fgetc(f) == ';');
}
	      
struct closure ascii_cl = { (ifunc)normalDispatch, 0 };

static void printValue(connection c, any v)
/* Send a textual representation of the value in "v" to "c". */
{
    char buf[100]; /* 100 is large enough for any printable value. */

    class_ cl = get_any_class(v);

    if (cl == Int) {
      connection_printf(c,  "%d", any_get_value(v).i);
    } else if (cl == Char) {
      string s = char_unparse_fancy(any_get_value(v).c);
      connection_put(c,s);
      delete_string(s);
    } else if (cl == Bool) {
      connection_printf(c,  "%s", any_get_value(v).b ? "true" : "false");
    } else if (cl == Null) {
      connection_printf(c,  "nil");
    } else if (cl == Real) {
      connection_printf(c,  "%f", any_get_value(v).r);
    } else {
      handle h = FEConf->ht->any_to_handle(v);
      connection_printf(c,  "#%d", h);
    }

    if (cl == String && 
	string_length(any_get_value(v).s) < max_string_display_length) {
      string s = string_unparse_fancy(any_get_value(v).s);;
      print(c, " (");
      connection_put(c, s);
      print(c, ")");
      delete_string(s);
    }
    
}

/***************************
 * Some grungy Lex stuff
 */

static char const *inp_posn;
static char const *accepted_inp_posn;

static char string_input()
{
    return *inp_posn++;
}

extern char *feyy_text;

static int lex() {
    accepted_inp_posn = inp_posn;
    return feyy_lex();
}

/*
 * End of grungy Lex stuff
 **************************/

any parseValue(int tag, exception *the_exc)
/*
   Given the contents of the global variable "feyy_lval" and "tag",
   return a corresponding "any". ("feyy_lval" is what we'd ordinarily
   call "yylval" in yacc-land.)
       Place the exception "not_possible(string)" in "exc" with an
   appropriate error message if this cannot be done because the token
   does not represent a literal constant.
*/
{
    any ret;
    switch(tag) {
    case StringTok:
      ret = obj_as_any(string_as_obj(feyy_lval.s));
      break;
    case CharTok:
      ret = char_as_any(feyy_lval.c);
      break;
    case IntegerTok:
      ret = int_as_any(feyy_lval.i);
      break;
    case RealTok:
      ret = real_as_any(feyy_lval.r);
      break;

    case HandleTok: case FutureTok:
      ret = FEConf->ht->handle_to_any(feyy_lval.i, TRUE).val;
      if (*the_exc) {
	*the_exc = &exc_not_possible;
	exc_value = "Invalid handle";
      } else
	  return ret;
      break;

    case IdTok:
      *the_exc = &exc_not_possible;
      exc_value = "Expected literal constant, not identifier.";
      break;
    case DotTok:
      *the_exc = &exc_not_possible;
      exc_value = "Expected literal constant, not dot.";
      break;
    case LexErr:
      *the_exc = &exc_not_possible;
      exc_value = "Lexical error.";
      break;
    case BoolTok:
      ret = bool_as_any(feyy_lval.b);
      break;
    case NilTok:
      ret = null_as_any(0);
      break;
    case 0:
      *the_exc = &exc_not_possible;
      exc_value = "Unexpected end of input stream";
      break;
    default:
      th_fail("Unknown token type encountered while parsing a value");
    }
    if (!*the_exc) return ret;
}

/* Raises "not_possible" if there is no value there.*/
any readValue(exception *the_exc)
{
    int tag = feyy_lex();
    return parseValue(tag, the_exc);
}

void printParseError(connection c, char const inp[])
{
    char buf[STDBUFSIZE];
    exc = EXC_NONE;
    finish_lex_input();
    sprintf(buf,"4 %.150s\n", (char *)exc_value);
    print(c, buf);
    print(c, "  ");
    print(c, inp);
    print(c, "\n");
    {
	int i;
	print(c, "  ");
	for (i=0; i< accepted_inp_posn - inp; i++)
	  print(c, " ");
	print(c, "^\n");
    }
    return;
}

	
	
	
static void printNormalResult(connection c, method m, vec r);
static void printExceptionalResult(connection c, method m,
				   vec r, exception e);
	
	
static objtype advance_to_method_name(connection c, char const inp[])
	/*
	 * Advance the parsing state until the name of the method is the next token.
	 * If a method type is encountered, return it. See "invoke_cmd". Raise
	 * an exception and print an appropriate message if there is a parsing error.
	 *
	 * WARNING: If "printParseError" is called out of this routine, the
	 *          global "exc" variable must be set to "&exc_not_possible"
	 *          immediately afterward, for crufty reasons.
	 */
{
  objtype ret = 0;
  any method_type_a;
  int tag = lex();
  if (tag == IdTok) return ret;
  method_type_a = parseValue(tag, &exc);
  CATCH { printParseError(c, inp);
	  exc = &exc_not_possible;
	  return ret; }

  if (!isSubtype(class_as_type(get_any_class(method_type_a)),
		 objtype_as_type(class_as_objtype(ObjType)))) {
      exc_value = "Expected an ObjType object";
      printParseError(c, inp);
      exc = &exc_not_possible;
      return ret;
  }

  ret = (objtype)any_get_obj(method_type_a); /* XXX but: for class_instns? */
  tag = lex();
  if (tag != DotTok) { exc_value = 
		       "Expected . as separator in <type> . <method>";
		       printParseError(c, inp);
		       exc = &exc_not_possible;
		       return ret;
		     }
  tag = lex();
  if (tag != IdTok) {
    exc_value = "Expected method name";
    printParseError(c, inp);
    exc = &exc_not_possible;
    return ret;
  }
}
      
      
static bool checkMethodTypeQualifier(connection c, char const inp[],
				     objtype method_type, objtype
				     receiver_type)
/* Return TRUE if the declared method type and the type of the receiver
   do not match. Print out an appropriate error message to "c".
   See "invoke_cmd". */
{
  char buf[STDBUFSIZE];
  if (!isSubtype(objtype_as_type(receiver_type),
		 objtype_as_type(method_type))) {
    sprintf(buf, "Actual type of self (%.50s) is not a subtype"
	    " of the method qualifier (%.50s).",
	    string_charp(type_name(objtype_as_type(receiver_type))),
	    string_charp(type_name(objtype_as_type(method_type))));
    exc_value = buf;		/* XXX see below */
    printParseError(c, inp);
    return TRUE;
  }
  return FALSE;
}

struct rtc_env {
  connection c;
  method m;
};

void reportToClient(struct rtc_env *env, obj val0, obj val1, obj val2, obj val3, obj val4, obj val5, obj val6, obj val7, obj val8, obj val9)
{
    fevalue fv;
    fevalue *vals;
    method m = env->m;
    print(env->c, "> ");
    vals = (fevalue *)malloc(9 * sizeof(fevalue));
    vals[0].o = val0;
    vals[1].o = val1;
    vals[2].o = val2;
    vals[3].o = val3;
    vals[4].o = val4;
    vals[5].o = val5;
    vals[6].o = val6;
    vals[7].o = val7;
    vals[8].o = val8;
    vals[9].o = val9;
    {
	vec vals2;
	int num_vals = vec_length(m->returns);
	int i;
	th_assert(num_vals < 9, "Can only handle 9 results from an iterator");
	vals2 = make_vec_Any(num_vals, TRUE);
	for (i = 0; i<num_vals; i++) {
	    type t = UNPV(type, vec_fetch(m->returns, i));
	    vec_store(vals2, i, PV(any_as_obj(make_any(t, vals[i]))));
	}
	printNormalResult(env->c, env->m, vals2);
    }
    print(env->c, "\n");
    connection_fflush(env->c);
}

				   
static void invoke_cmd(connection c, char const inp[], 
			 bool futures_specified)
{
    any receiver;
    objtype method_type = 0;
    int result_futures[9] = {0,0,0,0,0,0,0,0,0};
    int resultc = 0;
    vec args;
    int argc, i, tag, mismatch;

  
    start_lex_input(string_input, &feyy_lineno);
    inp_posn = inp;
    accepted_inp_posn = inp;

/*
 * First, obtain the value that will receive the invocation.
 */
    receiver = readValue(&exc);
    CATCH { printParseError(c, inp); return; }
    
/* Get the method type, if any.  */

    method_type = advance_to_method_name(c, inp);
    CATCH {
	exc = EXC_NONE;
	return;
    }
/*
  Check whether the receiver is consistent with the method name,
  and get the "method" object so the arguments can be type-checked.
  */
    {
	char buf[STDBUFSIZE];
	string m_name = feyy_lval.s;
	objtype t = class_as_objtype(get_any_class(receiver));
	method m;
	objtype method_type_2;
	fevalue tgm_ret[2];
	if (!method_type) method_type = t;
	else if (checkMethodTypeQualifier(c, inp, method_type, t)) return;
	objtype_get_method(method_type, tgm_ret, m_name);
	m = (method)tgm_ret[0].o;
	method_type_2 = (objtype)tgm_ret[1].o;
	
	CATCH {
	    sprintf(buf, " Type \"%.50s\" has no method \"%.50s\".\n",
		    string_charp(type_name(objtype_as_type(t))),
		    string_charp(m_name));
	    exc_value = buf;	/* XXX oog -- addr of stack value */
	    printParseError(c, inp);
	    return;
	}

/* Read the arguments 
 */	    
	argc = vec_length(m->arguments);
	
	th_assert(!exc, "An exception was raised, probably before entry to"
		  "invoke_cmd");
	
	args = make_vec_Any(argc, TRUE); /* read args */
	for (i=0; i < argc; i++) {
	    any arg = readValue(&exc);
	    CATCH { printParseError(c, inp); return; }
	    vec_store(args, i, any_as_obj(arg));
	}


	if (futures_specified) { 
	    for (i=0; i< vec_length(m->returns); i++) {
		tag=lex();
		if (tag == FutureTok)
		    result_futures[resultc++] = feyy_lval.i;
		else {
		    exc_value = "Syntax error in return futures";
		    printParseError(c, inp);
		}
	    }
	}
      

/* Typecheck the arguments 
 */
	if (!typecheck_args(m, args, &mismatch)) {	
	    any arg = UNPV(any, vec_fetch(args, mismatch));
	    type arg_type = class_as_type(get_any_class(arg));
	    formal f = UNPV(formal, vec_fetch(m->arguments, mismatch));
	    
	    connection_printf
		(c, "5 Argument type mismatch on argument %.20s: "
		 "expected %.20s but got %.20s\n",
		 string_charp(f->name),	
		 string_charp(type_name(f->t)),
		 string_charp(type_name(arg_type)));
	    
	    finish_lex_input();
	    return;
	}
	
	tag = lex();
	if (tag != 0) {
	    exc_value = "Unexpected extra argument?";
	    printParseError(c, inp);
	    return;
	}
	
	finish_lex_input();
/*
 * Everything typechecked. Bundle up the arguments
 * for a call to invoke, then call it and print the result.
 */
	{
	    fevalue self;
	    int i, num_args = vec_length(args);
	    int hdroff = hdr_offset(method_type, method_type_2);
	    vec vargs;
	    rt_result result;
	    self.o = any_as_obj(receiver);
	    th_assert(!exc, "Unexpected exception raised");
	    vargs = make_vec_Any(num_args, TRUE);
	    
	    for (i = 0; i < num_args; i++) {
		any a = obj_as_any(vec_fetch(args, i));
		vec_store(vargs, i, PV(a));
	    }
	    
	    if (hdroff) {
		self.o = (obj)(((DV)self.o) + hdroff);
	    }
	    
	    /* 
	      XXX We need to bump all the arguments, too.
	      XXX Right now we are not doing this, but it should work
	      XXX for all situations encounterable at the moment.
	      */
	    
	    if (method_isIter(m)) {
		struct closure rtc_closure;
		struct rtc_env env;
		rtc_closure.f = (ifunc)reportToClient;
		rtc_closure.env = &env;
		env.c = c;
		env.m = m;
		result = iterate(self, method_type, m, vargs,
				 rtc_closure);
	    } else {
		result = invoke(self, method_type, m, vargs);
	    }
	    
/* Print the results 
 */
	    print_results(c, m, result, futures_specified, result_futures, resultc);
	}
    }
}


static void print_exception(connection c, int resultc, method m,
			    bool futures_specified, int *result_futures,
			    rt_result result) {
    /* The specs of this are very similar to the specs of send_exception in
       binary_interface.cc. See that file for the specs
       The extra result parameter is present since the vector information
       etc has been calculated
    */
    int i;
    string exception_string = result.u.exceptional.exc->name;
    if (futures_specified) { 
	Exception exception_obj =  new_Exception
	    (exception_string, result.u.exceptional.val);
	
	/* map the specified result handles to the exception */
	for (i=0; i < resultc; i++) {
	    FEConf->ft->any_to_future(obj_as_any((obj)(exception_obj)),
			result_futures[i]);
	}
	return;
    } 
    /* Normal exception sent */
    print(c, "! ");
    connection_put(c, result.u.exceptional.exc->name);
    printExceptionalResult(c, m, result.u.exceptional.val,
			   result.u.exceptional.exc);
    print(c, "\n");
}

static void print_abort(connection c) {
    /* Send an abort exception to the client */

    rt_result result;
    result.tag = rt_exceptional;
    result.u.exceptional.exc = &exc_abort;
    result.u.exceptional.val = make_vec_Int(0, TRUE);
    fprintf(stderr, "Immediate abort at FE (Ascii Interface)\n");
    print_exception(c, 0, NULL, FALSE, NULL, result);
}

static void print_results (connection c,method m, rt_result result, 
			   bool futures_specified, int result_futures[], int resultc)
{
    int i;
    if (result.tag == rt_normal) {
	if (method_isIter(m)) {
	    print(c, "= \n");
	} else {
	    vec results = result.u.normal;
	    int num_results = vec_length(results);
		      
	    if (futures_specified)
		for (i=0; i<num_results; i++) {
		    FEConf->ft->any_to_future
			(obj_as_any(UNPV(obj, vec_fetch(results, i))),
			 result_futures[i]);
		}
	    else {
		print(c, "= ");
		printNormalResult(c, m, result.u.normal);
		print(c, "\n");
	    }
	}
    }
	
    else  {
	print_exception(c, resultc, m, futures_specified, result_futures,
			result);
    }
}

static void commit_cmd(connection c, char const inp[]) 
{
  switch (commit_trans()) {
    case COMMITTED: 
      print(c, "@Transaction committed.\n");    break;
    case FAILED_ABORT:
      print(c, "@Transaction aborted.Connection failure/Invalid object.\n"); break;
    case STALE_READ_ABORT:
      print(c, "@Transaction aborted. Old data read.\n"); break;
    case UNEXPECTED_MSG_ABORT:
      print(c, "@Unexpected message from OR.\n"); break;
  }
}

static bool get_persistent_obj_from_stream (connection c, char const inp[], 
					    obj* res) {
  any receiver;
  obj o;

  start_lex_input(string_input, &feyy_lineno);
  inp_posn = inp;
  accepted_inp_posn = inp;
  receiver = readValue(&exc);
  CATCH { printParseError(c, inp); return FALSE; }
  finish_lex_input();
  o = any_as_obj(receiver);
  if (!is_persistent_object (o)) {
      print(c, "Not a persistent object.\n");
      return FALSE;
  }
  *res = o;
  return TRUE;
}

static void invalidate_obj_cmd (connection c, char const inp[]) {
    /* For debugging purposes only */

    obj o;
    if (!get_persistent_obj_from_stream (c, inp, &o)) return;
    if (cache_obj_invalidate (o))
	print(c, "Object invalidated.\n");
    else 
	print(c, "Object is already a surrogate.\n");
}

static void info_obj_cmd (connection c, char const inp[]) {
    /* For debugging purposes only */
    obj o;
    core cp;
    Xref xref;
    char s[200];
    
    if (!get_persistent_obj_from_stream (c, inp, &o)) return;
    cp = cache_real_obj(o, &xref);
    sprintf(s, "Xref = %-d:%-d:%-d\t\tVnum = %-u\nHandle = %-d\t\tUid = %-u\n",
	    xref.or, OREF_SEGMENT(xref.oref), OREF_INDEX(xref.oref),
	    0, 172737, 17);    /* XXX Handle not being sent any more */
    print (c, s);
}

/*
  "pr_env" is an environment used by both "printExceptionalResult"
  and "printNormalResult". 
  */
struct pr_env {
  connection c;
  vec v; /* vec[any] */
  int i;
  exception e;			/* used only by printExceptionalResult */
  bool first;			/* used only by printNormalResult */
};

void pnr_loop_body(struct pr_env *env, type t)
/*
  Loop body for "printNormalResult"
  */
{
  any a = UNPV(any, vec_fetch(env->v, env->i++));
  if (!env->first) { print(env->c, ", "); }
  printValue(env->c, a);
  env->first = FALSE;
}

static void printNormalResult(connection c, method m, vec v /* vec[any] */)
{
  struct closure cl;
  struct pr_env env;
  cl.f = (ifunc)pnr_loop_body;
  cl.env = &env;
  env.c = c;
  env.v = v;
  env.i = 0;
  env.first = TRUE;
  method_rets(m, cl);
}

void per_loop_body(struct pr_env *env, string exc_name, list l)
/*
  Loop body for "printExceptionalResult"
  */
{
    if (string_equal(exc_name, env->e->name)) {
	bool fst = TRUE;
	loop {
	    any ta = first(l);
	    any va;
	    CATCH { exc = EXC_NONE; break; } /* end of list reached */
	    th_assert(isSubtype(class_as_type(get_any_class(ta)), Type),
		      "A signals list of some type contained things that"
		      "were not types.");
	    if (fst) print(env->c, " ");
	    else print(env->c, ", ");
	    va = obj_as_any(UNPV(obj, vec_fetch(env->v, env->i++)));
	    printValue(env->c, va);
	    l = rest(l);
	}
	exc = &exc_break; /* Theta break statement */
    }
}
		   
void printExceptionalResult(connection c, method m, vec v /* vec[any] */,
			    exception e)
{
    if (e == &exc_abort) {
	return;
    }
    if (e != &exc_failure) {
	struct closure cl;
	struct pr_env env;
	cl.f = (ifunc) per_loop_body;
	cl.env = &env;
	env.e = e;
	env.c = c;
	env.v = v;
	env.i = 0;
	method_sigs(m, cl);
    } else {
	print(c, " ");
	printValue(c, obj_as_any(vec_fetch(v, 0)));
    }
}


static void wellknown_cmd(connection c, char const line[])
{
  string name = string_newn(line, strcspn(line, whitespace));
  any value = get_wellknown(name);
  delete_string(name);
  CATCH {
    print(c, "6 Not a well-known value\n");
    exc = EXC_NONE;
    return;
  }
  print(c, "= ");
  printValue(c, value);
  print(c, "\n");
}

static void input_cmd(connection c, char const line[])
{
  print(c, "= ");
  printValue(c, (any)string_newn(line, strcspn(line, "\r\n")));
  print(c, "\n");
}

static void print_cmd(connection c, char const inp[])
{
  any a;
  class_ the_class;
  char buf[32];

  start_lex_input(string_input, &feyy_lineno);
  inp_posn = inp;
  accepted_inp_posn = inp;

  a = readValue(&exc);
  CATCH { printParseError(c, inp); return; }
      
  if (lex() != 0) {
    exc_value = "Expected end of input";
    printParseError(c, inp);
    return;
  }
  finish_lex_input();

  the_class = get_any_class(a);

  if (the_class == String) {
    string s = (string)any_get_obj(a);
    sprintf(buf, "D %d bytes\n", string_length(s));
    print(c, buf);
    connection_put(c, (string)any_get_obj(a));

    print(c, "\n");
  }
  else if (isSubtype(class_as_type(the_class), Type)) {
    connection_put(c, type_unparse((type)a));
  }

  else
    printValue(c, a);
    
  print(c, "\n");
}

static void stringlen_cmd(connection c, char const inp[])
{
    int tag;
    start_lex_input(string_input, &feyy_lineno);
    inp_posn = inp;
    
    accepted_inp_posn = inp;
    tag = lex();
    if (tag != IntegerTok) {
	exc_value = "Expected integer, e.g. 60";
	printParseError(c, inp);
	return;
    }
    max_string_display_length = feyy_lval.i;
    fprintf(stderr, "Maximum string display length set to %d\n", 
	    max_string_display_length);

    tag = lex();
    if (tag != 0) {
	exc_value = "Expected end of input";
	printParseError(c, inp);
	return;
    }

    finish_lex_input();
}

#if COLLECT_STATS

static void stats_cmd(connection c)
{
  char buf[STDBUFSIZE];
  sprintf(buf, 
	  "7 Stats: %ld ops, %ld surrogates created, %ld surrogates filled, %ld surrogates used, "
	  "%ld references swizzled, %ld hits, %ld misses, %ld OR fetches.\n", 
          stats->ops,
	  stats->surrogate_creations,
	  stats->surrogate_fills,
	  stats->surrogate_uses,
	  stats->swizzled_refs,
	  (stats->fixes - stats->cache_misses),
	  stats->cache_misses,
	  stats->fetches_sent);
  print(c, buf);
}

static void zero_stats_cmd(connection c)
{
  zero_stats();
  print(c, "= \n");
}

#endif /* COLLECT_STATS */

#endif /* !SHM */

bool typecheck_args(const method m, const vec args, int *mismatch)
/* 
  requires -  args is a vector of anys
  effects  -  checks that the args match the signature for method m 
   If the argument typechecks, returns true.  Otherwise returns
   false and sets mismatch to the index of the argument that
   failed to match .
*/

{
  int i; 
  for (i=0; i<vec_length(m->arguments); i++) {
    any arg = UNPV(any, vec_fetch(args, i));
    type arg_type = class_as_type(get_any_class(arg));
    formal f = UNPV(formal, vec_fetch(m->arguments, i));
    
    if (!isSubtype(arg_type, f->t)) {
      *mismatch = i;
      return FALSE; 
    }
  }
  return TRUE;
}
