// Copyright 1995 Barbara Liskov

/* currently assumes the same architecture for both the FE and the
/* client.  This assumption will usually be true because FE's and clients
/* often run on the same machine.
/* See binary_interface.h for a full description of the encoding.
*/


#include "config/vdefs/SHM.h"
#include "config/vdefs/USE_TIMER.h"
#include "config/vdefs/DISPLAY_CALLS.h"
#include "config/vdefs/COLLECT_STATS.h"
#include "config/vdefs/PROMISES.h"
#include "config/vdefs/LINKEDOO7_HACK.h"
#include "config/vdefs/COMPILER.h"
#include "runtime/except.h"
#include "runtime/force.h"
#include "binary_interface.h"
#include "types/type.h"
#if !SHM
#include "ascii_interface.h"
#endif
#include "handle.h"
#include "future.h"
#include "meth_handle_tab_iter.h"
#include "fast_invoke.h"
#include "types/pclass.h"
#include "types/interface.h"
#include "types/vec_class.h"
#include "types/vec_instns.h"
#include "boot/wellknown.h"
#include "runtime/obj_class.h"
#include "runtime/surr.h"
#include "runtime/disphdr.h"
#include "runtime/commit.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <stdarg.h>
#include "runtime/stats.h"
#include "types/string_class.h"
#include "types/class_class.h"
#include "types/objtype_class.h"
#include "types/exception.h"
#include "common/th_assert.h"
#include "common/intarray.h"
#include "cache/cache.h"
#include "cache/net.h"
#include "cache/indirect_table.h"
#include "cache/gc_register.h"
#include "types/method.h"
#include "types/cell.h"
#include "fe_config.h"

#include "common/other_unix.h"

#if USE_TIMER
#include "common/cyclecounter.h"
#endif

extern "C" {
#include <sys/time.h>
#include <sys/resource.h>
}

//#if PROMISES
#include "parse_tree.h"
#include "controls.h"
//#endif // PROMISES

#include "invoke.h"

extern "C" { void print_class(obj o); }

static void invoke_cmd(connection c, bool futures_specified, bool promises);
static void input_cmd(connection c);
static void print_cmd(connection c);
static void wellknown_cmd(connection c);
static void method_lookup_cmd(connection c);
static void convert_type_name(connection c);

static void get_results(fevalue* v, vec returns,
			int* send_arr, int num_rets);
static void send_results(connection c, int* send_arr, int retc);
static void put_results(connection c, fevalue* v, vec returns, int num_rets);
static bool is_basic_type(type t);
static void fe_put_string(connection c, char const *s);
static void init_interface_object (connection c);

#if USE_TIMER
bool use_timer = FALSE;
#endif

/* "magic_m_name" is a very badly-behaved "string" that cuts down 
   on superfluous allocation at invocations.  It should only be used to 
   hold a method name for long enough to look up the method.  Terrible 
   things may happen if it is used in any other way, since it is 
   mutated at every method invocation from the client. 
*/

#define MAX_METHOD_NAME_SIZE 50
static string magic_m_name;

#define MAX_CLIENT_RESULTS 50
// Gives the maximum number of return values in any reply to a client

extern bool message_from_server; // Declared in net.cc

const MAX_STATIC_RET_VALS = 100;
static fevalue return_buff[MAX_STATIC_RET_VALS];
/* Maximum number of return values from a method invocation*/

/* "gc_thrashing" is set when the gc time is longer than the inter-gc time. */
static bool gc_thrashing;

static bool initialized = FALSE;

static char errmsg[255];

static debug_calls = FALSE; // Set to TRUE along with DISPLAY_CALLS to display
// all the invoke commands

static int last_free_handle_count = 0;
// The number of the last free_handle call recevied by the FE

// Method handle table:
// Methods are unsigned indexes into the  method_handle_table.
// Use method_handle_atoh and method_handle_htoa to manipulate
// the table. 
//
// Comments: Having a separate method handle table helps
// checking that a handle is really a method handle and is faster
// since no reusing is necessary in this table.
//
static const unsigned MAX_METHOD_HANDLE_TABLE_SIZE = 8*1024;
static unsigned max_method_handle = 1;
static any method_handle_table[MAX_METHOD_HANDLE_TABLE_SIZE];


static inline unsigned method_handle_atoh(any a) {
    th_assert(max_method_handle < MAX_METHOD_HANDLE_TABLE_SIZE,
	      "Error: Out of space in method handle table");
    method_handle_table[max_method_handle] = 
	obj_as_any(get_storable_pointer(any_as_obj(a)));
    return max_method_handle++;
}

static inline any method_handle_htoa(unsigned h) {
    if (h > 0 && h < max_method_handle)
	return ACTUAL_ANY_POINTER(method_handle_table[h]);
 
    // The method handle is invalid throw an exception.
    exc_value = "Error: Invalid Method handle\n";
    exc = &exc_not_possible;
    return (any) 0;
}

Method_handle_table_iter::Method_handle_table_iter() {
    cur = 1;
}

Method_handle_table_iter::~Method_handle_table_iter() {
}

bool Method_handle_table_iter::get(obj& o) {
    if (cur >= max_method_handle)
	return FALSE;
    o = any_as_obj(method_handle_table[cur]);
    cur++;
    return TRUE;
}

void Method_handle_table_iter::replace(obj o) {
    th_assert(cur <= max_method_handle && cur, "Bad iterator value");
    method_handle_table[cur - 1] = obj_as_any(o);
}


// XXX
// The following assumes the vectors are not shrunk. This assumption must
// hold for method objects in the method handle table.
// XXX
static inline unsigned fast_unsafe_vec_length(vec v) {
    return (* ((vecdv) (v)->hdr.methods)->length)(v);
}

static inline pval *fast_unsafe_vec_items(vec v) {
    return v->items;
}

void send_exception(connection c, method m) {
    // effects: Send the exception given in the global variable "exc" on
    //          connection c. m is the current method
    //          This procedure resynchronizes the Fe-client connection also.
    
    // The Exception Protocol:
    // Shared Memory --- The FE sets the immediate flag, sends the
    // exception and waits for the flag to be reset by the client. In the
    // meanwhile, the client checks for this flag at "safe points", gets the
    // exception, cleans the client-to-FE and FE-to-client channels and sets
    // the flag to FALSE.
    
    // Sockets --- The FE sends the exception to the client and waits
    // for a resynchronization message from the client ('R' message). Whenever
    // the client receives an exception, it sends an 'R' message
    // and waits for an '=' message

#if SHM
    c->set_immediate_flag();
#endif

    int send_arr[MAX_CLIENT_RESULTS];
    string exception_string = exc->name;
    vec types = get_exc_types(m); 
    exc = EXC_NONE;
    int len = types? vec_length(types) : 0;
	
    // Send the actual exception to the client
    get_results(TH_EXC_VALUE, types, send_arr, len);
    char const *exc_str = string_charp(exception_string);
    // Note: None of the following calls should result in abort
    // i.e no fetches from server. Otherwise, the connection buffer
    // between the FE and client will have inconsistent data
    connection_putc(c, '!');
    fe_put_string(c, exc_str);
    connection_fputi(c, len);
    send_results(c, send_arr, len);
    connection_fflush(c);

    // Resynchronize the connection now
#if SHM
    connection_resynchronize(c);
#else
    connection_resynchronize(c, 'R');
    connection_fputc(c, '=');
    connection_fflush(c);
#endif
}

void send_abort_to_client(connection c, bool if_immediate) {
    // effects: Send the abort exception to the client.

    exc = &exc_abort;
    // Setting futures_specified to FALSE here. Is that ok ? XXX
    if (if_immediate)
	fprintf(stderr, "Immediate abort at FE\n");
    send_exception(c, NULL);
}

bool binarySingleOp(connection c, obj env, FILE *f) {
    // requires: initDispatch has been called
    // effects:  Keeps waiting for commands from the client and executes them
    // returns:  TRUE if EOF on connection c

    // Important:
    // While sending any results to the client, the Fe must ensure that
    // there are no aborts in between (i.e. no fetches). Otherwise, the
    // connection buffer will be inconsistent. That is why in some of the
    // places, the code first gets the handle in a variable or an array of
    // handles and then sends it to the client (since handle_atoh can
    // result in a fetch)

    static IntArray handles, futures;
    static first_time = TRUE;
    bool aborted;
    th_assert(initialized, "Dispatch information has not been initialized");

    STATS(stats->ops++);

    if (first_time) {
	init_interface_object(c); // Initialize the special object Iobj
#if !COMPILER
	allow_meta = FALSE;
	cache_correct_meta_heap();
	if (!get_or_stats(FEConf->initial_or_num, initial_stat))
	    return 0;
#endif
	first_time = FALSE;
    }


    // The outer loop is executed every time a longjmp is performed by the 
    // FE code. This happens whenever there is an abort. So the setjmp call
    // is made on every abort and not every call

    while (1) {
	int to_abort = CATCH_ABORT();
	if (to_abort) {
	    // A longjmp has been made. Send abort exception to client
	    send_abort_to_client(c, TRUE);
	    continue; // So that the real setjmp gets called now
	}
	// Note: Code not indented since the rest of the code looks ugly then
    while (1) {
	char ch = connection_getc(c);
	
	// If seems to be faster.
	if (ch == 'I') {
	    invoke_cmd(c, FALSE, FALSE);
	} else if (ch == 'J') {
	    invoke_cmd(c, TRUE, FALSE);
	} else 
	    switch(ch) {

	    case EOF: {
		return TRUE;
		break;
	    }

	    case 'H': { // free handle
		int call_num =  connection_fgeti(c);
		th_assert(call_num == last_free_handle_count + 1 ||
			  call_num == last_free_handle_count,
			  "Bad free handle call");
		int num_handles =  connection_fgeti(c);
		handles.clear();
		handles._enlarge_by(num_handles+1);
		connection_fread(c, handles.as_pointer(), 
				 num_handles*sizeof(int));
		if (call_num == last_free_handle_count)
		    break; // Repeeated call received by FE
		FEConf->ht->free_handles(handles.as_pointer(), num_handles);
		connection_fputc(c, '=');
		connection_fflush(c);
		last_free_handle_count = call_num;
		break;
	    }

	    case 'F': {
		// Send has to be "atomic", i.e. no fetches while the FE
		// is sending a message to the client
		int nfh = FEConf->ft->num_future_handles();
		handles.clear(); futures.clear();
		handles._enlarge_by(nfh+1); futures._enlarge_by(nfh+1);
		FEConf->ft->get_future_handles(futures, handles);
		th_assert(nfh+1 == handles.size(),
			  "Wrong size while sending future remappings");

		// Send the data now atomically
		connection_fputc(c, '=');
		connection_fwrite(c, &nfh, sizeof(int));
		for (int i = 1; i < nfh+1; i++) {
		    future f = futures[i];
		    connection_fwrite(c, &f, sizeof(i));
		    connection_fwrite(c, &handles[i], sizeof(handles[i]));
		}
		connection_fflush(c);
		break;
	    }
	    case 'W': {		/* wellknown */
		wellknown_cmd(c); break;
	    }
	    case 'L': {
		method_lookup_cmd(c); break;
	    }
	    case 'N': {
		input_cmd(c); break;
	    }
	    case 'P': {
		print_cmd(c); break;
	    }  
	    case 'C': {
	        convert_type_name(c); break;
	    }

#if PROMISES
	    case 'K': {		/* future invoke with promises */
		invoke_cmd(c, TRUE, TRUE); break;
	    }
	    case 'D': { // Start a new control structure.
		if (!begin_control(c)) {
		    exc = &exc_not_possible;
		}
		break;
	    }
	    case 'E': { // End a control structure.
		if (!end_control(c)) {
		    exc = &exc_not_possible;
		}
		break;
	    }
	    case 'M': { // Invocation inside a control structure.
		if (!control_level)
		    exc = &exc_not_possible;
	        else
		    control_invoke(c);
		break;
	    }
	    case 'U': { /* create a new cell, and return a handle to it. */
		cell c_ = new_cell();
		if (!c_) {
		    exc = &exc_not_possible;
		    break;
		}

		handle h = FEConf->ht->any_to_handle((any)c_);
		th_assert(h, "Failed to allocate handle for cell!\n");
		connection_fputc(c, '='); // ?
		connection_fwrite(c, &h, sizeof(int));
		connection_fflush(c);
		break;
	    }
	    case 'V': { /* create a new cell, and return a future to it. */
		cell c_ = new_cell();
		th_assert(c_, "Failed to allocate a new cell!\n");
		future f;
		connection_fread(c, &f, sizeof(int));
		th_assert(f < 0, "Illegal future supplied to new_cell");
		FEConf->ft->any_to_future((any)c_, f);
		break;
	    }
#endif

	    default: {
		sprintf(errmsg, "Unrecognized binary command: %c [%d]\n", 
			ch, (int)ch);         
		exc_value = errmsg;
		exc = &exc_not_possible;
	    }
	}

	if (!Iobj)
	    // A close request received from the client and
	    // Fe was not client's slave
	    return TRUE;
	CATCH {  
            /* if the client misbehaves, terminate the connection */
	    if (exc_value) 
		fputs((char *)exc_value, stderr);
	    
#if !SHM
	    connection_setDisp(c, ascii_cl);
#endif
	    connection_close(c);
	    fprintf(stderr, "Exception caught\n"); 
	    exc = EXC_NONE; // reset the exception for another connection
	    return TRUE;
	}
	if (message_from_server) {
	    bool or_res = read_from_net(NULL, 0, FE_NO_BLOCK, &aborted,
					FALSE, NULL, WAIT_CLIENT);
	    th_assert(or_res, "Error on FE-OR connection");
	    if (aborted)
		// Transaction aborted
		send_abort_to_client(c, TRUE);
	}
#if INSANELY_AGGRESSIVE_GC
	cache_force_gc();
#else
      	gc_thrashing = cache_finished_op();
#endif

    }
    }  // For the CATCH_ABORT loop
    return FALSE;
}

static void init_interface_object (connection c) {
    // effects: Initialize the special FE object "Iobj". Iobj will now
    //          correspond to the client connection specified by "c"
    bool saved = normal_heap;
    normal_heap = FALSE;
    Iobj = interface_new(c);
    add_wellknown("Iobj", (obj *)&Iobj);
    normal_heap = saved;
}

void initDispatch() {
    char junk[MAX_METHOD_NAME_SIZE];
    gc_thrashing = FALSE;
    if (initialized) return;
    bool saved = normal_heap;
    normal_heap = FALSE;
    magic_m_name = string_newn(junk, MAX_METHOD_NAME_SIZE);
    gc_register_meta_root((obj *)&magic_m_name);
    initialized = TRUE;
    normal_heap =  saved;
}


void binaryDispatch(connection c, obj env, FILE *f) {
    initDispatch();
    binarySingleOp(c, env, f);   
}

/* Use readbuf for reading reasonable-sized strings, only 
   allocate dynamically for big strings */

static char readbuf[BUFSIZ];

char *read_string(connection c, int *length/* out */,
		  bool *allocated/* out */) {
    // Read a string value from the connection. Return a character pointer to the
    // string. If "*allocated", the character pointer should be freed after you
    // are done with it. Otherwise, the pointer will point to a static region
    // that may be overwritten by subsequent calls. The string will be
    // null-terminated. Its length (notincluding the null) is in "*length". 
    char *buffer;
    int len; // the length including the null
    connection_fread(c, length, sizeof(int));
    len = *length;
    *length = len -1;
    if (len > BUFSIZ) {
	buffer = (char *)malloc(len);
	*allocated = TRUE;
    } else {
	buffer = readbuf;
	*allocated = FALSE;
    }
#if !SHM
    connection_fread(c, buffer, len);
#else
    c->read((int *)buffer, (len + sizeof(int) - 1)/sizeof(int));
#endif
    return buffer;
}


// Code for reading values.
fevalue readValue(connection c, type &t) {
    fevalue result;
    char ch = connection_getc(c);

    switch (ch) {
    case 'i':  {
	result.i = connection_fgeti(c);
	t = (type)Int;
	break;
    }
    case 'c': {
	result.i = connection_getc(c);
	t = (type)Char;
	break;
    }
    case 'b': {
	result.i = connection_getc(c);
	t = (type)Bool;
	break;
    }
    case 'n':{
	// An integer gets sent, even though it is not necessary...
	result.i = connection_fgeti(c);
	t = (type)Null;
	break;
    }
    case 'h': {
	handle_val res = FEConf->ht->handle_to_any(connection_fgeti(c), TRUE);
	switch(res.tag) {
	case Anyf:
	    result.o = (obj)res.val;
	    t = (type)Any;
	    break;
	case Intf:
	    result.i = (long)res.val;
	    t = (type)Int;
	    break;
	  
	case Boolf:
	    result.b = (bool)(long)res.val;
	    t = (type)Bool;
	    break;
	    
	case Charf:
	    result.c = (char)(long)res.val;
	    t = (type)Char;
	    break;
	      
	case Realf:
	    result.r = (real)(long)res.val;
	    t = (type)Real;
	    break;
	}
	break;
    }
    default: {
	static char buf[128];
	sprintf(buf, " readValue: Unrecognized type code %c [%d]\n",
		ch, (int)ch);
	exc_value = buf;
	exc = &exc_not_possible;
	return result;
	break;
    }
    }
    return result;
}


// With promises basic values need to be tagged without promises they
// don't the following function and macro hide the different encodings.
//
#if PROMISES
// Read arguments of basic types, the type parameter is not relevant.
static inline
int readBasic(connection c) 
{
    type t; 
    return readValue(c, t).i;
}
#else
#define readBasic(c) connection_fgeti(c)
#endif



/* Special low-overhead mechanism for reading method names.
   The string returned must only be used for looking up 
   methods; it is a very badly-behaved string (it shares storage
   with the strings produced on other calls to this routine) 

   May signal "not_possible" if the method name is too long.
*/
static string read_method_name(connection c) {
    any v;

    int len;
    string result;

    connection_fread(c, &len, sizeof(int));
#if !SHM
    if (len <= MAX_METHOD_NAME_SIZE) {
	connection_fread(c, magic_m_name->chars,  len);
#else
    if (((len + sizeof(int) - 1)&(-sizeof(int))) <= MAX_METHOD_NAME_SIZE) {
	c->read((int *)magic_m_name->chars,
		(len + sizeof(int) - 1)/(sizeof(int)));
#endif
	magic_m_name->size = len - 1;
	return magic_m_name;
    } else {
	exc_value = "Method name is too long\n";
	exc = &exc_not_possible;
    }
}    

// Accumulate the actual arguments to pass to the function, as
// described in the file "calling_conventions". 

// return true if the arguments are the right number and type and
// false otherwise.

static inline int read_and_setup_arguments(any receiver, connection c,
					   method m, fevalue *args,
					   int nformals, int args_index) {
    formal *forms = UNPV(formal *, fast_unsafe_vec_items(m->arguments));
    type t;

    for (int i = 0; i < nformals; i++) {
	formal f = forms[i];
	th_assert(f->hdr.methods == &formal_methods, 
		  "Not a formal object - perhaps a surrogate ?");
		        
	// XXX 
        // This assumes that any item sent in the connection
	// can be stored in an integer and therefore can be stored
        // in an obj (i.e. sizeof(anyitem) <= sizeof(int) <= sizeof(obj)).
        // XXX

	// Don't need to check basic types.
	if (is_basic_type(f->t)) {
            // Handle arguments of basic types: the fast case
            // no type checking.
	    args[args_index].i = readBasic(c);
	}
#if PROMISES
	else if (f->t == (type) Any) {
            // Handle arguments of type Any. Must create
            // an Any of the right type.
 	    fevalue v = readValue(c, t);

 	    if (exc)
 		return FALSE;
	    //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	    // Another gross cell related hack.  I really don\'t like
	    // this!  --Qyz
	    if (m->self_type == (type)cellC) {// Put.  Get has no arguments.
	      args[args_index] = v; // Don\'t wrap it as any.
	      cell_put_type((cell)receiver, t); // Remember the type.
	    }
	    else
	    //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	      args[args_index].o = any_as_obj(make_any(t, v));
	}
#endif
        else {                            
            // Handle arguments of non-basic types.
	    // Get object from handle table.
	    handle hval = connection_fgeti(c);
	    handle_val res = FEConf->ht->handle_to_any(hval, TRUE);
	   
	    if (exc || res.tag != Anyf)
		// The handle was invalid or it was a promise.
		return FALSE;
	    
	    // Check object argument type conforms with method
            // signature and cast argument to expected type.
	    obj o = any_as_obj(res.val);
	    objtype o_type = (objtype)get_obj_class(o);
    
	    if (o_type == (objtype) f->t)
		// The fast case -- the object actual type 
                // equals the expected type
		args[args_index].o = (obj)((DV)o);
	    else {
		int hdroff = hdr_offset(o_type, (objtype) f->t);
		
		if (exc) 
		    // The argument object is not of the right type.
		    return FALSE;
		
		args[args_index].o = (obj)(((DV)o)+hdroff);
	    }
	}
	
	args_index++;
    }
    
    return (args_index == nformals);
}
					   
static void invoke_cmd(connection c, bool futures_specified, bool promises) {
    type t;

    // First, obtain the value that will receive the invocation.
#if PROMISES
    any v = (any) readValue(c, t).o;
#else
    int h = connection_fgeti(c);
    handle_val res = FEConf->ht->handle_to_any(h, TRUE);

    if (exc) {
	fprintf(stderr, "INVALID receiver handle %d\n", h);
	return;
    }

    t = (type)Any;
    any v = (any)res.val;
#endif

    // Read the method handle and look up the corresponding method object
    int method_handle = connection_fgeti(c);
    method m = (method)method_handle_htoa(method_handle);

    if (exc) {
	fprintf(stderr, "INVALID method handle\n");
	return;
    }

    if (method_isIter(m)) {
      exc_value = "Iterators not supported\n";
      exc = &exc_unsupported;
    }

    /* Read and setup the arguments  */
    int num_rets = fast_unsafe_vec_length(m->returns);
    int num_args = fast_unsafe_vec_length(m->arguments);
    static fevalue args[MAX_ARGS];
    int args_index = 0;

    // Save first slot in args vector for return buffer if 
    // method returns more than 1 value.
    if (num_rets > 1) {
	args[0].o = (obj)return_buff;
	args_index = 1;
    }

    if (num_args > 0 && 
	!read_and_setup_arguments(v, c,  m, args, num_args, args_index)) {
	fprintf(stderr, "Wrong argument type or count\n");
	// The method arguments are either the wrong number or type.
	exc_value = "Error: Invalid arguments\n";
	exc = &exc_not_possible;
    }
	
    dtentry *dte;
    obj receiver;
    ofunc f;
      
    /* Fetch the dispatch entry from the object.
       Get the receiver's actual type */

    receiver = any_as_obj(v);

    if (t != (type) Any) {        // Receiver is a basic value.
	if (t != m->self_type) {
	    exc_value = "Error: Invalid receiver type\n";
	    exc = &exc_not_possible;
	}
	RESIGNAL_ANY_EXC;
    	
        f = compute_basic_func(m);
    } else {
	objtype actual_type = (objtype) get_obj_class(receiver);

	/*
	  Check that object expects the invoked method and 
	  cast it to the appropriate type.
	*/ 
	if (actual_type == (objtype)m->self_type) {
            /* The receiver's type equals the expected type */
	    /* avoid call to hdr-offset. */
	    RESIGNAL_ANY_EXC;
        } else {
	    int hdroff = hdr_offset(actual_type, (objtype)m->self_type);    
	    RESIGNAL_ANY_EXC;
	    receiver = (obj)(((DV)receiver)+hdroff);
	}
    
        f = compute_any_func(receiver, m);
    }
 
    int result_futures[9]; 
    int future;
    
    if (futures_specified) {  
	type *returns = (type *)fast_unsafe_vec_items(m->returns);
	for (int i=0; i < num_rets; i++) {
	    if (promises || !is_basic_type(returns[i])) {
		connection_fread(c, &future, sizeof(int));
		result_futures[i] = future;
	    }
	}
    }

#if USE_TIMER
    unsigned long int timer;
    if (use_timer) TIMER_START(timer);
#endif

    volatile int num_rets_save = num_rets;
    jmp_entry tmp;
    FAST_TOP_LEVEL_FAILURE(tmp);
    /* Set the top level failure. Go read the comments in "runtime/except.h"
       if you don't understand what is going on here. */
    
    /* The actual method call, the number of arguments includes the receiver 
       and, if num_rets > 1, the buffer for the multiple return values. */
    fevalue result;
    if (!exc) {
#if DISPLAY_CALLS
	if (debug_calls) {
	    fprintf(stderr, "Result: %x: ", result);
	    fprintf(stderr, "Called method %s = <%x> = #%d ",
		    string_charp(m->name), f, method_handle);
	    fprintf(stderr, "on object: %d, Class = ", h);
	    print_class(receiver);
	}
#endif
	result = perform_call(receiver, f, args,
			      num_args+((num_rets > 1) ? 2 : 1));
    }


#if USE_TIMER
    if (use_timer) {
	TIMER_STOP(timer);
	fprintf(stderr, "%10lu\n", timer);
    }
#endif

    /* 
     * Send the results 
     */
    if (!exc)  {
	if (num_rets == 1)
	    return_buff[0] = result;

	// Else what??  Where do the other results get put in
	// return_buff? --Qyz

	if (futures_specified) {
	    type *returns = (type *)fast_unsafe_vec_items(m->returns);
	    
	    for (int i=0; i<num_rets; i++) {
#if PROMISES
	      //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	      // This is a hack that will go away when we have real
	      //parameterized types.  See "types/cell.h" for explanation. --Qyz
	      if (m->self_type == (type)cellC) { // Get. Put has no results.

		// Map f to the correct type in the future table.
		FEConf->ft->basic_to_future(cell_get_type((cell)receiver), 
			    return_buff[i], result_futures[i]);
	      }
	      //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	      else
#endif
		if (promises && is_basic_type(returns[i])) {
		    FEConf->ft->basic_to_future(returns[i], return_buff[i],
						result_futures[i]);
		} else {
		    FEConf->ft->any_to_future(obj_as_any(return_buff[i].o),
				result_futures[i]);
		}
	    }
	} else {
	    int send_arr[MAX_CLIENT_RESULTS];
	    get_results(return_buff, m->returns, send_arr, num_rets);
	    // Note: None of the following calls should result in abort
	    // i.e no fetches from server. Otherwise, the connection buffer
	    // between the FE and client will have inconsistent data
	    connection_putc(c, '=');
	    send_results(c, send_arr, num_rets);
	    connection_fflush(c);
	}
    } else {
	// Exception has to be sent
	// local variables used by this section from above must be stored
	// into volatile variable before the call:
	//
	num_rets = num_rets_save;
	send_exception(c, m);
    }
}

static inline void get_results(fevalue *results, vec types_vec,
			       int* send_arr, int retc) {
    // requires: send_arr has space for retc elements
    // effects: Gets the handles/basic values given in results and puts
    //          them in send_arr retc is the number of return values and
    //          types_vec contains information about their types

    type *types = types_vec? UNPV(type*, fast_unsafe_vec_items(types_vec))
				: NULL;

    th_assert(retc <= MAX_CLIENT_RESULTS, "too many client results");
    for (int i=0; i<retc; i++) {
      if (is_basic_type(types[i]))
	  send_arr[i] = results[i].i;
      else
	  // Note: There could be an abort here since handle_atoh
	  // may result in a fetch
	  send_arr[i] = FEConf->ht->any_to_handle((any) results[i].o);
    }
}

static inline void send_results(connection c, int* send_arr, int retc)
{
    th_assert(retc <= MAX_CLIENT_RESULTS, "Too many client results");
    for (int i=0; i<retc; i++)
	  connection_fputi(c, send_arr[i]);
}

static inline void put_results(connection c, fevalue *results,
			       vec types_vec, int retc)
{
    type *types = UNPV(type*, fast_unsafe_vec_items(types_vec));

    for (int i=0; i<retc; i++)
      if (is_basic_type(types[i]))
	  connection_fputi(c, results[i].i);
      else {
	  connection_fputi(c, FEConf->ht->any_to_handle((any) results[i].o));
      }
}


void fe_put_string(connection c, char const *s)  
    /* effects - send to the client the string length (including the
       terminating null), followed by the chars of the string.
       For shared memory, pad the string with arbitrary data to
       align the output.
       */
{
#if !SHM
    int len = strlen(s)+1;
    
    connection_fwrite(c, &len, sizeof(int));
    connection_fwrite(c, s, len);
#else
    int len = strlen(s) + 1;
    c->put(len);
    c->write((int *)s, (len + 3) / sizeof(int));
#endif
}

bool fe_get_string(connection c, char **s, int *len)  
    //   effects - read from the client the character s, followed by the string
    //   length (including the terminating null), followed by the chars of the
    //   string. After the call, "*len" contains the number of characters
    //   in the string, including the null terminator.
{
#if !SHM
    connection_fread(c, len, sizeof(int));
    *s = (char *)malloc(*len);
    connection_fread(c, *s, *len);
#else
    *len = c->get();
    int len2 = (*len + sizeof(int) - 1)/sizeof(int);
    *s = (char *)malloc(len2 * sizeof(int));
    c->read((int *)*s, len2);
#endif
    return TRUE;
}



static void method_lookup_cmd(connection c) {
    char *name;
    int namelen;
    
    bool ok = fe_get_string(c, &name, &namelen);
    assert(ok);
    string s = string_newn(name, namelen-1);
    free(name);
    int method_index;
    connection_fread(c, &method_index, sizeof(int));

    objtype t = (objtype)get_wellknown(s);
    CATCH {		/* wellknown not found */
	return;
    }

    if (method_index<0 || (vec_length(t->methods_) <= method_index)) {
/* index out of range */
	th_fail("Invalid method index\n");
	return;
    }

    int method_handle = method_handle_atoh(UNPV(any, vec_fetch(t->methods_, method_index)));
    
    th_assert(!exc, "Exception signalled by vec_fetch\n");
    connection_fputc(c, '=');
    connection_fputi(c, method_handle);
    connection_fflush(c);
}



static 
void wellknown_cmd(connection c)
{
    string s;
    char *name;
    bool ok;
    int namelen;
    
    ok = fe_get_string(c, &name, &namelen);
    assert(ok);
    s = string_newn(name, namelen-1);
    obj o = any_as_obj(get_wellknown(s));
    delete_string(s);
    free(name);

    CATCH {		/* wellknown not found */
	return;
    }
    
    handle tosend = FEConf->ht->any_to_handle(obj_as_any(o));
    // Important: Get the handle before putting anything in the buffer
    // since an abort may ruin things
    connection_putc(c, '=');
    connection_fputi(c, tosend);
    connection_fflush(c);
}

static void input_cmd(connection c)
{
    char *s;
    int len;
    bool alloc;
    string st;

    s = read_string(c, &len, &alloc);
    st = string_newn(s, len);
    if (alloc) free(s);

    handle tosend = FEConf->ht->any_to_handle(obj_as_any((obj)st));
    // Important: Get the handle before putting anything in the buffer
    // since an abort may ruin things
    connection_putc(c, '=');
    connection_fputi(c, tosend);
    connection_fflush(c);
}


static void print_cmd(connection c)
{
    any a;
    class_ the_class;

    // can only get handles string  objects.    
    handle_val res = FEConf->ht->handle_to_any(connection_fgeti(c), TRUE);
    RESIGNAL_ANY_EXC;

    // is a FIX needed here XXXX?
    a = res.val;
    the_class = get_any_class(a);
    
    if (the_class ==  String) {
        string s = (string)any_get_obj(a);
	char const *res = string_charp(s);
        connection_putc(c, '=');
        fe_put_string(c, res);
    } else {
	exc_value = "can only print strings";
	exc = &exc_not_possible;
	return;
    }
    
    connection_fflush(c);
}


static void convert_type_name(connection c)
{
    any a;
    class_ the_class;

    // can only get handles string  objects.    
    handle_val res = FEConf->ht->handle_to_any(connection_fgeti(c), TRUE);
    RESIGNAL_ANY_EXC;

    // is a FIX needed here XXXX?
    a = res.val;
    the_class = get_any_class(a);
    
    if (the_class !=  String) {
	exc_value = "can only convert strings";
	exc = &exc_not_possible;
	return;
      }

    string extNm = (string)any_get_obj(a);
    string intNm = typeExtNm2IntNm(extNm);

    handle h = FEConf->ht->any_to_handle((any)intNm);
    th_assert(h, "Failed to allocate handle for intName string!\n");
    connection_fputc(c, '='); // ?
    connection_fwrite(c, &h, sizeof(int));
    connection_fflush(c);
}
