// Copyright 1995 Barbara Liskov

#include "controls.h"
#include "handle.h"
#include "runtime/except.h"
#include "runtime/disphdr.h"
#include "config/vdefs/DEBUG_CONTROLS.h"
#include "type_check.h"

#define  INITIALSIZE  100       // Initial size of type checking table.

static Node* Tree;
static Node* Stack;
static Node* Current;

int control_level = 0;

exception control_exc = EXC_NONE;
void *control_exc_value = 0;

static void initialize() {
#if DEBUG_CONTROLS
  fprintf(stderr, "Starting new Control Structure Batch\n");
#endif
  init_type_table(INITIALSIZE);
  Current = Tree = Stack = 0;
}

inline void push_stack(Node* n) {
//  fprintf(stderr, "Pushing Node:\n");
//  n->print_node();
  n->down = Stack;
  Stack = n;
}

inline Node* pop_stack() {
  Node *ret = Stack;
  if (Stack) {
    Stack = Stack->down;
//    fprintf(stderr, "Popping Node:\n");
//    ret->print_node();
  }
  return ret;
}

static inline void new_control_node() {
  control_level++;
  if (Current) {
    push_stack(Current);
  }
}

static inline void end_control_node() {
  Node *tmp;

  if (tmp = pop_stack()) {
    tmp->add_node(Current);
    Current = tmp;
  }
  else {
    Tree = Current;
  }
}

static inline void splice_node_list(Node *l) {
  // Splice a list of nodes into the current tree.
  Node *tmp;

  // fprintf(stderr, "Splicing list\n");
  if (tmp = pop_stack()) {
    Current = tmp;
    while (l) {
      tmp->add_node(l);
      l = l->next;
    }
  }
  else {
    Tree = l;
    if (l)
      while (l->next)
	l = l->next;
    Current = l;
  }
}

static inline VarNode* get_condition(connection c) {
  char tag;

  tag = connection_getc(c);
  
  // fprintf(stderr, "Got tag: %c\n", tag);

  if (tag == 'h') {
    int handle;
    connection_fread(c, &handle, sizeof(int));
    // th_assert(handle < 0, "Basic value handle must be future!");
    // fprintf(stderr, "Got handle: %d\n", handle);
    obj_ref r;
    type t = find_type(handle, r);
    return new FutureNode(r);
  }
  else if (tag == 'b') {
    bool val;
    val = connection_getc(c);
    // fprintf(stderr, "Got val: %d\n", val);
    return new ValNode((any)val);
  }
  else {
    fprintf(stderr, "Illegal type for condition variable!\n");
    return 0;
  }
}

bool begin_control(connection c)
{
  char ch = connection_getc(c);
  VarNode *cond;

  if (control_level == 0)
    initialize();

  // fprintf(stderr, "Current = %p\n", Current);

  switch (ch) {
    // Cases for the IF control structure.
    case 'f':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin If Expr\n");
#endif
      new_control_node();
      Current = new IfNode();
      break;
    }

    case 'g':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin If Body\n");
#endif
      if (Current->tag != If_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }

      if (cond = get_condition(c))
	((IfNode*)Current)->begin_then(cond);
      else
	return FALSE;

      break;
    }

    case 'h':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin Else\n");
#endif
      if (Current->tag != If_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }
      ((IfNode*)Current)->begin_else();

      break;
    }

    // Cases for the LOOP control structure.
    case 'l':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin Loop Expr\n");
#endif
      new_control_node();
      Current = new LoopNode();

      break;
    }

    case 'm':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin Loop Body\n");
#endif
      if (Current->tag != Loop_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }

      if (cond = get_condition(c))
	((LoopNode*)Current)->begin_body(cond);
      else
	return FALSE;

      break;
    }

    // Cases for the FOREACH control structure.
    case 'x':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "Begin Foreach body\n");
#endif

      new_control_node();
      Current = new IterNode();

      break;
    }

    default:
    {
      fprintf(stderr, "Unknown control structure type %c\n", ch);
      return FALSE;
    }
  }

  return TRUE;

#if DEBUG_CONTROLS
  fprintf(stderr, "Control_level = %d\n", control_level);
#endif
}

bool end_control(connection c)
{
  char ch = connection_getc(c);

  switch (ch) {
    // END_IF
    case 'g':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "End If\n");
#endif
      if (Current->tag != If_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }

      //Try to optimize away IfNode
      Node *tmp = ((IfNode*)Current)->optimize();
      if (Current != tmp) {
#if DEBUG_CONTROLS
	fprintf(stderr, "Optimizing IF node\n");
#endif
	splice_node_list(tmp);
      } 
      else
	end_control_node();

      break;
    }

    // END_LOOP
    case 'm':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "End Loop\n");
#endif
      if (Current->tag != Loop_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }
      end_control_node();

      break;
    }

    // END_LOOP
    case 'y':
    {
#if DEBUG_CONTROLS
      fprintf(stderr, "End Foreach\n");
#endif

      if (Current->tag != Iter_Node) {
	fprintf(stderr, "Illegal control command\n");
	return FALSE;
      }

      ((IterNode*)Current)->end_body();

      end_control_node();

      break;
    }

    default:
    {
      fprintf(stderr, "Illegal control structure type %c\n", ch);
      return FALSE;
    }
  }

  if (--control_level == 0) {

#if DEBUG_CONTROLS
    fprintf(stderr, "\n**Parse tree: \n");
    print_tree(Tree);
    fprintf(stderr, "\n**Type table: \n");
    print_type_table();
#endif

    jmp_entry tmp;
    FAST_TOP_LEVEL_FAILURE(tmp);  
    
    if (exc) {
      fprintf(stderr, "Exception signaled during evaluation!\n");
      //XXX Set all result futures????
    }
    else {
      eval_tree(Tree);
#if DEBUG_CONTROLS
      fprintf(stderr, "\n**Evaluation succeeded.\n");
#endif
    }

    delete Tree;
    destroy_type_table();
  }

#if DEBUG_CONTROLS  
  fprintf(stderr, "Control_level = %d\n", control_level);
#endif

  return TRUE;
}

bool add_method_node(MethodNode *m) {
  // fprintf(stderr, "Adding method node to tree\n");
  // m->print_node();

  if (!Current) {
    fprintf(stderr, "No parse tree currently exists!!!\n");
    return FALSE;
  }
  return Current->add_node(m);
}



#if PROMISES
inline void control_readValue(connection c, char &t, int &v) {
  // Read a tagged value inside a control structure (don\'t expand the
  // type, and don\'t get the object from the handle.

  t = connection_getc(c);  // Read the tag.

  switch(t) {
    case 'h':
    case 'i':
    case 'r':
      v = connection_fgeti(c);  // Read the value as an integer.
      break;
    case 'c':
    case 'b':
      v = connection_getc(c);  // Read the value as a character.
      break;
    default: {
      //XXX Send exception?
      fprintf(stderr, "control_readValue: Unrecognized type code %c [%d]\n",
	      t, (int)t);
      th_fail("Unrecognized type code");
      break;
    }
  }
}

void control_invoke(connection c) {
//  This function is almost identical to invoke_cmd, except that it
//  *statically* type checks the method call, and does not execute the
//  method.  Instead, it reads in the information and passes it to the
//  control-structure code to construct a parse-tree node. (controls.cc).

  type t;
  VarNode *self;
  int i;
  obj_ref r;

  //fprintf(stderr, "Starting control invoke\n");

  // Read the receiver.  This could be optimized if we knew the
  // self-type in advance (i.e. no tag required for non-basic types)!!

  char ch;
  int val;

  control_readValue(c, ch, val);

  // Read the method handle and look up the corresponding method object
  method m = (method)method_handle_htoa(connection_fgeti(c)); 
  
  if (exc) {  // XXX
    fprintf(stderr, "INVALID receiver or method handle\n");
    
    // Either the object or method handles were invalid.
    return;
  }

  // Construct the Node for the receiver, and type check.
  if (ch == 'h') {
    obj_ref r;

    if (val > 0) {
      handle_val res = FEConf->ht->handle_to_any(val, TRUE);
      if (res.tag != Anyf)
	; // XXX Signal exception!!!
      any s = res.val;
      t = get_type(res);
      int hdroff = type_check(t, m->self_type);
      RESIGNAL_ANY_EXC;
      self = new ValNode(s, hdroff);
    }
    else {
      t = find_type(val, r);
      if (is_basic_type(m->self_type))
	self = new FutureBasicNode(r);	// No need to type check.
      else {
	int hdroff = type_check(t, m->self_type);
	RESIGNAL_ANY_EXC;
	if (hdroff)
	  self = new FutureOffsetNode(r, hdroff);
	else
	  self = new FutureNode(r);
      }
    }
  }
  else
    self = new BasicValNode((any)val);     //Basic value receiver.

  // fprintf(stderr, "Processed receiver\n");

  // Read the arguments

  int num_args = fast_unsafe_vec_length(m->arguments);
  int nformals = num_args;
  int args_index = 0;

  // XXX Multiple returns??

  VarNode **args;

  formal *forms = UNPV(formal *, fast_unsafe_vec_items(m->arguments));

  if (method_isIter(m)) {
    args = new VarNode*[num_args + 2];
    args_index = args_index + 2;        // Save 2 slots for the closure.
  }
  else
    args = new VarNode*[num_args];

  //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX 
  int  cell_arg_handle = 0;
  //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX 

  for (i = 0; i < nformals; i++) {
    formal f = forms[i];
    th_assert(f->hdr.methods == &formal_methods, 
	      "Not a formal object - perhaps a surrogate ?");
		        
    if (is_basic_type(f->t)) {
      // Basic value arguments need not be type checked!
      char ch;
      int val;

      control_readValue(c, ch, val);

      if (ch == 'h') {        // val is the future
	find_type(val, r);
	args[args_index] = new FutureNode(r);   
      }
      else {
	args[args_index] = new ValNode((any)val);  
      }
    }
    else if (f->t == (type) Any) {
      //XXX Not fully implemented yet!!!
      //Need to keep the type if basic value is sent so make_any can
      //be called each time.  Specialize FutureNode/ValNode (sigh).

      // Arguments need not be type checked since everything is a
      // valid subtype of Any.  However, they do need to be
      // dynamically converted to any\'s using the make_any call.

      //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX 
      // Another gross hack regarding cells (types/cell.h)!!! --Qyz
      if (m->self_type == (type)cellC) {
	int val;
	char ch;

	control_readValue(c, ch, val);

	if (ch == 'h') {
	  if (val < 0) {
	    find_type(val, r);         // val is the future.
	    RESIGNAL_ANY_EXC;
	    args[args_index] = new FutureNode(r);
	    cell_arg_handle = val;
	  }
	  else if (val > 0) {
	    handle_val res = FEConf->ht->handle_to_any(val, TRUE);        // val is the handle
	    args[args_index] = new ValNode((any)res.val);  
	    cell_arg_handle = val;
	  }
	  else {
	    // Shouldn\'t happen!!!
	    th_fail("handle == 0!");
	  }
	}
	else                                        // val is basic value
	  args[args_index] = new ValNode((any)val);
      }
      //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

    } else {                           
      handle_val res;
      int handle = connection_fgeti(c);

      if (handle > 0) { // This is not a future, so we can find it\'s type
	handle_val res = FEConf->ht->handle_to_any(handle, TRUE);
	if (exc || res.tag != Anyf)
	  // The handle was invalid or it was a promise.
	  ; // Raise exception!
	any o = res.val;
	t = get_type(res);

	int hdroff = type_check(t, f->t);
	RESIGNAL_ANY_EXC;

	args[args_index] = new ValNode(o, hdroff);
      } else { 
        // This is a future, so we type check against the type entered
	// for this future when it was processed as the result of a
	// previous call.
	t = find_type(handle, r);
	int hdroff = type_check(t, f->t);
	RESIGNAL_ANY_EXC;
	if (hdroff == 0)
	  args[args_index] = new FutureNode(r);
	else
	  args[args_index] = new FutureOffsetNode(r, hdroff);
      }
    }
    args_index++;
  }

  // Read the result futures.  Futures HAVE to be specified.  This
  // function should only be called if J or K has been sent!

  int num_rets = fast_unsafe_vec_length(m->returns);
  type *ret_types = UNPV(type *, fast_unsafe_vec_items(m->returns));

  int *results = new int[num_rets];
  int resultc = 0, f;

  cell_method cm = NONE;

  //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
  // This is a hack that will go away when we have real
  //parameterized types.  See "types/cell.h" for explanation. --Qyz
  if (m->self_type == (type)cellC) {
    cell tmp = (cell)self->get_val();   // XXX This will totally break if
				  // cells are used as first class objects!!
    //fprintf(stderr, "Method = %s\n", string_charp(method_name(m)));
    if (m->index == 1) { // cell->put
      // Store the apparent type of the argument.
      if (cell_arg_handle != 0)
	cell_put_type(tmp, find_type(cell_arg_handle, r));
      else {
	// cell_arg_handle == 0 means that the argument was a basic
	// type.  Since basic type arguments will not be type checked,
	// mapping it to Int will do.
	cell_put_type(tmp, (type)Int);
      }
      cm = PUT;
    }
    else if (m->index == 2) { // cell->get
      // Read the result future.
      connection_fread(c, &f, sizeof(int));

      // Change the return type to the apparant type of the content (Groan).
      // XXX Have to reinitialize ret_types since fast_unsafe_vec just
      // returned a pointer to the vec inside the method object!

      ret_types = new type[1];            //XXX memory leak!
      ret_types[0] = cell_get_type(tmp);

      // Map f to the apparant type of the content of cell.
      enter_result(f, ret_types[0]);
      results[0] = f;
      cm = GET;
    }
    else { // Shouldn\'t happen!
      th_fail("Method index for cell was not 1, nor 2!\n");
    }
  }
  else
  //XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

    for (i=0; i < num_rets; i++) {
      connection_fread(c, &f, sizeof(int));
      // Make sure the result futures get entered in type table
      enter_result(f, ret_types[i]);
      results[i] = f;
    }

  if (method_isIter(m)) {
    add_method_node(new MethodIterNode(self, m, num_args, args, num_rets,
				       results, ret_types));
  }
  else {
    MethodNode *mn =
      create_specialized_method_node(self, m, num_args, args, num_rets,
			       results, ret_types, cm);
    add_method_node(mn);
  }
}
#endif //PROMISES
