%{
#include "boot/lex_input.h"
#include "runtime/except.h"
#include "types/str.h"
#include "fe_expr.h"
#include "common/basic.h"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

static int lexstr();
static int lexchar(char const []);
static int lexbnum(char const []);

#define text ((char *)yytext)
%}

digit [0-9]
octal [0-7]
letter [A-Za-z]
intstr (([1-9]{digit}*)|0)
integer (-?{intstr})
special [_]
S [ \t\n\r]
id ({letter}({letter}|{digit}|{special})*)
charac (\'([^\'\n\\]|{bscombos})\')
bscombos (\\\\|\\[nbtvrfa]|\\\"|\\\'|\\{octal}{octal}{octal})
hexnum ({intstr}_{hexstuff})
hexdigit [0-9a-fA-F]
hexstuff ({hexdigit}{hexdigit}*)
real (-?{intstr}\.({digit}*)({exp}?))
exp ([EeDdXx][+-]{intstr})
handle (#{intstr})
future (#-{intstr})
dot (\.)

%%

{S}+            ;
"true"          { yylval.b = TRUE; return BoolTok; }
"false"         { yylval.b = FALSE; return BoolTok; }
"nil"           { return NilTok; }
{id}		{ yylval.s = string_new(text); return IdTok; }
"\""	        { return lexstr(); }
{charac}	{ return lexchar(text); }
{integer}       { yylval.i = atoi(text); return IntegerTok; }
{hexnum}        { return lexbnum(text); }
{real}          { yylval.r = (real)atof(text); return RealTok; }
{handle}        { yylval.i = atoi(text + 1); return HandleTok; }
{future}        { yylval.i = atoi(text + 1); return FutureTok; }
{dot}		{ return DotTok; }
.		{ return LexErr; }

%%

#include "types/textwr.h"

static char const octals[] = "01234567";

static char parseChar(char const txt[])
{
    if (txt[0] != '\\') return txt[0];
    switch(txt[1]) {
      case 'a': return '\a';
      case 'b': return '\b';
      case 'f': return '\f';
      case 'n': return '\n';
      case 'r': return '\r';
      case 't': return '\t';
      case 'v': return '\v';
      case '\'': return '\'';
      case '\"': return '\"';
      case '\\': return '\\';
      default:
	if(!strchr(octals, txt[1]) ||
	   !strchr(octals, txt[2]) ||
	   !strchr(octals, txt[3])) {
	    exc = &exc_not_possible;
	    return 0;
	}
	return ((txt[1] - '0')<<6) |
	  ((txt[2] - '0')<<3) |
	    (txt[3] - '0');
    }
}

static int lexstr()
{
    textwr tw = textwr_new();
    wr w = textwr_as_wr(tw);
    for (;;) {
	char buf[5];
	char *b = buf;
	char c = *b++ = input();
	switch (c) {
	    case 0: return LexErr; /* end of file */
	    case '\\':
	      c = *b++ = input();
	      if (strchr(octals, c)) {
		    *b++ = input();
		    *b++ = input();
		}
	      c = parseChar(buf);
	      CATCH {
		   exc = EXC_NONE;
		   return LexErr;
	      }
	      wr_putChar(w, c);
	      break;
	    case '"':
	      goto done;
	    case '\n':
	      yylineno++;
	      wr_putChar(w, c);
	      break;
	    default:
	      wr_putChar(w, c);
	      break;
	}
    }
  done: wr_close(w);
    yylval.s = textwr_toString(tw);
    return StringTok;
}

static int lexchar(char const txt[])
{
    assert(txt[0] == '\'');
    yylval.c = parseChar(txt + 1);
    CATCH {
	exc = EXC_NONE;
	return LexErr;
    } else {
	return CharTok;
    }
}

char const digits[] = "0123456789abcdef";

static int lexbnum(const char txt[])
{
    int bsize = strcspn(txt, "_");
    char basechars[100];
    int base;
    int num = 0;
    char const *c;
    strncpy(basechars, txt, bsize);
    base = atoi(basechars);
    for (c = txt + bsize + 1; *c; c++) {
	char digitch = tolower(*c);
	int digitval = (char const *)strchr(digits, digitch) - digits;
	if (digitval < 0 || digitval >= base) return LexErr;
	num *= base;
	num += digitval;
    }
    yylval.i = num;
    return IntegerTok;
}

int yywrap()
{
    return TRUE;
}
