// Copyright 1995 Barbara Liskov


#include "handle_gc.h"
#include "runtime/except.h"
#include "runtime/alloc.h"
#include "runtime/obj_class.h"
#include "types/type.h"
#include "types/class.h"
#include "types/any.h"

#include "cache/cache.h"
#include "common/hash.h"
#include "common/th_assert.h"
#include <string.h>

#include "fe_config.h"
#include "handle.h"
#include "future.h"
#include "config/vdefs/DEBUG_HANDLES.h"
#include "cache/indirect_table.h"

#if DEBUG_HANDLES
#include <stdio.h>
#endif

implementArray(FutureToAny, any)
implementArray(FutureToTag, char)


future_table::future_table() {
  ftoa_table =  FutureToAny(1000);
  ftoa_tags = FutureToTag(1000);

  ftoa_table.append((any)0);      // Eat up future 0
  ftoa_tags.append(Emptyf);

  num_futures = 0;
}


inline bool future_table::lookup(future p, char &tag, any &a) {
  int index = -p;

  if ((index <= 0) || (index >= ftoa_table.size()))
    return FALSE;

  tag = ftoa_tags[index];
  if (tag == Emptyf)
    return FALSE;

  a = ACTUAL_ANY_POINTER(ftoa_table[index]);
  return TRUE;
}


inline void future_table::free_all_futures() {
    for (int i=1; i < ftoa_table.size(); i++) {
	free_future(-i);
    }
    // num_futures = 0;  We should not need to set this to 0
    th_assert(num_futures == 0, "Number of futures not 0 after freeing them");
}

void future_table::get_future_handles(IntArray& futures, IntArray& handles) {

  // Note that this function does not modify the future table till
  // all the handle information has been obtained (an abort cannot take
  // place after that)

  int count = 1; // Have to start writing from first future

  // For each future, find the corresponding handle or value (for basic types)
  for (int i = 1; i < ftoa_table.size(); i++) {
    if (ftoa_tags[i] != Emptyf) {
      handle h;

      if (ftoa_tags[i] == Anyf)
	h = FEConf->ht->any_to_handle(ACTUAL_ANY_POINTER(ftoa_table[i]));
      else
	h = (long)ftoa_table[i];
#if DEBUG_HANDLES
      int f = -i; // Futures are tagged with a negative sign to
      // distinguish them from handles 
      fprintf(stderr, "Future = %d, Handle = %d\n", f, h);
#endif
      th_assert(count < num_futures+1, "Too many tags in ftoa_tags are non-empty");
      futures[count] = -i;
      handles[count++] = h;
    }
  }
  // Now this call will succeed, (an abort cannot cause problems)
  // I.e. if the FE frees the futures now, this remapping information
  // will be sent to the client definitely and not be lost due to an abort
  free_all_futures();
}

handle_val future_table::future_to_any(future p) {
  handle_val result;

  if (!lookup(p, result.tag, result.val))
    exc = &exc_not_possible;
  else {
    if (result.tag == Anyf) {
      FIX(result.val, any);
    }

#if DEBUG_HANDLES
    fprintf(stderr, "looked up future %d, found object %d\n", p, result.val);
#endif

  }
  return result;
}

inline bool future_table::store(future p, char t, any a) {
  int index = -p;
  int size = ftoa_table.size();
  
  th_assert(size == ftoa_tags.size(), "Future table invariant failed");
  
  if (index >= size) {
    any dummy;
    
    ftoa_table.append(dummy, index - size + 1);
    ftoa_tags.append(Emptyf, index - size + 1);
  }

  th_assert(ftoa_tags[index] == Emptyf, "Assignment to existing future!");

  ftoa_table[index] = obj_as_any(get_storable_pointer(any_as_obj(a)));
  ftoa_tags[index] = t;

  num_futures++;

  return TRUE;
}

void future_table::any_to_future(any a, future p) {
  th_assert(p < 0, "Invalid future passed to atof");
  
  store(p, Anyf, a);
}

void future_table::basic_to_future(type bt, fevalue v, future p) {
  th_assert(p < 0, "Invalid future passed to btof");

  // Look at type kind first!
  // Actually, should only need to store Any or Basic!
  if (bt == (type)Bool) {
    store(p, Boolf, (any)v.b);
    return;
  }
  if (bt == (type)Int) {
    store(p, Intf, (any)v.i);
    return;
  }
  if (bt == (type)Char) {
    store(p, Charf, (any)v.c);
    return;
  }
  if (bt == (type)Real) {
    store(p, Realf, (any)(long)v.r);
    return;
  }
  store(p, Anyf, obj_as_any(v.o));
  return;
}

FutureIter::FutureIter() {
    index = 1;
    for (index = 1;index < FEConf->ft->ftoa_table.size(); index++) {
	if (FEConf->ft->ftoa_tags[index] == Anyf) return;
    }
}

bool FutureIter::ok() {
    return (index < FEConf->ft->ftoa_table.size());
}

future FutureIter::get_future() {
    return (future) -index;
}

obj FutureIter::get_obj() {
    return any_as_obj(FEConf->ft->ftoa_table[index]);
}

void FutureIter::next() {
    index++;
    for (; index < FEConf->ft->ftoa_table.size(); index++) {
	if (FEConf->ft->ftoa_tags[index] == Anyf) return;
    }
}

void FutureIter::replace(future f, obj val) {
  if (-f == index)
    FEConf->ft->ftoa_table[index] = obj_as_any(val);
  else
    th_fail("Tried to mutate the future table outside the correct iteration");
}
