// Copyright 1995 Barbara Liskov

// Generate the veneer include and stub files for a given type
// using the binary interface to the Thor FE
#ifdef __cplusplus
extern "C" {
#endif

#include "types/ptype_class.h"    
#include "types/param.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "types/objtype_class.h"
#include "types/string_class.h"
#include "types/wr.h"
#include "types/textwr.h"
#include "types/param_class.h"
#include "types/method.h"
#include "types/instn.h"
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "common/openhashset.h"
#include "config/vdefs/PROMISES.h"

#include "types/class_instn.h"
#include "types/dict.h"
    
void  make_Cxx_veneer_files(objtype t);
void note_instantiated_type(type t);
#ifdef __cplusplus
}
#endif

// Dictionary for fully instantiated types.
extern dict inst_typeEnv;

// Enumeration used for different return types of a generated method
typedef enum {
    RETURN_VOID, RETURN_BASIC, RETURN_HANDLE
} Return_Type;


static void make_impl    (string filename, string contents);
static void make_veneer_include (string filename, string contents);
static string gen_Cxx_veneer_file(objtype t, bool header);
static void gen_Cxx_veneer_stub (wr w, objtype t, method m, bool header,
				bool promises);
static void gen_Cxx_veneer_rtn_stub(wr w, objtype t, bool header);
static void gen_Cxx_args(wr w, method m, objtype t, bool promises);
static void gen_Cxx_veneer_args(wr w, method m, objtype t, bool promises);
static void gen_Cxx_veneer_arg(wr w, formal f, bool types);

static void typename(wr w, type t, bool& is_handle,
	             bool include_params, bool promises);

static  void method_Cname  (wr w, objtype t, method m, bool header, bool promises);
static  void rtn_name(wr w, objtype t);
static  void put_type_name(wr w, type t);
static  bool is_parameterized(method m);
static  bool is_basic_type(type t);
static	bool is_basic_type(objtype t);
static  bool is_rtn(objtype t);
static	void put_formal_name(wr w, formal f);
static	bool is_cxx_keyword(string s);
static	void put_param(wr w, ptype pt);

/* TypeSet is used to store the set of types referenced in the methods of a given type */

#define addr_hash(x)  ((int)(long)x)
#define addr_compare(x,y)  ((x)==(y))

declareOpenHashSet(TypeSet, type, addr_hash, addr_compare)
implementOpenHashSet(TypeSet, type, addr_hash, addr_compare)
    
bool is_template(objtype t)
{
    switch(type_kind(objtype_as_type(t))) {
	case PTYPE_KIND:
	case PCLASS_KIND:
	    return TRUE;
	default:
	    return FALSE;
    }
}

bool is_fully_instantiated_type(type t)
{
  return ((type_kind(t) == CLASS_INSTN_KIND) &&
           (type_kind(UNPV(type, vec_fetch(
			           class_instn_get_pargs(
				     type_as_class_instn(t)),
			             0))) != PARAM_KIND));
}

bool test_for_basic_types(method m)
// Returns true iff any of the arguments or return values of m are a
// basic type.
{
  int i;

  int num_returns = vec_length(m->returns);
  type *returns = UNPV(type *, vec_items(m->returns));
  for (i=0; i<num_returns; i++)
    if (is_basic_type(returns[i]))
      return TRUE;

  int num_args = vec_length(m->arguments);
  formal *args = UNPV(formal *, vec_items(m->arguments));
  for (i=0; i<num_args; i++)
    if (is_basic_type(args[i]->t))
      return TRUE;

  return FALSE;
}

void gen_return_if_exception(wr w, Return_Type ret_type, string ret_type_str) {
    // effects: Generate code for:
    //   If an unhandled exception exists, then the method/routine call should
    //   return immediately with a relevant "null" value.
    //   ret_type indicates what the return type of the method/routine is
    //   ret_type_str is the name of the type (without the *) if ret_type is
    //   RETURN_HANDLE.

    switch (ret_type) {
	case RETURN_VOID:
	    wr_putChars(w, "   RETURN_VOID_IF_EXC();\n");
	    break;
	case RETURN_BASIC:
	    wr_putChars(w, "   RETURN_ZERO_INT_IF_EXC();\n");
	    break;
	case RETURN_HANDLE:
	    th_assert(string_length(ret_type_str) > 0, "Null type name");
	    wr_putChars(w, "   RETURN_INVALID_HANDLE_IF_EXC(");
	    wr_putString(w, ret_type_str);
	    wr_putChars(w, ");\n");
	    break;
    }
}



void gen_Cxx_veneer_invoke_rest(wr w, method m, bool promises, bool allow_deferred_invoke)
{
    wr_putChars(w, "method_handle");
    wr_putChars(w, allow_deferred_invoke ? ", TRUE" : ", FALSE");
    wr_putChars(w, promises ? ", TRUE);\n" : ", FALSE);\n");
}

// Generate the C veneer stub for a method, or just the header if
// "header" is true.  If promises is true, generate stubs that use 
// promises instead of basic values.
    
void gen_Cxx_veneer_stub(wr w, objtype t, method m, bool header, bool promises)
{
    int len, i;
    type ret;
    int num_rets = vec_length(m->returns);
    int sig_len=vec_length(m->signals);
    bool is_handle;
    Return_Type ret_type;
    string ret_type_str;
    
#if !PROMISES
    if (m->iter)		// iterators not supported 
	return;
#else
    //XXX Parameterized iterators not supported?? (bug in header file
    //stuff).
    if (m->iter && is_parameterized(m)) 
      return;
#endif

    if (!header && is_template(t)) {
      wr_putChars(w, "template ");
      put_param(w, type_as_ptype(t)); 
      UNHANDLED_EXC;
      wr_putChars(w, " ");
    }

// Output the function type
    if (header) 
	wr_putChars(w, "   ");	// indent declarations
    
    int typ_len;
    switch (num_rets) {
      default:
	wr_putChars(w, "void ");
	ret_type = RETURN_VOID;
	break;
      case 1:
	ret = UNPV(type, vec_fetch(m->returns, 0));
	if (header &&
	    (is_fully_instantiated_type(ret))) {
	  note_instantiated_type(ret);
	}
	{
	    textwr tw = textwr_new();
	    typename(textwr_as_wr(tw),
		ret, is_handle, TRUE, promises);
	    ret_type_str = textwr_toString(tw);
	    wr_putString(w, ret_type_str);
	}
	if (is_handle) {
	    ret_type = RETURN_HANDLE;
	} else
	      ret_type = RETURN_BASIC;
	wr_putChar(w, ' ');
    }
    
    
// Output the function name and arguments
    method_Cname(w, t, m, header, promises);
    wr_putChar(w, '(');
    gen_Cxx_veneer_args(w, m, t, promises);
    wr_putChars(w, ")");
    
    if (header)  wr_putChars(w, ";");
    
// Output a comment describing the exceptions signalled
    if (sig_len > 0) {
	wr_putChars(w, "\n	/* signals(");
	for (i=0; i < sig_len ; i++) {
	    signal_ s = UNPV(signal_, vec_fetch(m->signals, i));
	    if (i>0) wr_putChars(w, ", ");
	    wr_putString(w, s->name);    
	}
	wr_putChars(w, ") */");
    }

    if (header) {
	wr_putChar(w, '\n');
	return;
    }
    
    // Continue generating the code for the .cc file
    wr_putChars(w, "\n{\n");
    
    gen_return_if_exception(w, ret_type, ret_type_str);

    if (num_rets == 1) {
	wr_putChars(w, "   ");
	typename(w, ret, is_handle, TRUE, promises);
	wr_putChars(w, " res;\n");
    } else {
	type *returns = UNPV(type *, vec_items(m->returns));
	for (i = 0; i < num_rets; i++) {
	    wr_putChars(w, "   res");  wr_putChar(w, '1'+i); 
	    wr_putChars(w, " = CAST_TYPE(");
	    typename(w, returns[i], is_handle, TRUE, promises);
	    wr_putChars(w, ", th_config->invalid_obj);\n");
	}
    }


    // Output code to ensure that the method handle has been loaded,
    // which is mapped in the database to a corresponding method object.
    // The code to memoize the method handle includes two pieces of
    // information: the name of the apparent type for the method, and
    // the method index in the "methods" vector of the type object.

    // XXX Note that this assumes there is a single globally unique name
    // for each type, otherwise the FE will not be able to determine the
    // the type to use for looking up the method object.

    wr_putChars(w,
		"   static int method_handle = 0;\n"
		"   if (!method_handle) memoize_method_handle(");

    if (is_template(t)) {
      // put call of Class_Name
      wr_putChars(w, "Class_Name(), ");
    } else {
      // put type name
      wr_putChars(w, "\"");
      wr_putString(w, type_name((type)t));
      wr_putChars(w, "\", ");
    };
    // put the method's index in the "methods" vector of the Type object
    int starting_index = method_index(UNPV(method, vec_fetch(t->methods_, 0)));
    char mindex[16];
    sprintf(mindex, "%d", method_index(m) - starting_index);
    wr_putChars(w, mindex);
    wr_putChars(w, ", &method_handle);\n");

    gen_return_if_exception(w, ret_type, ret_type_str);

    // Find out whether the invoke can be deferred
    bool allow_deferred_invoke = TRUE;
    if (!promises)
      for (len=vec_length(m->returns), i=0; i < len ; i++) {
	if (is_basic_type(UNPV(type, vec_fetch(m->returns, i)))) {
	  allow_deferred_invoke = FALSE;
	  break;
	}
      }

// Special case for basic types

#if PROMISES
    if (is_basic_type(t)) {
      wr_putChars(w, "   if (handle == 0) \n");
      wr_putChars(w, "      begin_invoke(basic_value(val), ");
      gen_Cxx_veneer_invoke_rest(w, m, promises, allow_deferred_invoke);
      wr_putChars(w, "   else\n");
      wr_putChars(w, "      begin_invoke(Vtab->get_handle(__index), ");
      gen_Cxx_veneer_invoke_rest(w, m, promises, allow_deferred_invoke);
    }
    else 
#endif
    {

      // Need to calculate the size of the message that is being sent
      int no_args = vec_length(m->arguments);
      char tmpstr[500];
      tmpstr[0] = 0;
      int num_chars = 0;
      int num_bools = 0;
      int num_handles = 0;
      int num_nulls = 0;
      int num_reals = 0;
      int num_ints = 0;
      // Find out the number of arguments of each type (in terms of size)
      if (no_args) {
          for (i = 0; i < no_args; i++) {
	      formal f = UNPV(formal, vec_fetch(m->arguments, i));
	      if (f->t == (type)Int)
		  num_ints++;
	      else if (f->t == (type)Char)
		  num_chars++;
	      else if (f->t == (type)Bool)
		  num_bools++;
	      else if (f->t == (type)Real)
		  num_reals++;
	      else if (f->t == (type)Null)
		  num_nulls++;
	      else
		  num_handles++;
	  }
	  
#define OUTPUT_SIZEOF_ARG(str, typename, num)\
	  if (num) sprintf(str + strlen(str), "+%d*sizeof(int)", num)
        // if (num) sprintf(str + strlen(str), "+%d*sizeof(" typename ")", num)
	  OUTPUT_SIZEOF_ARG(tmpstr, "int", num_ints);
	  OUTPUT_SIZEOF_ARG(tmpstr, "bool", num_bools);
	  OUTPUT_SIZEOF_ARG(tmpstr, "char", num_chars);
	  OUTPUT_SIZEOF_ARG(tmpstr, "real", num_reals);
	  OUTPUT_SIZEOF_ARG(tmpstr, "null", num_nulls);
	  OUTPUT_SIZEOF_ARG(tmpstr, "int", num_handles);

	  wr_putChars(w, "   INCR_BY_SIZE(0");
	  wr_putChars(w, tmpstr);
	  wr_putChars(w, "); // Message size of the arguments\n");
	  
#if PROMISES
	  // For the tags Needs to be fixed XXXX
	  wr_putChars(w, "   INCR_BY_SIZE(");
	  wr_putChars(w, tmpstr);
	  wr_putChars(w, "); // Message size of the tags\n");
#endif
      }

      // If this is a deferred invoke then we need to calculate the size
      // of the futures that follow it also (for the non-basic return types)

      if (allow_deferred_invoke) {
	  int no_returns = vec_length(m->returns);
	  int num_non_basic = 0;
	  for (i = 0; i < no_returns; i++) {
	      type ret = UNPV(type, vec_fetch(m->returns, i));
	      if (is_basic_type(ret) == 0)
		  num_non_basic++;
	  }
	  
	  if (num_non_basic) {
	      sprintf(tmpstr, "%d*sizeof(int)", num_non_basic);
	      wr_putChars(w, "   INCR_BY_SIZE_IF_FUTURES(");
	      wr_putChars(w, tmpstr);
	      wr_putChars(w, "); // Message size of non-basic return futures\n");
	  }
      }

// Output the invoke call
      wr_putChars(w, "   begin_invoke(");
    
// If a type has multiple supertypes, C++ requires that we disambiguate the
// reference to "handle".
    
//  XXX The fix here does not work in general-- it fails if a type
//  with multiple supertypes has subtypes.  However, it works with the
//  currently existing Thor type hierarchy.
    
      wr_putChars(w, "Vtab->get_handle(");

      if (vec_length(t->supertypes_) > 1) {
	typename(w, UNPV(type, vec_fetch(t->supertypes_, 0)),
			    is_handle, TRUE, promises);
	wr_putChars(w, "::");
      }

      wr_putChars(w, "__index), ");
 
      gen_Cxx_veneer_invoke_rest(w, m, promises, allow_deferred_invoke);    
    }

// Send the arguments    
    for (len=vec_length(m->arguments), i=0; i < len ; i++) {
	formal f = UNPV(formal, vec_fetch(m->arguments, i));
	if (promises && is_basic_type(f->t)) {
	    wr_putChars(w, "   put_val_or_handle(");
	    put_formal_name(w, f);
	    wr_putChars(w, ");\n");
	}
	else {
	    wr_putChars(w, "   put_");
	    put_type_name(w, f->t);
//	    if (f->t == (type) Any)
//              wr_putChars(w, "_tagged");
	    wr_putChars(w, "(");
	    if (!is_basic_type(f->t))
		wr_putChars(w, "Vtab->get_handle(");
	    put_formal_name(w, f);
	    if (!is_basic_type(f->t))
		wr_putChars(w, ".__index)");
	    wr_putChars(w, ");\n");
	}
    }

    wr_putChars(w, "   if (do_invoke(");
    wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
    wr_putChars(w, ")) {\n");
    
    
// Get the results
    for (len=vec_length(m->returns), i=0; i < len ; i++) {
	type ret = UNPV(type, vec_fetch(m->returns, i));
	
	if (is_basic_type(ret) && !promises) {
	    wr_putChars(w, "      get_");  
	    put_type_name(w, ret);
	    wr_putChar(w, '(');
	    wr_putChar(w, '&');
	    wr_putChars(w, "res");  
	    if (len > 1) {wr_putChar(w, '1'+i); }
	    wr_putChars(w, ");\n");
	} else {

	    // XXX Workaround for preprocessor bug 

	    // C++ parameterized type instantations with multiple
	    // parameters is broken up into different macro arguments
	    // because the commas inside the angle braces are not
	    // handled correctly by the preoprocessor (e.g. "map<K,V)"
	    // is split into "map<K," and "V>")

	    // We work around the problem by #defining "result_type"
	    // to be appropriate parameterized type instantiation

	    bool hack_cpp_bug =  0 != strchr(type_name((type)ret)->chars, ',');
	    
	    if (hack_cpp_bug) {
		wr_putChars(w, "#undef result_type \n");
		wr_putChars(w, "#define result_type ");
		typename(w, ret, is_handle, TRUE, promises);
		wr_putChars(w, " // Workaround for preprocessor bug \n");
	    }

	    if (is_basic_type(ret))
	      wr_putChars(w, "      get_result_promise(&res);\n");
	    else {
	      wr_putChars(w,  "      NEW_TH_OBJ(get_handle(), ");

	      if (hack_cpp_bug) {
		wr_putChars(w, "result_type");
	      }
	      else {
		    typename(w, ret, is_handle, TRUE, promises);
	      }
	      wr_putChars(w, ", ");
	      wr_putChars(w, "res");  
	      if (len > 1) {wr_putChar(w, '1'+i); }
	      wr_putChars(w, ");\n");
	    }
	}
    }

    wr_putChars(w, "   }\n");
    wr_putChars(w, "   end_invoke(");
    wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
    wr_putChars(w, ");\n");
    if (vec_length(m->returns)==1)
	wr_putChars(w, "   return res;\n");
    wr_putChars(w,"}\n\n");
}

void gen_Cxx_veneer_rtn_stub(wr w, objtype t, bool header)
{
    int len, i;
    type ret;
    method m = UNPV(method, vec_fetch(t->methods_, 0));
    int num_rets = vec_length(m->returns);
    int sig_len = vec_length(m->signals);
    bool promises = FALSE;
    bool allow_deferred_invoke = FALSE;
    bool is_handle;
    string ret_type_str;
    Return_Type ret_type;

#if !PROMISES
    if (m->iter)		// iterators not supported 
	return;
#else
    //XXX Parameretized iterators not supported?? (bug in header file
    //stuff).
    if (m->iter && is_parameterized(m)) 
      return;
#endif

    if (!header && is_template(t)) {
      wr_putChars(w, "template ");
      put_param(w, type_as_ptype(t)); 
      wr_putChars(w, " ");
    }

// Output the routine type
//    if (header) 
//	wr_putChars(w, "   ");	// indent declarations
    
    int typ_len;
    switch (num_rets) {
      default:
	wr_putChars(w, "void ");
	ret_type = RETURN_VOID;
	break;
      case 1:
	ret = UNPV(type, vec_fetch(m->returns, 0));
	if (header &&
	    (is_fully_instantiated_type(ret))) {
	  note_instantiated_type(ret);
	}
	{
	    textwr tw = textwr_new();
	    typename(textwr_as_wr(tw),
				ret, is_handle, TRUE, promises);
	    ret_type_str = textwr_toString(tw);
	    wr_putString(w, ret_type_str);
	}
	if (is_handle) {
	    ret_type = RETURN_HANDLE;
	} else
	    ret_type = RETURN_BASIC;
	wr_putChar(w, ' ');
    }
    
    
// Output the routine name and arguments
    wr_putChars(w, "th_");
    rtn_name(w, t);
    wr_putChar(w, '(');
    gen_Cxx_veneer_args(w, m, t, promises);
    wr_putChars(w, ")");
    
    if (header)  wr_putChars(w, ";");
    
// Output a comment describing the exceptions signalled
    if (sig_len > 0) {
	wr_putChars(w, "\n	/* signals(");
	for (i=0; i < sig_len ; i++) {
	    signal_ s = UNPV(signal_, vec_fetch(m->signals, i));
	    if (i>0) wr_putChars(w, ", ");
	    wr_putString(w, s->name);    
	}
	wr_putChars(w, ") */");
    }
    
    if (header) {wr_putChar(w, '\n'); return;}
    
    wr_putChars(w, "\n{\n");
    
    gen_return_if_exception(w, ret_type, ret_type_str);
    if (ret_type == RETURN_HANDLE) free(ret_type_str);

    if (num_rets==1) {
	wr_putChars(w, "   ");
	typename(w, ret, is_handle, TRUE, promises);
	wr_putChars(w, " res;\n");
    } else {
	for (i=0; i < num_rets; i++) {
	    if (i>0) wr_putChars(w, ", ");
	    wr_putChars(w, "res");  wr_putChar(w, '1'+i); 
	    wr_putChars(w, " = th_config->invalid_obj");
	}
	if (i>0) wr_putChars(w, ";\n");
    }

    // Output boilerplate decls here...
    wr_putChars(w, "   ");	// indent declarations
    wr_putChars(w, "static th_");
    wr_putString(w, t->name);
    wr_putChars(w, " xtra_long_name = CAST_TYPE(th_");
    wr_putString(w, t->name);
    wr_putChars(w, ", th_config->invalid_obj);\n");

    // Output lookup
    wr_putChars(w, "   ");	// indent declarations
    wr_putChars(w, "if (th_is_invalid(xtra_long_name)) {\n");
    wr_putChars(w, "      ");	// indent body
    wr_putChars(w, "th_any class_res = memoize_routine_class(\"");
    wr_putString(w, t->name);
    wr_putChars(w, "\");\n      ");
    wr_putChars(w, "xtra_long_name = CAST_TYPE(th_");
    wr_putString(w, t->name);
    wr_putChars(w, ", class_res);\n   }\n");


    // Do invoke
    wr_putChars(w, "   ");	// indent declarations
    if (num_rets) wr_putChars(w, "res = ");
    wr_putChars(w, "xtra_long_name.invoke(");

    // Add the arguments    
    gen_Cxx_args(w, m, t, promises);
    // Add pointers to the results    (Not yet)

    wr_putChars(w, ");\n");

    // If xtra_long_name is invalid, set it to 0
    wr_putChars(w, "   if (th_is_invalid(xtra_long_name))\n");
    wr_putChars(w, "       xtra_long_name = CAST_TYPE(th_");
    wr_putString(w, t->name);
    wr_putChars(w, ", th_config->invalid_obj);\n");

    // Return results
    
    if (vec_length(m->returns)==1)
	wr_putChars(w, "   return res;\n");
    wr_putChars(w,"}\n\n");
}

void make_Cxx_veneer_files(objtype t)
{
    UNHANDLED_EXC;
    if (is_basic_type(t))
      return;		 // PRJ: don't generate veneers files for basic types
    bool is_templ = is_template(t);
    string contents = gen_Cxx_veneer_file(t, FALSE);
    string contents2 = gen_Cxx_veneer_file(t, TRUE);

    string vname = string_concat(string_new("th_"), t->name);
    if (is_templ)
    	make_impl(string_concat(vname, string_new(".t")),
	      contents);
    else
    	make_impl(string_concat(vname, string_new(".cc")),
	      contents);
    
    make_veneer_include(string_concat(vname, string_new(".h")),
			contents2);

    FILE *typelist;
    if (typelist= fopen("_th-typelist.h", "a")) {
	if (is_template(t)) {
	    ptype pt = type_as_ptype(t);
	    UNHANDLED_EXC;
	    int num_params = vec_length(pt->params);
	    param *parm = UNPV(param *, vec_items(pt->params));

	    fprintf(typelist, "template <");
	    for (int i=0; i<num_params; i++) {
		if (i) fprintf(typelist, ", ");
		fprintf(typelist, "class %s", string_charp(parm[i]->name));
	    }
	    fprintf(typelist, "> ");
	}
	fprintf(typelist, "class th_%s;\n", string_charp(t->name));
	fclose(typelist);
    }
}

static void make_impl(string filename, string contents)
{
    char const *fn = string_charp(filename);

    FILE *f = fopen(string_charp(filename), "w");
    fputs(string_charp(contents), f);
    fclose(f);
}

void gen_Cxx_veneer_arg(wr w, formal f, bool promises, bool types)
{
/*
    if (f->t == class_as_type(Null)) return;
*/
    bool is_handle;
    UNHANDLED_EXC;
    if (types) {
	typename(w, f->t, is_handle, TRUE, promises);
	wr_putChar(w, ' ');
    }
    put_formal_name(w, f);
}

void gen_Cxx_veneer_method_args(wr w, method m, bool promises, bool types)
{
    int num_args = vec_length(m->arguments);
    formal *args = UNPV(formal *, vec_items(m->arguments));
    int i;
    // Generate code for the arguments
    for (i=0; i<num_args; i++) {
      if (i>0) wr_putChars(w, ", ");
	  UNHANDLED_EXC;
      gen_Cxx_veneer_arg(w, args[i], promises, types);
    }
    if (vec_length(m->extra_args)) {
      wr_putChars(w, ", ...");
    }
    
    // If number of return values is > 1, generate code for them in the
    // header
    int num_rets = vec_length(m->returns);
    type *returns = UNPV(type *, vec_items(m->returns));
    
    if (num_rets > 1)
      for (i=0; i<num_rets; i++) {
	if (!((num_args == 0) && (i == 0)))
	  wr_putChars(w, ", ");
	bool is_handle;
	typename(w, returns[i], is_handle, TRUE, promises);
	wr_putChars(w, "& res");
	wr_putChar(w, i+'1');
    }
}

void gen_Cxx_args(wr w, method m, objtype t, bool promises)
{
  gen_Cxx_veneer_method_args(w, m, promises, FALSE);
}

void gen_Cxx_veneer_args(wr w, method m, objtype t, bool promises)
// Write out the C arguments to this method
{
  gen_Cxx_veneer_method_args(w, m, promises, TRUE);
}

static void put_param_name_only(wr w, ptype pt)
{
    int num_params = vec_length(pt->params);
    param *parm = UNPV(param *, vec_items(pt->params));

    wr_putChar(w, '<');
    for (int i=0; i<num_params; i++) {
	bool is_handle;
	if (i) wr_putChars(w, ", ");
	typename(w, param_as_type(parm[i]), is_handle, FALSE, FALSE);
    }
    wr_putChars(w, " >");
}

static void put_param(wr w, ptype pt)
{
    int num_params = vec_length(pt->params);
    bool is_handle;
    param *parm = UNPV(param *, vec_items(pt->params));

    wr_putChar(w, '<');
    for (int i=0; i<num_params; i++) {
	if (i) wr_putChars(w, ", ");
	wr_putChars(w, "class ");
	typename(w, param_as_type(parm[i]), is_handle, FALSE, FALSE);
    }
    wr_putChars(w, " >");
}


string gen_Cxx_veneer_file(objtype t, bool header)
{
    int i;
    bool is_handle;

    textwr tw = textwr_new();
    wr w = textwr_as_wr(tw);
    bool is_templ = is_template(t);

    method *m = UNPV(method *, vec_items(t->methods_));
    int num_methods = vec_length(t->methods_);
    
    if (header) 		// include necessary include files
	wr_putChars(w, "#include \"binary_veneer.h\"\n");
	
    TypeSet types_used;
    // keep a set of types mentioned in the methods of type t
    
    if (header) {
	int len = vec_length(t->supertypes_);
	for (i=0; i < len ; i++) {
	    type st = UNPV(type, vec_fetch(t->supertypes_, i));
	    types_used.insert(st);
	}
    } else  {
	types_used.insert(objtype_as_type(t));
	objtype class_type = class_as_objtype(Class);
	types_used.insert(objtype_as_type(class_type));
	for (i = 0; i < num_methods; i++) {
	    formal *args = UNPV(formal *, vec_items(m[i]->arguments));;
	    type   *ret = UNPV(type *, vec_items(m[i]->returns));;
	
	    int num_args = vec_length(m[i]->arguments),
	    num_rets = vec_length(m[i]->returns);	    

	    if (!is_parameterized(m[i])) {
		for (int j = 0; j < num_args; j++)
		    types_used.insert(args[j]->t);
		for (j = 0; j < num_rets; j++)
		    types_used.insert(ret[j]);
	    }
	}
    }

    for (TypeSet::Elements each(&types_used); each.ok(); each.next()) {
	type included_t = each.get();
		
	if (is_basic_type(included_t)) 
	  continue;		// Don't include basic type header files.
	wr_putChars(w, "#include \"");
	typename(w, included_t, is_handle, FALSE,
			        FALSE);
	wr_putChars(w, ".h\"\n");
    }

    wr_putChar(w, '\n');
    
    if (header) {
	if (is_templ) {
	  wr_putChars(w, "template ");
	  put_param(w, type_as_ptype(t));
	  wr_putChars(w, " ");
	}
	wr_putChars(w, "class th_");
	    wr_putString(w, t->name);
	
	// Specify the superclasses
	for (i=0; i < vec_length(t->supertypes_); i++) {
	    wr_putChars(w, i==0 ? " : public " : ", public ");
	    bool is_handle;
	    typename(w,
		UNPV(type,vec_fetch(t->supertypes_, i)), is_handle, TRUE, FALSE);
	}

	// Any is a subclass of Th_Referencce
	if (t == Any) {
	    wr_putChars(w, " : public Th_Reference");
	}

	wr_putChars(w, " {\n public:\n");

// Output a constructor for constructing the object from a handle
	if (t == Any) {		

#if PROMISES
	    wr_putChars(w, "   th_any () {handle = 0; promise = FALSE;}\n");
	    wr_putChars(w, "   th_any (int aHandle) {handle = aHandle; ");
	    wr_putChars(w, "promise = FALSE;}\n");
	    wr_putChars(w, "   int handle;\n");
	    // Tag the object as promise or non-promise
	    wr_putChars(w, "   bool promise;\n");
#else
	    wr_putChars(w, "   th_any () : Th_Reference(){}\n");
	    wr_putChars(w, "   th_any (Th_Reference ref_) : Th_Reference(ref_){}\n");
#endif

// Also define an equals method on th_any XXX Should this be generated???
// wr_putChars(w, "   bool equals(th_any *a) {return handle==a->handle;}\n");
	} else {
            // default constructor
	    wr_putChars(w, "   th_");
	    wr_putString(w, t->name)
		;	    wr_putChars(w, "()" );

	    for (i=0; i < vec_length(t->supertypes_); i++) {
		wr_putChars(w, i==0 ? " :  "  :  ", ");
		typename(w, UNPV(type, vec_fetch(t->supertypes_, i)),
				    is_handle, TRUE, FALSE);
		wr_putChars(w, "()");
	    }
#if PROMISES
	    if (is_basic_type(t))
		wr_putChars(w, " { promise = TRUE; }\n\n");
	    else
#endif
		wr_putChars(w, " {}\n\n");


//             // Th_Reference constructor.
// 	    wr_putChars(w, "   th_");
// 	    wr_putString(w, t->name);
// 	    wr_putChars(w, "(Th_Reference ref_");
// 	    wr_putChars(w, ")");

// 	    for (i=0; i < vec_length(t->supertypes_); i++) {
// 		wr_putChars(w, i==0 ? " :  "  :  ", ");
// 		typename(w, UNPV(type, vec_fetch(t->supertypes_, i)),
// 				    is_handle, TRUE, FALSE);
// 		wr_putChars(w, "(ref_)");
// 	    }
// 	    wr_putChars(w, " {}\n\n");

//             // Constructor from th_any. A dummy parameter has been added
// 	    // to prevent implicit conversions from th_any to this type
// 	    wr_putChars(w, "   th_");
// 	    wr_putString(w, t->name);
// 	    wr_putChars(w, "(th_any a, int dummy)");

// 	    for (i=0; i < vec_length(t->supertypes_); i++) {
// 		wr_putChars(w, i==0 ? " :  "  :  ", ");
// 		typename(w, UNPV(type, vec_fetch(t->supertypes_, i)),
// 				    is_handle, TRUE, FALSE);
// 		wr_putChars(w, "(a)");
// 	    }
// 	    wr_putChars(w, " {}\n\n");
	}
    }

    // Output a method for returning the Class/PClass object for this type
    // This is a template for what will be an instantiated type.  Thus it
    // should have a Class method that returns a th_Class.


    // Output the methods
    for (i = 0; i < num_methods; i++) {

	if (vec_length(m[i]->extra_args)) {
	    wr_putChars(w, "\n#if VENEER_SUPPORTS_VARARGS"
			"	/* currently not supported */\n");
	    gen_Cxx_veneer_stub(w, t, m[i], header, FALSE);
	    wr_putChars(w, "#endif\n\n");
	}
	else {
	    gen_Cxx_veneer_stub(w, t, m[i], header, FALSE);
	}
    }

#if PROMISES    

    wr_putChars(w, "\n// Basic value promises stuff...\n\n");

    if (is_basic_type(t) && header) {
    // Constructor for basic-value promises
      wr_putChars(w, "   th_");
      wr_putString(w, t->name);
      wr_putChars(w, "(");
      wr_putString(w, t->name);
      wr_putChars(w, " val_) : th_any() { val = val_; promise = TRUE;}\n\n");
    }

    // For promises, add a field containing the actual value of the promise.

    if (is_basic_type(t) && header) {
	wr_putChars(w, "   ");
	typename(w, objtype_as_type(t), is_handle, FALSE, FALSE);
	wr_putChars(w, " val;\n\n");
    }

    for (i = 0; i < num_methods; i++) {
      // Generate a second stub for a method using promises instead
      // of basic values.
      if (test_for_basic_types(m[i])) {
	gen_Cxx_veneer_stub(w, t, m[i], header, TRUE);
      }
    }

    // Claim method for basic value promises
    // XXX This method should just be { return val; }  The futures will
    // be remapped to basic values after each batch.
    if (is_basic_type(t)) {
      if (header)
	wr_putChars(w, "   ");
      wr_putString(w, t->name);
      if (!header) {
	wr_putChars(w, " th_");
	wr_putString(w, t->name);
	wr_putChars(w, "::claim()\n");
	wr_putChars(w, "{\n  if (handle)\n    claim_promise();\n");
	wr_putChars(w, "  return val;\n}\n\n");
      }
      else
	wr_putChars(w, " claim();\n");
    }
#endif
    
    if (header) {
  	//wr_putChars(w, " private:\n    friend "
	//    "th_any _th_force(th_any x, char const* class_str);\n");
  	wr_putChars(w, "    static th_Class Class();\n");
	if (is_template(t)) {
	  wr_putChars(w, " private:\n    static char *Class_Name();\n");
	}
	wr_putChars(w, "};\n");
	if (is_template(t)) {
	  wr_putChars(w, "#include \"");
	  typename(w, objtype_as_type(t), is_handle, FALSE,
			      FALSE);
	  wr_putChars(w, ".t\"\n");
	}
    } else {
	// Generate the code for the class method "Class"
	// XXX  We need to add procedures to make repetition less
	if (is_template(t)) {
	    wr_putChars(w, "template ");
	    put_param(w, type_as_ptype(t)); 
	    wr_putChars(w, " ");
	}
  	wr_putChars(w, "th_Class th_");
   	wr_putString(w, t->name);
	if (is_template(t)) {
	    put_param_name_only(w, type_as_ptype(t));
	}
   	wr_putChars(w, "::Class() {\n   ");
   	wr_putChars(w, "static th_any class_stub = th_config->invalid_obj;\n");
   	wr_putChars(w, "   if (th_is_invalid(class_stub)) {\n");
   	wr_putChars(w, "      class_stub = lookup_wellknown(");

	if (is_template(t)) {
	  // put call of Class_Name
	    wr_putChars(w, "Class_Name());\n   }\n");
	} else {
	  // put type name
	    wr_putChars(w, "\"");
	  wr_putString(w, t->name);
	  wr_putChars(w, "\");\n   }\n");
	};

   	wr_putChars(w, "   return CAST_TYPE(th_Class, class_stub);\n}\n");

	if (is_template(t)) {
	  // Generate the code for the class method "Class_Name"
	  // XXX  We need to add procedures to make repetition less
	  // XXX  and handle multiple parameters and actually use the
	  // XXX  parameter names in the type rather than just assuming "T".
	  wr_putChars(w, "template ");
	  put_param(w, type_as_ptype(t)); 
	  wr_putChars(w, " ");
	  wr_putChars(w, "char *th_");
	  wr_putString(w, t->name);
	  put_param_name_only(w, type_as_ptype(t));
	  wr_putChars(w, "::Class_Name() {\n");
	  wr_putChars(w, "  static char *CName = 0;\n");
	  wr_putChars(w, "  if (! CName) {\n");
	  // **** Need to get length of t->name plus "_OF_"
          wr_putChars(w, "    char *PNm = th_string_to_chars"
	                 "(_th_convert_type_name(T::Class().name()));\n");
		// XXX huh? Can't just assume "T" here
	  wr_putChars(w, "    int len = strlen(\"");
	  wr_putString(w, t->name);
	  wr_putChars(w, "_OF_\") + strlen(PNm) + 1;\n");
	  wr_putChars(w, "    char *p = new char[len];\n");
	  // **** Need to output t->name plus "_OF_"
	  wr_putChars(w, "    strcpy(p, \"");
	  wr_putString(w, t->name);
	  wr_putChars(w, "_OF_\");\n");
	  wr_putChars(w, "    strcat(p, PNm);\n");
	  wr_putChars(w, "    CName = p;\n  }\n");
	  wr_putChars(w, "  return (CName);\n}\n");
	}
      }  
  

    if (is_rtn(t)) {
	    gen_Cxx_veneer_rtn_stub(w, t, header);
    }

    return textwr_toString(tw);
}


static void put_type_name(wr w, type t)
{
  if (t == (type)Int)
    wr_putChars(w, "int");
  else if (t == (type)Char)
    wr_putChars(w, "char");
  else if (t == (type)Bool)      
    wr_putChars(w, "bool");
  else if (t == (type)Real)      
    wr_putChars(w, "real");
  else if (t == (type)Null)
    wr_putChars(w, "null");
  else
    wr_putChars(w, "handle");
}

struct cxxvtp_s {
// environment for "typename_pargs"
    instn i;
    wr w;
    bool first, include_params, promises;
};

void typename_pargs(struct cxxvtp_s *env, type t) {
// closure function used for body of loop in typename.
    if (!first) wr_putChars(env->w, ", ");
    bool dummy;
    typename(env->w, t, dummy, env->include_params, env->promises);
    env->first = FALSE;
}

void typename(wr w, type t, bool& is_handle,
		         bool include_params, bool promises) {
    // Write the veneer typename for type "t" on wr "w".
    
    // If "include_params" is true, and "t" is a parameterized type, print
    // out the parameters for "t".    
    // Sets "is_handle" to "TRUE" if the typename refers to an object
    // represented by a handle, otherwise sets it to "FALSE".

    is_handle = TRUE;
    UNHANDLED_EXC;

    switch(type_kind(t)) {
	 case PRIMITIVE_KIND:
	    if (!promises) {
		is_handle = FALSE;
		wr_putString(w, type_name(t));
		return;
	    }
	    /* fall-through */
	 case OBJECT_KIND:
	 case CLASS_KIND:
	    wr_putChars(w, "th_");
	    wr_putString(w, type_name(t));
	    return;
	 case PARAM_KIND:
	    wr_putString(w, type_name(t));
	    return;
	 case INSTN_KIND:
	 case CLASS_INSTN_KIND: {
	    instn i = type_as_instn(t);
	    UNHANDLED_EXC;
	    wr_putChars(w, "th_");
	    wr_putString(w, ptype_name(instn_ptype(i)));
	    if (include_params) {
		wr_putChar(w, '<');
		{ /* loop */
		    struct closure cl; struct cxxvtp_s env;
		    env.i = i; env.w = w; env.first = TRUE;
		    env.include_params = include_params;
		    env.promises = promises; cl.env = &env;
		    cl.f = (ifunc)typename_pargs;

		    instn_pargs(i, cl);
		}
		wr_putChars(w, " >");
	    }
	 } return;
	 case PCLASS_KIND:
	 case PTYPE_KIND:
	    wr_putChars(w, "th_");
	    ptype pt = type_as_ptype(t);
	    UNHANDLED_EXC;
	    wr_putString(w, ptype_name(pt));
	    return;
    }
}

static void method_Cname(wr w, objtype t, method m, bool header, bool promises)
{
    int l = string_length(m->name);
    bool ends_bang = string_fetch(m->name, l - 1) == '!';
    
    if (!header) {
	type t2 = objtype_as_type(t);
	wr_putChars(w, "th_");
	wr_putString(w, type_name(t2 ));
	if (!header && is_template(t)) 
	  put_param_name_only(w, type_as_ptype(t)); 
	wr_putChars(w, "::");
    }
    if (ends_bang) {
	wr_putChars(w, "change_");
	l--;
    }
#if PROMISES
    // Our convention is: first letter of name is uppercase when
    // function takes and returns promises, lower case otherwise.
    char c = string_fetch(m->name, 0);
    if (promises)
      c = toupper(c);
    else
      c = tolower(c);
    wr_putChar(w, c);
    int i;
    for (i=1; i<l; i++) {
	char c = string_fetch(m->name, i);
	wr_putChar(w, c);
    }
#else //PROMISES
    wr_putString(w, m->name);
#endif //PROMISES
}

#define RTN_TRAILER "_rtn_class"
static void rtn_name(wr w, objtype t)
{
   int index;
   int max = string_length(t->name) - (sizeof(RTN_TRAILER) - 1);
   for (index = 0; index < max ; index++)
   	wr_putChar(w, t->name->chars[index]);
}


static void make_veneer_include(string filename, string contents)
{
    char const *fn = string_charp(filename);

    FILE *f = fopen(fn, "w");
    string ifdef_name;
    {
      textwr tw = textwr_new();
	wr w = textwr_as_wr(tw);
	char const *end = fn + strlen(fn) - 1;
	while (end > fn && *end != '/') end--;
	wr_putChar(w, '_');
	while (*end) {
	    if (*end != '.')
	      wr_putChar(w, toupper(*end));
	    else
	      wr_putChar(w, '_');
	    end++;
	}
	ifdef_name = textwr_toString(tw);
    }
    fprintf(f, "#ifndef %s\n#define %s\n\n",
	    string_charp(ifdef_name),
	    string_charp(ifdef_name));
    fputs(string_charp(contents), f);
    fprintf(f, "\n#endif /* %s */\n", string_charp(ifdef_name));
    fclose(f);
}


static bool is_parameterized(method m)
/* Return true if an argument or return value of "m" is parameterized */
{
  formal *args = UNPV(formal *, vec_items(m->arguments));
  type *returns = UNPV(type *, vec_items(m->returns));

  int i, 
      argc = vec_length(m->arguments),
      returnc = vec_length(m->returns);
  
  for (i=0; i<argc; i++)
      if (type_kind(args[i]->t) == PARAM_KIND) return TRUE;

  for (i=0; i<returnc; i++)
      if (type_kind(returns[i]) == PARAM_KIND) return TRUE;

  return FALSE;
}


static bool is_basic_type(type t) {
    return (t==(type)Null || t==(type)Int || t==(type)Bool 
		|| t==(type)Char || t==(type)Real);
}

static bool is_basic_type(objtype t) {
    return is_basic_type(objtype_as_type(t));
}

static bool is_rtn(objtype t)
{
    // check if name ends with _rtn_class
    int index = string_length(t->name) - (sizeof(RTN_TRAILER) - 1);
    if (index < 1) return FALSE;
    if (strcmp(RTN_TRAILER, &t->name->chars[index])) {
	return FALSE;
	}
    // check if exactly one method named invoke
    if ((vec_length(t->methods_) == 1)  && 
		!strcmp("invoke", string_charp(method_name(
			UNPV(method, vec_fetch(t->methods_, 0)))))) {
	return TRUE;
	}
    return FALSE;
}

static bool is_cxx_keyword(string s)
// Return true is "s" is known to be a C++ keyword
// XXX Currently only contains the the one keyword that causes trouble
// with existing class.  Should be extended to check for all of the C++
// keywords.
{
    char const *sc = string_charp(s);
    return (0==strcmp(sc, "delete"));
}	
	

static void put_formal_name(wr w, formal f)
{
    wr_putString(w, f->name);
    if (is_cxx_keyword(f->name))
	wr_putChar(w, '_');
}

void note_instantiated_type(type t) {
  string TN = type_name(t);
  if (dict_contains(inst_typeEnv, TN))
    return;
  dict_add(inst_typeEnv, type_name(t), (any)t);
  class_instn ci = type_as_class_instn(t);
  UNHANDLED_EXC;
  type p = UNPV(type, vec_fetch(class_instn_get_pargs(ci), 0));
  if (is_fully_instantiated_type(p)) {
    note_instantiated_type(p);
  }
}

// This routine will need to be updated when we support multiple
// parameters.  It currently assumes only one.
void output_instantiations(void *env, string key) {
    dict d = (dict)env;
    any a = dict_fetch(d, key);
    type t = UNPV(type, obj_as_type(any_get_obj(a)));
    UNHANDLED_EXC;
    // Dont generate pragma for params that are primitive values.
    class_instn ci = type_as_class_instn(t);
    UNHANDLED_EXC;
    type pt = UNPV(type, vec_fetch(class_instn_get_pargs(ci), 0));
    if (type_kind(pt) == PRIMITIVE_KIND)
      return;
    FILE *instF = fopen("th_instantiations.h", "a");
    textwr instTw = textwr_new();
    wr instWr = textwr_as_wr(instTw);
    bool Is_H;
    // Output a #include line for non-primitive, non-parameterized
    // types used as parameters.
    if (type_kind(pt) == OBJECT_KIND)
    {
	wr_putChars(instWr, "#include \"");
	typename(instWr, pt, Is_H, FALSE, FALSE);
	wr_putChars(instWr, ".h\"\n");
    }
    wr_putChars(instWr, "#pragma define_template ");
    typename(instWr, t, Is_H, TRUE, FALSE);
    fputs(string_charp(textwr_toString(instTw)), instF);
    fprintf(instF, "\n");
    fclose(instF);
}

void make_Cxx_veneer_inst_types(dict inst_typeEnv) {
  FILE *instF = fopen("th_instantiations.h", "w");
  fprintf(instF, "#ifndef _TH_INSTANTIATIONS_H\n");
  fprintf(instF, "#define _TH_INSTANTIATIONS_H\n\n");

  /* All the builting parameterized types */

  fprintf(instF, "#include \"th_array.h\"\n");
  fprintf(instF, "#include \"th_maybe.h\"\n");
  fprintf(instF, "#include \"th_sequence.h\"\n");
  fprintf(instF, "#include \"th_vec.h\"\n");
  fprintf(instF, "#include \"th_vector.h\"\n\n");

  fclose(instF);

  struct closure cl2 = {(ifunc) output_instantiations, inst_typeEnv};
//  printf("Iterating through inst_typeEnv\n");
  dict_keys(inst_typeEnv, cl2);
  delete_dict(inst_typeEnv);
//  printf("Done with inst_typeEnv\n");
  instF = fopen("th_instantiations.h", "a");
  fprintf(instF, "\n#endif /* _TH_INSTANTIATIONS_H */\n");
  fclose(instF);
}
