// Copyright 1995 Barbara Liskov

 // Generate the veneer stub files for a given type
// using the binary interface to the Thor FE

#include <iostream.h>
#ifdef __cplusplus
extern "C" {
#endif
#include "gen-Erlang-veneer.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "types/objtype_class.h"
#include "types/string_class.h"
#include "types/wr.h"
#include "types/textwr.h"
#include "types/method.h"
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

void  make_Erlang_veneer_files(objtype t);

#ifdef __cplusplus
}
#endif

static  void make_impl    (string filename, string contents);

string	gen_Erlang_veneer_file(objtype t, bool header);
void	gen_Erlang_veneer_stub (wr w, objtype t, method m);
void	gen_Erlang_veneer_args(wr w, method m, objtype t);
void	gen_Erlang_veneer_arg(wr w, formal f);
static  void method_name  (wr w, objtype t, method m, bool meth);
static  void put_type_name(wr w, type t);
static  bool is_parameterized(method m);
static  bool is_basic_type(type t);
static	bool is_basic_type(objtype t);

// Generate the Erlang veneer stub for a method


void gen_Erlang_veneer_stub(wr w, objtype t, method m) {
  int len, i;
  type ret;
  int num_rets = vec_length(m->returns);
  
  if (m->iter)	{		// iterators not supported 
    return;
  }

// Output the function name and arguments
  method_name(w, t, m, FALSE);
  wr_putChar(w, '(');
  gen_Erlang_veneer_args(w, m, t);
  wr_putChars(w, ") ->\n");

  
// Output function body  
//   wr_putChars(w,"  method_handle = 0,\n");

//   if (num_rets==1) {
//     wr_putChars(w, "  ");
//     wr_putChars(w, "res = 0,\n");
//   } else {
//     cerr << "Multiple arguments not supported!(1)";
//     exit(1);
//   }

// Find out whether the invoke can be deferred
  bool allow_deferred_invoke = TRUE;
  for (len=vec_length(m->returns), i=0; i < len ; i++) {
    if (is_basic_type(UNPV(type, vec_fetch(m->returns, i)))) {
      allow_deferred_invoke = FALSE;
      break;
    }
  }
  
// Output code to ensure that the method handle has been loaded,
// which is mapped in the database to a corresponding method object.
// The code to memoize the method handle includes two pieces of
// information: the name of the apparent type for the method, and
// the method index in the "methods" vector of the type object.

// XXX Note that this assumes there is a single globally unique name
// for each type, otherwise the FE will not be able to determine the
// the type to use for looking up the method object.

  wr_putChars(w, "  case get(\"");
  method_name(w, t, m, FALSE);
  wr_putChars(w, "\") of\n");
  wr_putChars(w, "    undefined ->\n");
  wr_putChars(w, "      Method_H = binary_veneer_EC:memoize_method_H(\"");
  wr_putString(w, type_name((type)t));
  wr_putChars(w, "\", ");

  // put the method's index in the "methods" vector of the Type object
  int starting_index = method_index(UNPV(method, vec_fetch(t->methods_, 0)));
  char index[16];
  sprintf(index, "%d", method_index(m) - starting_index);
  wr_putChars(w, index);
  wr_putChars(w, "),\n      put(\"");
  method_name(w, t, m, FALSE);
  wr_putChars(w, "\", Method_H);\n");
  wr_putChars(w, "    Method_H -> true\n  end,\n");
  
// Output the invoke call
  char defer[5];
  wr_putChars(w, "  binary_veneer_EC:begin_invoke(Self, Method_H, ");
  sprintf(defer, "%d", allow_deferred_invoke);
  wr_putChars(w, defer);
  wr_putChars(w, "),\n");

// Send the arguments    
  for (len=vec_length(m->arguments), i=0; i < len ; i++) {
    formal f = UNPV(formal, vec_fetch(m->arguments, i));
    wr_putChars(w, "  binary_veneer_EC:put_");  put_type_name(w, f->t);
    wr_putChars(w, "(V");     // Need to capitalize the first letter for Erlang
    wr_putString(w, f->name);
    wr_putChars(w, "),\n");
  }

  wr_putChars(w, "  case binary_veneer_EC:do_invoke(");
  wr_putChars(w, defer);
  wr_putChars(w, ") of\n");

// Get the results
  if ((len=vec_length(m->returns)) == 0) {
    wr_putChars(w, "    _ -> true\n");
  } else if (len == 1) {
    wr_putChars(w, "    0 -> Res = 0;\n");
    type ret = UNPV(type, vec_fetch(m->returns, 0));
    wr_putChars(w, "    _ -> Res = binary_veneer_EC:get_");
    put_type_name(w, ret);
    wr_putChars(w, "_E");
    wr_putChars(w, "()\n");
  } else {
    cerr << "Multiple arguments not supported!(2)";
    exit(1);
  }
  wr_putChars(w, "  end,\n");
  wr_putChars(w, "  binary_veneer_EC:end_invoke(");
  wr_putChars(w, defer);
  wr_putChars(w, ")");
  if (vec_length(m->returns)==0) {
    wr_putChars(w,".\n\n");
  } else {
    wr_putChars(w, ",\n  Res.\n\n");
  }
}

void make_Erlang_veneer_files(objtype t)
{
  string contents = gen_Erlang_veneer_file(t, FALSE);
  string vname = string_concat(string_new("th_"), t->name);
  make_impl(string_concat(vname, string_new(".erl")),
	    contents);

  FILE *typelist;
  if (typelist= fopen("th-typelist.h-th", "a")) {
      fprintf(typelist, "THOR_TYPE(th_%s);\n", string_charp(t->name));
      fclose(typelist);
  }
}

static void make_impl(string filename, string contents) {
  char const *fn = string_charp(filename);

  FILE *f = fopen(string_charp(filename), "w");
  fputs(string_charp(contents), f); fclose(f); 
}

void gen_Erlang_veneer_arg(wr w, formal f) {
  if (f->t == class_as_type(Null)) return;
  wr_putChars(w, ", V");        // need capital letter for var in Erlang
  wr_putString(w, f->name);
}

void gen_Erlang_veneer_method_args(wr w, method m) {
  int num_args = vec_length(m->arguments);
  formal *args = UNPV(formal *, vec_items(m->arguments));
  int i;
  for (i=0; i<num_args; i++) {
    gen_Erlang_veneer_arg(w, args[i]);
  }
  if (vec_length(m->extra_args)) {
    cerr << "Extra arguments not supported\n";
    exit(1);
      wr_putChars(w, ", ...");
    }
    
    int num_rets = vec_length(m->returns);
    type *returns = UNPV(type *, vec_items(m->returns));

  if (num_rets > 1) {
    cerr << "Multiple return values not supported\n";
    exit(1);
  }
}

void gen_Erlang_veneer_args(wr w, method m, objtype t)
// Write out the C arguments to this method
{
  if (t != class_as_objtype(Null)) {
    // C_veneer_typename(w, objtype_as_type(t)); no type in erlang
    wr_putChars(w, "Self");
  } else {
//    wr_putChars(w, "obj null_dummy");
    wr_putChars(w, "null_dummy");
  }
  int num_rets = vec_length(m->returns);

  if (m->iter) {
    cerr << "Iterator not supported\n";
    exit(1);
    wr_putChars(w, ", struct closure");
  }
  gen_Erlang_veneer_method_args(w, m);
}

void gen_Erlang_export_stub(wr w, objtype t, method m) {
  wr_putChars(w, "-export([");
  method_name(w, t, m, FALSE);
  char str[10];
  sprintf(str, "/%d]).", (vec_length(m->arguments)+1));
  wr_putChars(w, str);

// Output a comment describing the exceptions signalled
  int sig_len=vec_length(m->signals);
  if (sig_len > 0) {
    wr_putChars(w, "     %% signals(");
    for (int i=0; i < sig_len ; i++) {
      signal_ s = UNPV(signal_, vec_fetch(m->signals, i));
      if (i>0) wr_putChars(w, ", ");
      wr_putString(w, s->name);    
    }
    wr_putChars(w, ")");
  }

  wr_putChars(w, "\n");
}

string gen_Erlang_veneer_file(objtype t, bool header) {
  textwr tw = textwr_new();
  wr w = textwr_as_wr(tw);
  method *m = UNPV(method *, vec_items(t->methods_));
  
  int num_methods = vec_length(t->methods_);
  
  wr_putChars(w, "-module(th_");
  wr_putString(w, t->name);
  wr_putChars(w, ").\n\n");

  for (int i=0; i<num_methods; i++) {
    if (is_parameterized(m[i])) {
      wr_putChars(w, "%% Function with parameterized type omitted\n\n");
    } else if (vec_length(m[i]->extra_args)) {
      wr_putChars(w, "%% Function with variable no. of args omitted\n\n");
    } else if (vec_length(m[i]->returns) > 1) {
      wr_putChars(w, "%% Function with multiple return values omitted\n\n");
    } else {
      gen_Erlang_export_stub(w, t, m[i]);
    }
  }

  wr_putChars(w, "\n");

  for (i = 0; i < num_methods; i++) {
    if (!((is_parameterized(m[i])) ||
          (vec_length(m[i]->extra_args)) ||
          (vec_length(m[i]->returns) > 1))) {
      gen_Erlang_veneer_stub(w, t, m[i]);
    }
  }
  
  return textwr_toString(tw);
}


static void put_type_name(wr w, type t)
{
  if (t == (type)Int)
    wr_putChars(w, "int");
  else if (t == (type)Char)
    wr_putChars(w, "char");
  else if (t == (type)Bool)      
    wr_putChars(w, "bool");
  else if (t == (type)Real)      
    wr_putChars(w, "real");
  else
    wr_putChars(w, "handle");
}



static void method_name(wr w, objtype t, method m, bool meth)
{
    int l = string_length(m->name);
    bool ends_bang = string_fetch(m->name, l - 1) == '!';

//     No need to include object type
//     if (!meth) {
// 	wr_putString(w, t->name);
// 	wr_putChar(w, '_');
//     }
    if (ends_bang) {
	wr_putChars(w, "change_");
	l--;
    }
    int i;
    for (i=0; i<l; i++) {
	char c = string_fetch(m->name, i);
	wr_putChar(w, c);
    }
    wr_putChars(w, "_");        // prevent conflicting with Erlang's built-in
                                // functions
}

static bool is_parameterized(method m)
/* Return true if an argument or return value of "m" is parameterized */
{
  formal *args = UNPV(formal *, vec_items(m->arguments));;
  type *returns = UNPV(type *, vec_items(m->returns));;

  int i, 
    argc = vec_length(m->arguments),
    returnc = vec_length(m->returns);
  
  for (i=0; i<argc; i++)
    if (type_as_class(args[i]->t) == Param)
      return TRUE;

  for (i=0; i<returnc; i++)
    if (type_as_class(returns[i]) == Param)
      return TRUE;

  return FALSE;
}


static bool is_basic_type(type t)
{
    return (t==(type)Int || t==(type)Bool || t==(type)Char || t==(type)Real);
}


static bool is_basic_type(objtype t)
{
    return (t==(objtype)Int ||  t==(objtype)Bool 
	    || t==(objtype)Char || t==(objtype)Real);
}
