// Copyright 1995 Barbara Liskov

// Generate the veneer include and stub files for a given type
// using the binary interface to the Thor FE
#ifdef __cplusplus
extern "C" {
#endif
#include "gen-binary-C-veneer.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "types/objtype_class.h"
#include "types/string_class.h"
#include "types/wr.h"
#include "types/textwr.h"
#include "types/method.h"
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

void  make_binary_C_veneer_files(objtype t);

#ifdef __cplusplus
}
#endif

static  void make_impl    (string filename, string contents);
static  void make_veneer_include (string filename, string contents);

string	gen_binary_C_veneer_file(objtype t, bool header);
void	gen_binary_C_veneer_stub (wr w, objtype t, method m, bool header_only);
void	gen_binary_C_veneer_args(wr w, method m, objtype t);
void	gen_binary_C_veneer_arg(wr w, formal f);
static  void C_veneer_typename(wr w, type t);
static  void method_Cname  (wr w, objtype t, method m, bool meth);
static  void put_type_name(wr w, type t);
static  bool is_parameterized(method m);
static  bool is_basic_type(type t);
static	bool is_basic_type(objtype t);

// Generate the C veneer stub for a method, or just the header if
// "header_only" is true.

void gen_binary_C_veneer_stub(wr w, objtype t, method m, bool header_only)
{
    int len, i;
    type ret;
    int num_rets = vec_length(m->returns);
    int sig_len=vec_length(m->signals);

    if (m->iter)			// iterators not supported 
	return;

// Output the function type
    switch (num_rets) {
    default:
	wr_putChars(w, "void ");
	break;
    case 1:
	ret = UNPV(type, vec_fetch(m->returns, 0));
	C_veneer_typename(w, ret);
	wr_putChar(w, ' ');
    }


// Output the function name and arguments
    method_Cname(w, t, m, FALSE);
    wr_putChar(w, '(');
    gen_binary_C_veneer_args(w, m, t);
    wr_putChars(w, ")");

    if (header_only)  wr_putChars(w, ";");

// Output a comment describing the exceptions signalled
    if (sig_len > 0) {
	wr_putChars(w, "\n   /* signals(");
	for (i=0; i < sig_len ; i++) {
	    signal_ s = UNPV(signal_, vec_fetch(m->signals, i));
	    if (i>0) wr_putChars(w, ", ");
	    wr_putString(w, s->name);    
	}
	wr_putChars(w, ") */");
    }

    if (header_only) {wr_putChar(w, '\n'); return;}

    wr_putChars(w, "\n{\n");

    wr_putChars(w,"   static int method_handle = 0;\n");

    if (num_rets==1) {
	wr_putChars(w, "   ");
	C_veneer_typename(w, ret);
	wr_putChars(w, " res=0;\n");
    } else {
	for (len=vec_length(m->returns), i=0; i < len ; i++) {
	    wr_putChars(w, "   ");
	    wr_putChars(w, "*res"); wr_putChar(w, '1'+i);
	    wr_putChars(w, "=0;  ");
	}
	wr_putChar(w, '\n');
    }

    // Find out whether the invoke can be deferred

    bool allow_deferred_invoke = TRUE;
    for (len=vec_length(m->returns), i=0; i < len ; i++) {
	if (is_basic_type(UNPV(type, vec_fetch(m->returns, i)))) {
	    allow_deferred_invoke = FALSE;
	    break;
	}
    }

// Output code to ensure that the method handle has been loaded,
// which is mapped in the database to a corresponding method object.
// The code to memoize the method handle includes two pieces of
// information: the name of the apparent type for the method, and
// the method index in the "methods" vector of the type object.

// XXX Note that this assumes there is a single globally unique name
// for each type, otherwise the FE will not be able to determine the
// the type to use for looking up the method object.

    wr_putChars(w, "   memoize_method_handle(");
    // put type name
    wr_putChars(w, "\"");
    wr_putString(w, type_name((type)t));
    wr_putChars(w, "\", ");
    // put the method's index in the "methods" vector of the Type object
    int starting_index = method_index(UNPV(method, vec_fetch(t->methods_, 0)));
    char index[16];
    sprintf(index, "%d", method_index(m) - starting_index);
    wr_putChars(w, index);
    wr_putChars(w, ", &method_handle);\n");


// Output the invoke call
    wr_putChars(w, "   begin_invoke(self, method_handle");
    wr_putChars(w, allow_deferred_invoke ? ", TRUE);\n" : ", FALSE);\n");

// Send the arguments    
    for (len=vec_length(m->arguments), i=0; i < len ; i++) {
	formal f = UNPV(formal, vec_fetch(m->arguments, i));
	wr_putChars(w, "   put_");  put_type_name(w, f->t);
	wr_putChars(w, "(");    wr_putString(w, f->name);    wr_putChars(w, ");\n");
    }

    wr_putChars(w, "   if (do_invoke(");
    wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
    wr_putChars(w, ")) {\n");


// Get the results
    for (len=vec_length(m->returns), i=0; i < len ; i++) {
	type ret = UNPV(type, vec_fetch(m->returns, i));
	wr_putChars(w, "      get_");  put_type_name(w, ret);
	if (len > 1) {
	    wr_putChars(w, "(res");   wr_putChar(w, '1'+i); 
	    wr_putChars(w, ");\n"); 
	}
	else  {
	    wr_putChars(w, "(&res);\n");
	}
    }

    wr_putChars(w, "   }\n");
    wr_putChars(w, "   end_invoke(");
    wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
    wr_putChars(w, ");\n");
    if (vec_length(m->returns)==1)
	wr_putChars(w, "   return res;\n");

    wr_putChars(w,"}\n\n");
}

void make_binary_C_veneer_files(objtype t)
{
  string contents = gen_binary_C_veneer_file(t, FALSE);
  string vname = string_concat(string_new("th_"), t->name);
  make_impl(string_concat(vname, string_new(".c")),
	    contents);

  contents = gen_binary_C_veneer_file(t, TRUE);
  make_veneer_include(string_concat(vname, string_new(".h")),
	       contents);

  FILE *typelist;
  if (typelist= fopen("th-typelist.h-th", "a")) {
      fprintf(typelist, "THOR_TYPE(th_%s);\n", string_charp(t->name));
      fclose(typelist);
  }
}

static void make_impl(string filename, string contents)
{
  char const *fn = string_charp(filename);

  FILE *f = fopen(string_charp(filename), "w");
  fputs(string_charp(contents), f);
  fclose(f);
}

void gen_binary_C_veneer_arg(wr w, formal f)
{
//     if (f->t == class_as_type(Null)) return;
    wr_putChars(w, ", ");
    C_veneer_typename(w, f->t);
    wr_putChar(w, ' ');
    wr_putString(w, f->name);
}

void gen_binary_C_veneer_method_args(wr w, method m)
{
    int num_args = vec_length(m->arguments);
    formal *args = UNPV(formal *, vec_items(m->arguments));
    int i;
    for (i=0; i<num_args; i++) {
      gen_binary_C_veneer_arg(w, args[i]);
    }
    if (vec_length(m->extra_args)) {
      wr_putChars(w, ", ...");
    }
    
    int num_rets = vec_length(m->returns);
    type *returns = UNPV(type *, vec_items(m->returns));

    if (num_rets > 1)
      for (i=0; i<num_rets; i++) {
	wr_putChars(w, ", ");
	C_veneer_typename(w, returns[i]);
	wr_putChars(w, " *res");
	wr_putChar(w, i+'1');
    }
}

void gen_binary_C_veneer_args(wr w, method m, objtype t)
// Write out the C arguments to this method
{
/*
  if (t != class_as_objtype(Null)) {
*/
    C_veneer_typename(w, objtype_as_type(t));
    wr_putChars(w, " self");
/*
  } else {
    wr_putChars(w, "obj null_dummy");
  }
*/
  int num_rets = vec_length(m->returns);

  if (m->iter) 
    wr_putChars(w, ", struct closure");

  gen_binary_C_veneer_method_args(w, m);
}

string gen_binary_C_veneer_file(objtype t, bool header)	
{
    textwr tw = textwr_new();
    wr w = textwr_as_wr(tw);
    method *m = UNPV(method *, vec_items(t->methods_));

    int num_methods = vec_length(t->methods_);

    if (header) {
      wr_putChars(w, "#include \"th-typelist.h\"\n");
      wr_putChars(w, "#include \"../binary_veneer.h\"\n");
    } else {
      wr_putChars(w, "#include \"th_");
      wr_putString(w, t->name);
      wr_putChars(w, ".h-th\"\n\n");
      int num_methods = vec_length(t->methods_);
    }

    for (int i = 0; i < num_methods; i++) {
      if (is_parameterized(m[i])) {
	wr_putChars(w, "\n#if VENEER_SUPPORTS_PTYPES   /* currently not supported */\n");
	gen_binary_C_veneer_stub(w, t, m[i], header);
	wr_putChars(w, "#endif\n\n");
      } else if (vec_length(m[i]->extra_args)) {
	wr_putChars(w, "\n#if VENEER_SUPPORTS_VARARGS   /* currently not supported */\n");
	gen_binary_C_veneer_stub(w, t, m[i], header);
	wr_putChars(w, "#endif\n\n");
      }
      else
	gen_binary_C_veneer_stub(w, t, m[i], header);
    }

    return textwr_toString(tw);
}


static void put_type_name(wr w, type t)
{
  if (t == (type)Int)
    wr_putChars(w, "int");
  else if (t == (type)Char)
    wr_putChars(w, "char");
  else if (t == (type)Bool)      
    wr_putChars(w, "bool");
  else if (t == (type)Real)      
    wr_putChars(w, "real");
  else if (t == (type)Null)      
    wr_putChars(w, "null");
  else
    wr_putChars(w, "handle");
}

static void C_veneer_typename(wr w, type t)
{

/*
  if (t == (type)Null)
    wr_putChars(w, "void");
  else {
*/
      char *param_pos;
      char const *typename = string_charp(type_name(t));

      if (t!=(type)Int && t!=(type)Bool && t!=(type)Char && t!=(type)Real
		&& t!=(type)Null)
	  wr_putChars(w, "th_");

// hack to handle vec[any] and other parameterized types
      param_pos = strchr(typename, '[');
      if (!param_pos)
	  wr_putChars(w, typename);
      else {
	  char const *c = typename;
	  while (c < param_pos) 
	      wr_putChar(w, *c++);
      }
/*
  }
*/
}


static void method_Cname(wr w, objtype t, method m, bool meth)
{
    int l = string_length(m->name);
    bool ends_bang = string_fetch(m->name, l - 1) == '!';

    if (!meth) {
	wr_putString(w, t->name);
	wr_putChar(w, '_');
    }
    if (ends_bang) {
	wr_putChars(w, "change_");
	l--;
    }
    int i;
    for (i=0; i<l; i++) {
	char c = string_fetch(m->name, i);
	wr_putChar(w, c);
    }
}


static void make_veneer_include(string filename, string contents)
{
    char const *fn = string_charp(filename);
    string filename2 = string_concat(filename, string_new("-th"));

    FILE *f = fopen(string_charp(filename2), "w");
    string ifdef_name;
    {
      textwr tw = textwr_new();
	wr w = textwr_as_wr(tw);
	char const *end = fn + strlen(fn) - 1;
	while (end > fn && *end != '/') end--;
	wr_putChar(w, '_');
	while (*end) {
	    if (*end != '.')
	      wr_putChar(w, toupper(*end));
	    else
	      wr_putChar(w, '_');
	    end++;
	}
	ifdef_name = textwr_toString(tw);
    }
    fprintf(f, "#ifndef %s\n#define %s\n\n",
	    string_charp(ifdef_name),
	    string_charp(ifdef_name));
    fputs(string_charp(contents), f);
    fprintf(f, "\n#endif /* %s */\n", string_charp(ifdef_name));
    fclose(f);
}


static bool is_parameterized(method m)
/* Return true if an argument or return value of "m" is parameterized */
{
  formal *args = UNPV(formal *, vec_items(m->arguments));;
  type *returns = UNPV(type *, vec_items(m->returns));;

  int i, 
    argc = vec_length(m->arguments),
    returnc = vec_length(m->returns);
  
  for (i=0; i<argc; i++)
    if (type_as_class(args[i]->t) == Param)
      return TRUE;

  for (i=0; i<returnc; i++)
    if (type_as_class(returns[i]) == Param)
      return TRUE;

  return FALSE;
}


static bool is_basic_type(type t)
{
    return (t==(type)Int || t==(type)Bool || t==(type)Char || t==(type)Real)
		|| t==(type)Null;
}


static bool is_basic_type(objtype t)
{
    return (t==(objtype)Int ||  t==(objtype)Bool 
	    || t==(objtype)Char || t==(objtype)Real
		|| t==(objtype)Null);
}
