// Copyright 1995 Barbara Liskov

// Generate the veneer include and stub files for a given type
// using the Modula-3 ascii interface to the Thor FE
#ifdef __cplusplus
extern "C" { 
#endif

#include "types/ptype_class.h"    
#include "types/param.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "types/objtype_class.h"
#include "types/string_class.h"
#include "types/wr.h"
#include "types/textwr.h"
#include "types/param_class.h"
#include "types/method.h"
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include "common/openhashset.h"
  
  void  open_m3_veneer_files(char *filename);
  void  make_m3_veneer_files(objtype t);
  void  close_m3_veneer_files();

#ifdef __cplusplus
}
#endif

static  void    make_m3_file (string filename, string contents);
static  string  gen_header(objtype t);
static  string	gen_file(objtype t, bool header);
static  void	gen_stub (wr w, objtype t, method m, bool header);
static  void	gen_args(wr w, method m, objtype t, bool header);
static  void    gen_method_args(wr w, method m, bool header);

static  void m3_veneer_typename(wr w, type t, bool ref, bool include_params,
				bool import, bool header);

static  void m3_print_sl_name  (wr w, string tryname);
static  void m3_method_sl_Cname  (wr w, objtype t, method m, bool header);
static  void m3_method_Cname  (wr w, objtype t, method m, bool header);
static  void m3_put_type_name(wr w, type t);
static  bool m3_is_parameterized(method m);
static  bool m3_is_basic_type(type t);
static	bool m3_is_basic_type(objtype t);
static  bool m3_is_template(objtype t);
static  void m3_put_template(wr w, ptype pt);

/* TypeSet_m3 is used to store the set of types referenced in the methods of a given type */

#define addr_hash(x)  ((int)(long)x)
#define addr_compare(x,y)  ((x)==(y))

declareOpenHashSet(TypeSet_m3, objtype, addr_hash, addr_compare)
implementOpenHashSet(TypeSet_m3, objtype, addr_hash, addr_compare)

char* iname = "Thor";      // name of the M3 interface with all Thor types
char* hname = "Thor.i3";   // name of the M3 file containing the above interface

FILE *hfile;
FILE *m3makefile;

void open_m3_veneer_files(char *name)
     // ignore name; types are always put in an interface called Thor
{

  hfile = fopen(hname, "w");
  fputs("INTERFACE ", hfile);
  fputs(iname, hfile);
  fputs(";\n\nFROM Thread IMPORT Alerted;\n"
	"IMPORT Stubs;\n\n"
	"EXCEPTION\n  Failure(TEXT);\n  Except(TEXT);\n\nTYPE\n\n", 
	hfile);

  m3makefile = fopen("m3makefile.add", "w");
}

void close_m3_veneer_files()
{
  fputs("\n\nEND ", hfile);
  fputs(iname, hfile);
  fputs(".\n", hfile);
  fclose(hfile);

  fputs("\nLibrary(stubs)\n", m3makefile);
  fclose(m3makefile);
}

void make_m3_veneer_files(objtype t)
{
  bool is_templ = m3_is_template(t);

  string contents = gen_file(t, FALSE);
  string contents2 = gen_file(t, TRUE);
  string contents3 = gen_header(t);
  string vname = string_concat(string_new("Thor_"), t->name);
  if (!is_templ) 
      make_m3_file(string_concat(vname, string_new(".m3")), contents);
  else
      make_m3_file(string_concat(vname, string_new(".mg")), contents);
  fputs(string_charp(contents2), hfile);
  if (!is_templ) {
      make_m3_file(string_concat(vname, string_new(".i3")), contents3);
      fputs("Module(", m3makefile);
   } else{
      make_m3_file(string_concat(vname, string_new(".ig")), contents3);
      fputs("Generic_module(", m3makefile);
   }
  fputs(string_charp(vname), m3makefile);
  fputs(")\n", m3makefile);
}

string gen_header(objtype t)
{
  textwr tw = textwr_new();
  wr w = textwr_as_wr(tw);
  bool is_templ = m3_is_template(t);

  // Output first line of file

  if (is_templ) wr_putChars(w, "GENERIC ");
  wr_putChars(w, "INTERFACE Thor_");
  wr_putString(w, t->name);
  if (is_templ) {
    wr_putChar(w, '(');
    m3_put_template(w, (ptype) t);
    wr_putChar(w, ')');
  }
  
  wr_putChars(w, ";\n\nIMPORT Thread, ");
  wr_putChars(w, iname);
  wr_putChars(w, ";\n");
  wr_putChars(w, "\nTYPE\n  T = ");
  wr_putChars(w, iname);
  wr_putChars(w, ".");
  wr_putString(w, t->name);
  wr_putChars(w, ";\n\n");

  if (!is_templ) {
      wr_putChars(w, "PROCEDURE Class(): Thor.Class \n");
  } else {
      wr_putChars(w, "PROCEDURE PClass(): Thor.PClass \n");
  }
  wr_putChars(w, "  RAISES {Thor.Failure, Thread.Alerted};\n\n");

  wr_putChars(w, "PROCEDURE Narrow(th: Thor.any): T \n"
	      "  RAISES {Thor.Failure, Thor.Except, Thread.Alerted};\n");

  wr_putChars(w, "\nEND Thor_");
  wr_putString(w, t->name);
  wr_putChars(w, ".\n");

  return textwr_toString(tw);
}


static void make_m3_file(string filename, string contents)
{
  char const *fn = string_charp(filename);

  FILE *f = fopen(string_charp(filename), "w");
  fputs(string_charp(contents), f);
  fclose(f);
}

string gen_file(objtype t, bool header)	
{
  int i;
  textwr tw = textwr_new();
  wr w = textwr_as_wr(tw);
  bool is_templ = m3_is_template(t);

  method *m = UNPV(method *, vec_items(t->methods_));
  int num_methods = vec_length(t->methods_);

  // In a fit of pique, I have turned off all generic generation, because
  // I am too stressed to deal otherwise.

  if (is_templ) return textwr_toString(tw);


  // Output first line of file

  if (!header) {
    if (is_templ) wr_putChars(w, "GENERIC ");
    wr_putChars(w, "MODULE Thor_");
    wr_putString(w, t->name);
    if (is_templ) {
      wr_putChar(w, '(');
      m3_put_template(w, (ptype) t);
      wr_putChar(w, ')');
    }
    wr_putChars(w, " EXPORTS Stubs, Thor_");
    wr_putString(w, t->name);
    wr_putChars(w, ";\n");
    wr_putChars(w, "\nFROM Thread IMPORT Alerted;\nFROM ");
    wr_putChars(w, iname);
    wr_putChars(w, " IMPORT Except, Failure;\nIMPORT ");
    wr_putChars(w, iname);
    wr_putChars(w, ", Veneer");
    wr_putChars(w, ";\n");

    // keep a set of types mentioned in the methods of type t

    TypeSet_m3 types_used;
    

    //  This should be right for M3:

    for (i = 0; i < num_methods; i++) {
      formal *args = UNPV(formal *, vec_items(m[i]->arguments));;
      type   *ret = UNPV(type *, vec_items(m[i]->returns));;
      
      int num_args = vec_length(m[i]->arguments),
      num_rets = vec_length(m[i]->returns);           

      if (!m3_is_parameterized(m[i])) {
	for (int j = 0; j < num_args; j++)
	  types_used.insert(type_as_objtype(args[j]->t));
	for (j = 0; j < num_rets; j++)
	  types_used.insert(type_as_objtype(ret[j]));
      }
    }
    int width=0;

    bool class_imported=FALSE;
    bool pclass_imported=FALSE;
    bool any_imported=FALSE;

    for (TypeSet_m3::Elements each(&types_used); each.ok(); each.next()) {
      objtype included_t = each.get();
      
      if (!m3_is_basic_type(included_t)) {
        if (0 == width++) wr_putChars(w, "\nIMPORT ");
	else wr_putChars(w, (((width++)%5)==0) ? ";\nIMPORT " : ", ");
	m3_veneer_typename(w, objtype_as_type(included_t), 
			     FALSE, TRUE, TRUE, header);
	if (strcmp(type_name(objtype_as_type(included_t))->chars, "Class")==0)
	  class_imported=TRUE;
	if (strcmp(type_name(objtype_as_type(included_t))->chars, "PClass")==0)
	  pclass_imported=TRUE;
	if (strcmp(type_name(objtype_as_type(included_t))->chars, "any")==0)
	  any_imported=TRUE;
      }
    }
    if (0 != width) wr_putChars(w, ";\n");
    if (is_templ && !pclass_imported)
      wr_putChars(w, "IMPORT Thor_PClass;\n");
    if (!is_templ && !class_imported)
      wr_putChars(w, "IMPORT Thor_Class;\n");
    if (!any_imported)
      wr_putChars(w, "IMPORT Thor_any;\n");
  }

  if (header) {
    wr_putChars(w, "\n\n  ");
    wr_putString(w, t->name);
    wr_putChars(w, " <: Public_");
    wr_putString(w, t->name);
    wr_putChars(w, ";\n  Public_");
    wr_putString(w, t->name);
    wr_putChars(w, " = ");
    
    // Specify the superclasses:
    // Specify the first superclass and then
    // add in methods for others, if any
    
    
    if (vec_length(t->supertypes_)>0)
      m3_veneer_typename(w, UNPV(type,vec_fetch(t->supertypes_, 0)), 
			 FALSE, TRUE, FALSE, header);
    
    wr_putChars(w, " OBJECT\n");

    if (t == Any) 
      wr_putChars(w, "    handle:INTEGER;\n");

    wr_putChars(w,"  METHODS\n");

    if (t == Any) {
      // define an equals method on Thor_any
      wr_putChars(w, "    equals(a:any):BOOLEAN;\n");
    }

  } else {
    wr_putChars(w, "REVEAL ");
    wr_putChars(w, iname);
    wr_putChars(w, ".");
    wr_putString(w, t->name);
    wr_putChars(w, " = ");
    wr_putChars(w, iname);
    wr_putChars(w, ".Public_");
    wr_putString(w, t->name);
    wr_putChars(w, " BRANDED OBJECT\n");
    wr_putChars(w, "  OVERRIDES\n");
    
    // Output the standard overrides
    if (t == Any) {
      // equals method on Thor_any
      wr_putChars(w, "    equals := Impl_equals;\n");
    }
    
    
    // Then its own methods
    for (i = 0; i < num_methods; i++) {
      if (!(m[i]->iter)) {
	wr_putChars(w, "\n    ");
	m3_method_sl_Cname(w, t, m[i], header);
	wr_putChars(w, " := Impl_");
	m3_method_Cname(w, t, m[i], header);
	wr_putChars(w, ";");
      }
    }
    wr_putChars(w, "\n  END;\n\n");
    
    // Now output the implementing procedures for the builtin methods/procs
    if (t == Any) {
      // equals method on Thor_any
      wr_putChars(w, 
		  "PROCEDURE Impl_equals(self:T; a:T):BOOLEAN = \n"
		  "  BEGIN\n"
		  "    RETURN(self.handle = a.handle);\n"
		  "  END Impl_equals;\n\n");
    }
    
    // Output a procedure for returning the Class/PClass object for this type
    if (is_templ) {
	wr_putChars(w, "PROCEDURE PClass():Thor_PClass.T ");
    } else {
	wr_putChars(w, "PROCEDURE Class():Thor_Class.T ");
    }
    wr_putChars(w,
		 "  RAISES{Alerted,Failure} =\n"
		 "  VAR res := NEW(Thor_Class.T);\n"
		 "  BEGIN\n"
		 "    Veneer.Lock();\n"
		 "    Veneer.InstallObj(Veneer.WellKnown(\"");
    wr_putString(w, t->name);
    wr_putChars(w, 
		"\").handle, res);\n"
		"    Veneer.Unlock();\n"
		"    RETURN res;\n"
	);

    if (is_templ) {
	wr_putChars(w, "  END PClass;\n\n");
    } else {
	wr_putChars(w, "  END Class;\n\n");
    }

    // Output a proc for narrowing an any to this type
    wr_putChars(w, "PROCEDURE Narrow(th: Thor_any.T): T \n"
		"  RAISES{Alerted,Failure,Except} = \n"
		"  VAR res: T; \n"
		"  BEGIN \n"
		"    IF th.getClass().subtype(Class()) THEN \n"
		"      res := NEW(T); \n"
		"      Veneer.Lock(); \n"
		"      Veneer.InstallObj(th.handle, res); \n"
		"      Veneer.Unlock(); \n"
		"      RETURN res; \n"
		"    ELSE \n"
		"      RAISE Except(\"Narrow failed\") \n"
		"    END; \n"
		"  END Narrow; \n\n");

  } 
  
  // Output the methods

  for (i = 0; i < num_methods; i++) {
    if (vec_length(m[i]->extra_args)) {
      wr_putChars(w, "\n(* currently VAR ARGS not supported *)\n");
    }
    gen_stub(w, t, m[i], header);
  }
  
  if (header) 
    wr_putChars(w, "\n  END;\n");
  else {
    wr_putChars(w, "\nBEGIN");
    wr_putChars(w,"\nEND Thor_");
    wr_putString(w, t->name);
    wr_putChars(w,".\n");
  } 
  return textwr_toString(tw);
}




// Generate the m3 veneer stub for a method, or just the header if
// "header" is true.

void gen_stub(wr w, objtype t, method m, bool header)
{
  int len, i;
  type ret;
  int num_rets = vec_length(m->returns);
  int sig_len=vec_length(m->signals);


  if (m->iter)		// iterators not supported 
    return;
  
  // Output the function
  if (header) {
    wr_putChars(w, "    ");	// indent declarations
    m3_method_sl_Cname(w, t, m, header);
  } else {
    wr_putChars(w, "PROCEDURE Impl_");
    m3_method_Cname(w, t, m, header);
  }

  wr_putChar(w, '(');
  gen_args(w, m, t, header);
  wr_putChar(w, ')');
  
  // Output the return type
  switch (num_rets) {
  default:
    break;
  case 1:
    wr_putChar(w, ':');
    ret = UNPV(type, vec_fetch(m->returns, 0));
    m3_veneer_typename(w, ret, TRUE, TRUE, FALSE, header);
  }
  
  
  // Output the exceptions signalled
  wr_putChars(w, "\n        RAISES{Alerted,Failure,Except}");
  if (!header) wr_putChar(w, '=');
  else wr_putChars(w, ";\n");
  if (sig_len > 0) {
    wr_putChars(w, "          (*");
    for (i=0; i < sig_len ; i++) {
      signal_ s = UNPV(signal_, vec_fetch(m->signals, i));
      wr_putChars(w, " ");
      wr_putString(w, s->name);
    }
    wr_putChars(w, " *)\n");
  }

  if (header) return;

  
  if (num_rets==1) {
    wr_putChars(w, "  VAR\n    res:");
    m3_veneer_typename(w, ret, TRUE, TRUE, FALSE, header);
    wr_putChars(w, ";\n  BEGIN\n");
  } else {
    wr_putChars(w, "  BEGIN\n");
    for (i=0; i < num_rets; i++) {
      wr_putChars(w, "    res");
      wr_putChar(w, '1'+i); 
      wr_putChars(w, " := NIL;\n");
    }
  }


  // Find out whether the invoke can be deferred -- this is not currently
  // terribly useful
  bool allow_deferred_invoke = TRUE;
  for (len=vec_length(m->returns), i=0; i < len ; i++) {
    if (m3_is_basic_type(UNPV(type, vec_fetch(m->returns, i)))) {
      allow_deferred_invoke = FALSE;
      break;
    }
  }
  
  // Output the locking call
  wr_putChars(w, "    Veneer.Lock();\n");

  // Output the invoke call
  wr_putChars(w, "      Veneer.BeginInvoke(");
  
  // Always use the method name, not index
  wr_putChars(w, "self.handle, \"");
  wr_putString(w, m->name);
  wr_putChars(w, allow_deferred_invoke ? "\", TRUE);\n" : "\", FALSE);\n");

  // Send the arguments    
  for (len=vec_length(m->arguments), i=0; i < len ; i++) {
    formal f = UNPV(formal, vec_fetch(m->arguments, i));
    wr_putChars(w, "      Veneer.Put");  m3_put_type_name(w, f->t);
    wr_putChars(w, "(");    
    m3_print_sl_name(w, f->name);  
    if (!m3_is_basic_type(f->t))
      wr_putChars(w, ".handle");
    wr_putChars(w, ");\n");
  }

  wr_putChars(w, "      Veneer.DoInvoke(");
  wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
  wr_putChars(w, ");\n");
  
  
  // Get the results
  for (len=vec_length(m->returns), i=0; i < len ; i++) {
    type ret = UNPV(type, vec_fetch(m->returns, i));
    
    if (m3_is_basic_type(ret)) {
      wr_putChars(w, "      res");
      if (len > 1) {wr_putChar(w, '1'+i); }
      wr_putChars(w, " := Veneer.Get");  m3_put_type_name(w, ret);
      wr_putChars(w, "();\n");
    } else {
      wr_putChars(w, "      res");  
      if (len > 1) wr_putChar(w, '1'+i); 
      wr_putChars(w, " := NEW(");
      m3_veneer_typename(w, ret, FALSE, TRUE, FALSE, header);
      wr_putChars(w, ");\n      Veneer.InstallObj(");
      wr_putChars(w, "Veneer.GetHandle(), res");
      if (len > 1) wr_putChar(w, '1'+i); 
      wr_putChars(w, ");\n");
    }
  }

  wr_putChars(w, "      Veneer.EndInvoke(");
  wr_putChars(w, allow_deferred_invoke ? "TRUE" : "FALSE");
  wr_putChars(w, ");\n    Veneer.Unlock();\n");
  if (vec_length(m->returns)==1)
    wr_putChars(w, "    RETURN res;\n");
  wr_putChars(w,"  END Impl_");
  m3_method_Cname(w, t, m, header);
  wr_putChars(w,";\n\n");
}

void gen_args(wr w, method m, objtype t, bool header)
     // Write out the m3 arguments to this method
{
  gen_method_args(w, m, header);

  if (m->iter) 
    wr_putChars(w, "(* iterator *)");
}

void gen_method_args(wr w, method m, bool header)
{
  int num_args = vec_length(m->arguments);
  formal *args = UNPV(formal *, vec_items(m->arguments));
  int i;
  bool comma = FALSE;
  if (!header) {
    comma = TRUE;
    wr_putChars(w, "self:T");
  }
  for (i=0; i<num_args; i++) {
    if (comma) wr_putChars(w, "; ");
    // XXX why this line? if (args[i]->t == class_as_type(Null)) continue;
    m3_print_sl_name(w, args[i]->name);
    wr_putChar(w, ':');
    m3_veneer_typename(w, args[i]->t, TRUE, TRUE, FALSE, header);
    comma = TRUE;
  }
  if (vec_length(m->extra_args)) {
    wr_putChars(w, " (* extra args not supported *) ");
  }
  
  int num_rets = vec_length(m->returns);
  type *returns = UNPV(type *, vec_items(m->returns));
  
  if (num_rets > 1)
    for (i=0; i<num_rets; i++) {
      if (comma) wr_putChars(w, "; ");
      comma = TRUE;
      wr_putChars(w, "VAR res");
      wr_putChar(w, i+'1');
      wr_putChar(w, ':');
      m3_veneer_typename(w, returns[i], FALSE, TRUE, FALSE, header);
    }
}


static void m3_put_type_name(wr w, type t)
{
  if (t == (type)Int)
    wr_putChars(w, "Int");
  else if (t == (type)Char)
    wr_putChars(w, "Char");
  else if (t == (type)Bool)      
    wr_putChars(w, "Bool");
  else if (t == (type)Real)      
    wr_putChars(w, "Real");
  else if (t == (type)Null)      
    wr_putChars(w, "Null");
  else
    wr_putChars(w, "Handle");
}

static void m3_veneer_typename(wr w, type t, bool ref, bool include_params,
			       bool import, bool header)
     // Print the veener typename for type "t" on wr "w".

     // If "include_params" is true, and t is a parameterized type, print
     // out the parameters for "t".    

     // Ignore ref--it doesn't do the right thing

     // If "import" is true, then leave off the .T
{
  bool builtin;

    char *param_pos;
    char typename[255];

    // if (ref && !m3_is_basic_type(t)) wr_putChars(w, "ref ");
    
    strcpy(typename, type_name(t)->chars);

    if (t!=(type)Int && t!=(type)Bool && t!=(type)Char && t!=(type)Real 
	&& t!=(type)Null) {
      if (header)
	wr_putChars(w, "");
      else
	wr_putChars(w, "Thor_");
      builtin = FALSE;
    } else {
      import = TRUE;      // don't put .T on built in types
      builtin = TRUE;
    }

    if (builtin && (strcmp(typename, "real") == 0))
      strcpy(typename, "REAL");
    if (builtin && (strcmp(typename, "int") == 0))
      strcpy(typename, "INTEGER");
    if (builtin && (strcmp(typename, "bool") == 0))
      strcpy(typename, "BOOLEAN");
    if (builtin && (strcmp(typename, "char") == 0))
      strcpy(typename, "CHAR");
    if (builtin && (strcmp(typename, "null") == 0))
      strcpy(typename, "NULL");

    // hack to handle vec[any] and other parameterized types -- huh huh
    param_pos = strchr(typename, '[');
    if (!param_pos)
      wr_putChars(w, typename);
    else 
      {
      char *c = typename; 
      while (*c) {
	if (*c=='[')
	  if (!include_params)
	    break;
	  else {
	      wr_putChars(w, "OfAny"); 
              // XXX for now all parameters are converted into Any
	      break;
	  }
	else
	  wr_putChar(w, *c);
	++c;
      }
    }
    if (!import && !header)
      wr_putChars(w, ".T");
    else
      if (strcmp(typename, "vec") == 0)
	wr_putChars(w, "OfAny");
}


static void m3_print_sl_name(wr w, string tryname)
{
  char* name;
  int i;
  int l = string_length(tryname);
  name = new char[l+3];
  for (i=0; i<l; i++) {
    char c = string_fetch(tryname, i);
    name[i+1]=c;
  }
  name[0]=' ';
  name[l+1]=' ';
  name[l+2]=0;
  wr_putString(w, tryname);
}
  

static void m3_method_sl_Cname(wr w, objtype t, method m, bool header)
{
  char* name;
  int i;
  int l = string_length(m->name);
  bool ends_bang = string_fetch(m->name, l - 1) == '!';
  
  if (ends_bang) {
    wr_putChars(w, "change_");
    l--;
  }
  name = new char[l+3];
  for (i=0; i<l; i++) {
    char c = string_fetch(m->name, i);
    name[i+1]=c;
  }
  name[0]=' ';
  name[l+1]=' ';
  name[l+2]=0;
  name[l+1]=0;
  wr_putChars(w, name+1);
}
  

static void m3_method_Cname(wr w, objtype t, method m, bool header)
{
  int l = string_length(m->name);
  bool ends_bang = string_fetch(m->name, l - 1) == '!';
  
  if (ends_bang) {
    wr_putChars(w, "change_");
    l--;
  }
  int i;
  for (i=0; i<l; i++) {
    char c = string_fetch(m->name, i);
    wr_putChar(w, c);
  }
}


static bool m3_is_parameterized(method m)
     /* Return true if an argument or return value of "m" is parameterized */
{
  formal *args = UNPV(formal *, vec_items(m->arguments));;
  type *returns = UNPV(type *, vec_items(m->returns));;

  int i, 
  argc = vec_length(m->arguments),
  returnc = vec_length(m->returns);
  
  for (i=0; i<argc; i++)
    if (type_as_class(args[i]->t) == Param)
      return TRUE;

  for (i=0; i<returnc; i++)
    if (type_as_class(returns[i]) == Param)
      return TRUE;

  return FALSE;
}


static bool m3_is_basic_type(type t)
{
    return (t==(type)Int || t==(type)Bool || t==(type)Char || t==(type)Real
	    || t==(type)Null);
}


static bool m3_is_basic_type(objtype t)
{
  return (t==(objtype)Int ||  t==(objtype)Bool 
	  || t==(objtype)Char || t==(objtype)Real || t==(objtype)Null);
}

bool m3_is_template(objtype t)
{
  return (get_any_class((any)t) == PClass);
}


void m3_put_template(wr w, ptype pt)
{
  int num_params = vec_length(pt->params);
  param *parm = UNPV(param *, vec_items(pt->params));

  for (int i=0; i<num_params; i++) {
    if (i) wr_putChars(w, ", ");
    wr_putChars(w, "Thor_");
    wr_putString(w, parm[i]->name);
  }
}



// Don't forget cast stubs!

