// Copyright 1995 Barbara Liskov


#include "runtime/except.h"
#include "runtime/alloc.h"
#include "runtime/obj_class.h"
#include "types/type.h"
#include "types/any.h"
#include "types/class.h"

#include "cache/cache.h"
#include "common/hash.h"
#include "common/intarray.h"
#include "common/th_assert.h"
#include <string.h>

#include "fe_config.h"
#include "handle.h"
#include "future.h"
#include "handle_gc.h"
#include "cache/indirect_table.h"

#include "config/vdefs/DEBUG_HANDLES.h"

#if DEBUG_HANDLES 
#include <stdio.h>
#endif

implementArray(HandleToAny, htoa_entry)

handle_table::handle_table() {
  htoa_entry dummy;

  htoa_table = HandleToAny(1000);

  htoa_table.append(dummy);     // Eat up handle 0

  first_free_handle = 0;
}

handle_val handle_table::handle_to_any(handle h, bool fix_object) {
  if (h < 0) {		// look in future table
    return FEConf->ft->future_to_any(h);
  }
  // look in handle table
  if (h == 0 || h >= htoa_table.size()) {
      if (h == 0)
	  th_fail("Null handle received: Probably due to uninitialized variable at client or a variable that contains an invalid handle returned during an unhandled exception");
      else
	  th_fail("Very big handle received");
  }
  if (!htoa_table[h].full)
    th_fail("Handle has already been freed");
  handle_val result;
  result.val = ACTUAL_ANY_POINTER(htoa_table[h].full);
  result.tag = Anyf;
  if (is_persistent_object((obj)result.val)) {
    if (fix_object) {
      FIX(result.val, any);
    } else {
      Xref x;
      result.val = (any) cache_real_obj((obj) result.val, &x);
    }
  }
  
#if DEBUG_HANDLES
  fprintf(stderr, "looked up handle %d, found object %x\n", h, result.val);
#endif
  
  return result;
}

handle handle_table::any_to_handle(any a)
{
  obj o;
  class_ c = get_any_class(a);
  o = any_as_obj(a);

  FIX(o, obj);     /* Make sure the object isn't a surrogate */    

  core co = BUMP(core, o, *o);
  handle h = co->handle_;
  if (!h) {                            // Get a new handle
    h = first_free_handle;
    if (!h) {                          // Extend array by 1
      htoa_entry dummy;
      htoa_table.append(dummy);
      h = htoa_table.size() - 1;
    }
    first_free_handle = htoa_table[h].next;

    // Store an indirect pointer if indirection is being used
    htoa_table[h].full = obj_as_any(get_storable_pointer(o));
    
    // This is the only place in the code where a handle is assigned
    // to an object (a future is never assigned by the FE)
    co->handle_ = h;

#if DEBUG_HANDLES
	fprintf(stderr, "New handle: %d for 0x%lx\n", h, o);
        fflush(stderr);
#endif

  }
  th_assert(h, "Zero handle at end of any_to_handle!?");

  return h;
}

type handle_table::get_type(handle_val val)
{
#ifdef __GNUC__
    return handle_val_get_type(val);
#else
    return val.get_type();
#endif
}

void handle_table::free_handles(int* handles, int size) {
  for (int i = 0; i < size; i++) {
    handle h = handles[i];
    th_assert(h, "Bad handle value to free_handles");
    if (h < 0) {
      FEConf->ft->free_future(h);
    } else {
      // Do not fix the object as it is not needed
      handle_val res = handle_to_any(h, FALSE);
      obj o = any_as_obj(res.val);
      if (o) {
	// Object is in the cache
	core c = BUMP(core, o, *o);
	// clear the handle in the objects handle_ field
	// so that a new handle will be assigned the next time
	// we try to use it.
	c->handle_ = 0;	    
      }
      // add the htoa table entry to the free list
      htoa_table[h].full = 0;
      htoa_table[h].next = first_free_handle;
      first_free_handle = h;

#if DEBUG_HANDLES
      fprintf(stderr, "Freed handle: %d \n", h);
      fflush(stderr);
#endif
    }
  }
}

//**********************************************************************
//
// Handle Iterator code...
//
//**********************************************************************

HandleIter::HandleIter() {
  index = 0;
  next();
}

bool HandleIter::ok() {
  return (index < FEConf->ht->htoa_table.size());
}

handle HandleIter::get_handle() {
  return (handle)index;
}

obj HandleIter::get_obj() {
  return any_as_obj(FEConf->ht->htoa_table[index].full);
}

void HandleIter::next() {
  index++;
  while (index < FEConf->ht->htoa_table.size()) {
    if (FEConf->ht->htoa_table[index].full)
      return;
    index++;
  }
}

void HandleIter::replace(handle h, obj val) {
  if (h == index)
    FEConf->ht->htoa_table[h].full = obj_as_any(val);
  else
    th_fail("Tried to mutate the handle table outside the correct iteration");
}

#ifdef __GNUC__
type handle_val_get_type(handle_val hv) {
  switch(hv.tag) {
    case Anyf:
      return class_as_type(get_obj_class(any_as_obj(hv.val)));
#else
type handle_val::get_type() {
  switch(tag) {
    case Anyf:
      return class_as_type(get_obj_class(any_as_obj(val)));
#endif
    case Intf:
      return (type)Int;
    case Boolf:
      return (type)Bool;
    case Charf:
      return (type)Char;
    case Realf:
      return (type)Real;
    default: {

#if DEBUG_HANDLES
#ifdef __GNUC__
      fprintf(stderr, "handle_val_get_type: Unrecognized tag!\n");
#else
      fprintf(stderr, "get_type: Unrecognized tag!\n");
#endif
#endif
      exc = &exc_not_possible;
      return 0;
    }
  }
}





