/*  Copyright 1995 Barbara Liskov */

#ifndef _HANDLE_H
#define _HANDLE_H

#include "types/any.h"
#include "handle_def.h"
#include "common/iter.h"
#include "common/array.h"
#include "runtime/value.h"
#include "runtime/obj.h"

/*
  A one-to-one map exists from Theta objects to handles, which are
really just integers.  The calls "atoh" and "htoa" apply the map and
its inverse.

  The set of handles that are not mapped onto are called ``free
handles''.  The call "free_handles" allows the handles and its
associated values to each be rebound to different handles and values,
respectively.  Initially, all handles are free.

  The set of values that are not mapped onto are the "primitive
values", such as integers and characters.

  The handle implementation tries to make the set of objects that
clients know about map onto handles that are integers as small as
possible.  "free_handles" facilitates this goal, while also helping
to minimize the total memory consumed by handle tables. Thus, it
should be called whenever reasonable.

Futures are transmitted as handles with negative indices (i.e. #-1).
The client can request that the result of an operation is mapped to a
given future by sending a "pinvoke" and transmitting the future
number(s) for the result(s) after the usual arguments.  Subsequent
invokes and pinvokes can refer to either the future or the actual
handle.  

   For example, the veneer might combine several calls to the list_rest
method using following invocation string:
	"pinvoke #1 rest #-1; pinvoke #-1 rest #-2; pinvoke #-2 rest #-3"
The veneer could then send:
	"invoke #-3 first"

The frontend sends no results back from a pinvoke command.  If an
exception occurs, the future will be mapped to an exception object, 
and client calls will be provided to detect this fact and determine 
the exception.  (This is not yet implemented.)

XXX Note that this **WILL BREAK** if multiple clients attempt to connect to
the same FE because they will write over each others future entries.
*/

enum handle_tag {
  Emptyf = 0,
  Anyf, Intf, Realf,  Charf,  Boolf
};

typedef struct {
  char tag;
  any val;

#ifndef __GNUC__
  type get_type();
  // Return the type of val.
#endif

} handle_val;


#ifdef __GNUC__
type handle_val_get_type(handle_val hv);
#endif

struct htoa_entry {

  htoa_entry();

  any full; /* is zero if free */
  int next; /* index of next free entry, 0 if none */
};

inline htoa_entry::htoa_entry() {
  full = 0;
  next = 0;
}

declareArray(HandleToAny, htoa_entry)

class handle_table {
 public:
  friend class HandleIter;

  handle_table();

handle_val handle_to_any(handle h, bool fixobject);
/*
  Signal "not_possible" if "h" is free or is not a valid handle.
  Note that the object pointer in the "any" is canonically the
  the most specific pointer possible. If fixobject is TRUE, this code will fix
  the object. If it is FALSE, it will return a pointer to the actual object
  (if object is not in cache, it sets the returned value to be 0)
  */

handle any_to_handle(any a);
/*
  Map "a" onto a handle, allocating a free handle if necessary.
  Some values cannot be mapped onto handles. It is a checked
  runtime error to do so.
  */

type get_type(handle_val);
/* Assumes handle_val is a valid structure returned by ftoa or htoa. */
/* Returns the type of the object contained in handle_val. */
/* Signals not_possible if it cannot recognize the type of the object. */

void free_handles(int* handles, int size);
/*
   Indicate that no expectation exists that the values bound to
   the handles in "handles" will continue to be bound to them after this call.
   Note: This call will never result in communication with the OR
*/
  
void allow_performance_tuning();
/*
   Allow the handle implementation to do whatever work is necessary to
   provide optimal performance. This call may be expensive, and should
   be called only when the system is idle.  We may want to have
   support for doing preemptive halting of this call.
*/

 private:

 int first_free_handle;

 HandleToAny htoa_table;

};

#endif /* _HANDLE_H */


