// Copyright 1995 Barbara Liskov

extern "C" {
#include <stdlib.h>
}
#include "cache/cache.h"
#include "cache/gc.h"
#include "cache/net.h"
#include "common/Timer.h"
#include "common/or_stat.h"
#include "common/network.h"
#include "common/Monitor.h"
#include "common/obj_bitfield.h"
#include "types/class_class.h"
#include "types/any.h"
#include "types/interface.h"
#include "runtime/commit.h"
#include "runtime/stats.h"
#include "binary_interface.h"
#include "runtime/same_object.h"
#include "fe_config.h"

#include "config/vdefs/COLLECT_STATS.h"
#include "config/vdefs/PROF_CLIENT_CONTROL.h"

interface Iobj;

struct interfacedv_s {
    struct dv_s super;
    bool (*interface_objequal)		(interface self, any a1, any a2);
    bool (*interface_commit)		(interface self);
    bool (*interface_abort)		(interface self);
    any  (*interface_int2any)		(interface self, int i);
    any  (*interface_bool2any)		(interface self, bool b);
    any  (*interface_char2any)		(interface self, char c);
    any  (*interface_null2any)		(interface self, int n);
    int  (*interface_force_to_int)	(interface self, any a);
    bool (*interface_force_to_bool)	(interface self, any a);
    char (*interface_force_to_char)	(interface self, any a);
    int  (*interface_force_to_null)	(interface self, any a);
    bool (*interface_close)		(interface self, bool slave);
    bool (*interface_force_gc)		(interface self);
    void (*interface_zero_stats)	(interface self);
    int  (*interface_get_stats)		(interface self);
#if PROF_CLIENT_CONTROL
    void (*interface_toggle_monitor)	(interface self, bool prof);
#endif
};

struct interface_s {
    union {
	struct core_s inh;
	struct interfacedv_s *methods;
    } hdr;
    connection conn;
};

class_ InterfaceC;

/* Methods of this type */

extern bool interface_objequal (interface self, any a1, any a2) {
    return same_object(0, a1, a2);
}

Timer commit_timer, make_timer, send_timer, recv_timer;
float commit_time, make_time, send_time, recv_time;
Timer cumm_make_timer, cumm_send_timer, cumm_recv_timer, cumm_commit_timer;

extern bool interface_commit(interface self) {
    // The commit times are printed only if the environment variable
    // THOR_PRINT_TIME has been set
    static bool printres = getenv("THOR_PRINT_TIME")? TRUE: FALSE;

    cumm_commit_timer.start();
    commit_timer.reset();  commit_timer.start();
    commit_result res = commit_trans();

    if (res != COMMITTED) {
      exc = &exc_abort;
    }
    commit_timer.stop(); commit_time = commit_timer.elapsed();
    cumm_commit_timer.stop(); commit_time = commit_timer.elapsed();
    if (printres) {
	fprintf(stderr, 
		"Make = %9.5f, Send = %9.5f, Recv  = %9.5f, Total = %9.5f\n",
		make_time, send_time, recv_time, commit_time);
    }
    return TRUE;
}

extern bool interface_abort(interface self) {
    abort_trans();
    return TRUE;
}

// Generate the different "objectifying" methods using this macro
#define BASIC2ANY(ctype, methodname, func) \
extern any interface_##methodname(interface self, ctype i) {\
    any result = func(i);\
    return result;\
}

BASIC2ANY(int,  int2any,  int_as_any)
BASIC2ANY(bool, bool2any, bool_as_any)
BASIC2ANY(char, char2any, char_as_any)
BASIC2ANY(int,  null2any, null_as_any)

// Generate the different methods for forcing obejcts to baisc values
#define ANY2BASIC(ctype, thortype, classname, fieldname)\
ctype interface_force_to_##thortype(interface self, any a) {\
    FIX(a, any);\
    class_ cl = get_any_class(a);\
    if (cl == classname) {\
	ctype result = any_get_value(a).fieldname;\
	return result;\
    }\
    exc = &exc_wrong_type;\
}

ANY2BASIC(int,  int,  Int,  i)
ANY2BASIC(bool, bool, Bool, b)
ANY2BASIC(char, char, Char, c)
ANY2BASIC(int,  null, Null, i)

extern bool interface_close(interface self, bool slave) {
    connection_close(self->conn);
    // Clear Iobj so that binarySingleOp can detect connection is closed
    Iobj = NULL;
    return TRUE;
}

extern bool interface_force_gc(interface self) {
    cache_force_gc();
    return TRUE;
}

extern void interface_zero_stats(interface self) {
#if COLLECT_STATS
    zero_stats();
#endif
}

#if PROF_CLIENT_CONTROL
extern void interface_toggle_monitor(interface self, bool prof) {
  if (prof)
    resume_monitor();
  else
    stop_monitor();
}
#endif

extern int interface_get_stats(interface self) {
    // Print the stats for the FE
    extern Timer fe_fetch_time;
    extern Timer fe_net_fetch_time;
    extern int no_fetches;
    extern int nbytes_fetched;
    extern int nduplicates;
    extern Timer gc_timer, shrink_timer, actual_gc_timer, rest_gc_timer;
    extern long ngc;
    extern float used_space_after_gc;
    extern float used_space_before_gc;
    extern Timer cumm_make_timer, cumm_send_timer, cumm_recv_timer;
    extern Timer cumm_ros_send_timer, cumm_mos_send_timer, cumm_nos_send_timer;

    extern int meta_calculated_size;
    gc->check_heap(FALSE, TRUE, 0, FALSE);
    int actual_meta_size = meta_calculated_size;
    gc->check_heap(FALSE, FALSE, 0, FALSE);

    fprintf(stderr, "Statistics STARTED\n");
    int no_alloc = cache_occupied_slots()*8;
    fprintf(stderr, "Size of meta heap = %d (reachable = %d, diff = %d) bytes)\n",
	    meta_size_alloc, actual_meta_size,
	    meta_size_alloc - actual_meta_size);
    fprintf(stderr, "Cache space used = %d Kbytes, No. fetches = %d\n",
	    no_alloc/1024, no_fetches);
    fprintf(stderr, "Total Fetch time = %f\n", 
	    fe_fetch_time.elapsed());
    fprintf(stderr, "Total Fetch time (w/o cache code) =  %f\n",
	    fe_net_fetch_time.elapsed());
    fprintf(stderr, "Total bytes fetched (w/o duplicates) = %d\n",
	    nbytes_fetched);
    fprintf(stderr, "Total number of duplicates fetched = %d\n",
	    nduplicates);
    fprintf(stderr, "Total accumulated time during shrinking and gc = %f\n",
	    gc_timer.elapsed());
    fprintf(stderr, "Number of gcs = %d\n", ngc);
    fprintf(stderr, "Average cache space used (bytes) before gc = %f\n", 
	    used_space_before_gc*8);
    fprintf(stderr, "Average cache space used (bytes) after gc = %f\n",
	    used_space_after_gc*8);

    float shrink_time = shrink_timer.elapsed();
    float actual_gc_time = actual_gc_timer.elapsed();
    float rest_gc_time = rest_gc_timer.elapsed();
    fprintf(stderr, "Shrink = %9.5f, Actual = %9.5f, Rest =%9.5f\n",
	    shrink_time, actual_gc_time, rest_gc_time);

    // Transaction statistics
    float cumm_make_time = cumm_make_timer.elapsed();
    float cumm_send_time = cumm_send_timer.elapsed();
    float cumm_recv_time = cumm_recv_timer.elapsed();
    float cumm_commit_time = cumm_commit_timer.elapsed();
    fprintf(stderr, "Transaction Statistics\n");
    fprintf(stderr, 
	    "C_Make = %9.5f, C_Send = %9.5f, C_Recv  = %9.5f, C_Total = %9.5f\n",
	    cumm_make_time, cumm_send_time, cumm_recv_time, cumm_commit_time);

    float cumm_ros_send_time = cumm_ros_send_timer.elapsed();
    float cumm_mos_send_time = cumm_mos_send_timer.elapsed();
    float cumm_nos_send_time = cumm_nos_send_timer.elapsed();
    fprintf(stderr, "C_Ros = %9.5f, C_Mos = %9.5f, C_Nos = %9.5f\n",
	    cumm_ros_send_time, cumm_mos_send_time, cumm_nos_send_time);

    // Get the OR statistics:
    or_stat stat;
    if (!get_or_stats(FEConf->initial_or_num, stat)) 
	// XXX Hard-wired the inital or
	return 0;
    fprintf(stderr, "OR statistics:\n");
    sub_stats(stat, initial_stat);
    report_stats(stderr, stat);
    fprintf(stderr, "Statistics DONE\n");
    return 0;
#if COLLECT_STATS
    int ops = stats->ops; /* Note that stats->ops is a long  */
    return ops;
#else
    th_fail("Getstats called with COLLECT_STATS off");
    return 0;
#endif
}

struct interfacedv_s interface_methods = {
    { 0, 0, STD_FOFFSET, 0, 0,
      normal_get_address, normal_get_class},
    interface_objequal,
    interface_commit,
    interface_abort,
    interface_int2any,
    interface_bool2any,
    interface_char2any,
    interface_null2any,
    interface_force_to_int,
    interface_force_to_bool,
    interface_force_to_char,
    interface_force_to_null,
    interface_close,
    interface_force_gc,
    interface_zero_stats,
    interface_get_stats
#if PROF_CLIENT_CONTROL
    ,interface_toggle_monitor
#endif
};

DV interface_dh[] = {(DV)&interface_methods};

/* Creator */

extern interface interface_new(void* c) {
    interface result = (interface) NEW(struct interface_s);
    init_obj_hdr((obj) result, InterfaceC);
    result->conn = (connection)c;

    result->hdr.inh.size = 1;
    result->hdr.inh.bitfields = OBJ_BF_ALLDATA;
    /* XXX The above shoud be correctly set in the class */

    return result;
}

void initInterface() {
    InterfaceC->dh = interface_dh;
    InterfaceC->dhsize = 1;
    interface_methods.super.c = InterfaceC;
}
