/*  Copyright 1995 Barbara Liskov */

#include "invoke.h"
#include "runtime/obj_class.h"
#include "runtime/alloc.h"
#include "runtime/except.h"
#include "types/type.h"
#include "types/method.h"
#include "types/string_class.h"
#include "types/class_class.h"
#include "types/int.h"
#include "types/char.h"
#include "types/real.h"
#include "types/any.h"
#include "types/vec_class.h"
#include "types/str.h"
#include "types/vec_instns.h"
#include "common/th_assert.h"
#include "common/fail.h"
#include "cache/gc_register.h"

#include <stdlib.h>

#define CAN_HANDLE_MORE_THAN_NINE_ARGS 0

/* "resultsN" is an N-element vec allocated once and reused, to cut 
   down on allocation during invocation.  It must be used very carefully: in
   particular, it must never be used as a "real" vec by any code.
   It should only be used for returning results.
*/

#define USUAL_MAX_RESULTS 2
static vec results0;
static vec results1;
static vec results2;

static bool initialized = FALSE;

void init_invocations() {
  if (initialized) return;
  results0 = make_vec_Any(0, TRUE);
  results1 = make_vec_Any(1, TRUE);
  results2 = make_vec_Any(2, TRUE);
  gc_register_root((obj *)&results0);
  gc_register_root((obj *)&results1);
  gc_register_root((obj *)&results2);
  initialized = TRUE;
}

static void vec_any_store(vec v, int i, any a)
{
    vec_store(v, i, any_as_obj(a));
}

obj perform_call(ofunc f, fevalue *args, int num_args)
#define a(i) args[i].o
{
    obj ret;
    switch(num_args) {
      case 0:
	fail("num_args = 0 in perform_call");
      case 1: ret = (*f)(a(0)); break;
      case 2: ret = (*f)(a(0), a(1)); break;
      case 3: ret = (*f)(a(0), a(1), a(2)); break;
      case 4: ret = (*f)(a(0), a(1), a(2), a(3)); break;
      case 5: ret = (*f)(a(0), a(1), a(2), a(3), a(4)); break;
      case 6: ret = (*f)(a(0), a(1), a(2), a(3), a(4), a(5)); break;
      case 7: ret = (*f)(a(0), a(1), a(2), a(3), a(4), a(5), a(6)); break;
      case 8: ret = (*f)(a(0), a(1), a(2), a(3), a(4), a(5), a(6),
			 a(7)); break;
      case 9: ret = (*f)(a(0), a(1), a(2), a(3), a(4), a(5), a(6),
			 a(7), a(8)); break;
      default:
	th_assert(CAN_HANDLE_MORE_THAN_NINE_ARGS,
	       "Cannot handle more than 9 arguments to invoke or iterate");
    }
    return ret;
#undef a
}


/*

  Accumulate the actual arguments to pass to the function, as
  described in the file "calling_conventions". Keeps track of the actual
  number of arguments in "*passed_arguments_p", and the actual arguments
  in "*args2".  Return the function to be called in "*f", and a buffer
  for multiple return values (if needed) in "*mret". If no buffer for
  multiple return values is needed, "*mret" is not accessed.

*/
static void setup_arguments(fevalue **args2_p, int *passed_arguments_p,
			    ofunc *f, fevalue **mret,
			    fevalue self, objtype t, method m, vec args,
			    struct closure cl)
#define args2 (*args2_p)
#define passed_args (*passed_arguments_p)
{
	DV __dv;
	dtentry *dte;
	int num_rets = vec_length(m->returns);
	int method_index = m->index;
	int nargs = vec_length(args);
	passed_args = 1 + nargs; /* "1 +" for "self" */
	if (m->iter) {
	    passed_args += 2;
	    /* The struct closure gets passed as two arguments */
	} else if (num_rets > 1) {
	    *mret = (fevalue *)malloc(num_rets * sizeof(fevalue));
	    /* This array will be more than big enough */
	    passed_args++;
	}
	  
	{
	   /* If this is not a primitive value, fetch the dispatch
	      entry from the object, else get it from the object's
	      class.
	      */
	    if (PRIMITIVE_KIND != type_kind(objtype_as_type(t))) {
		__dv = (*self.o);
		self.o = BUMP(obj, self.o, __dv);
		dte = (dtentry *)(&__dv->get_class) + method_index;
	    } else {
		class_ c = objtype_as_class(t);
		dte = (dtentry *)(&c->dh[c->dhsize - 1]->get_class) +
		  method_index;
	    }
	}
	
	*f = dte->f;

	args2 = (fevalue *)malloc(passed_args * sizeof(void *));

	/* Now, accumulate the actual arguments in args2 */
	{
	    int cur = 0;
	    int i;
	    formal *forms = UNPV(formal *, vec_items(m->arguments));
	    int nformals = vec_length(m->arguments);
	    if (m->self_type != objtype_as_type(Any))
		args2[cur++] = any_get_value(obj_as_any(self.o));
	    else	     /* Don't unwrap any if an any is called for! */
		args2[cur++] = self;
	    if (!m->iter && num_rets > 1) {
		args2[cur++].o = (obj)*mret;
	    }
	    if (m->iter) {
		args2[cur++].o = (obj)cl.f;
		args2[cur++].o = (obj)cl.env;
	    }
	    for (i = 0; i < vec_length(args); i++) {
		int j = (i < nformals) ? (i) : (nformals - 1);
		formal f = forms[j];
		FIX_FAST(f,formal,formal_methods);
		if (f->t != objtype_as_type(Any))
		    args2[cur++] = any_get_value(UNPV(any,vec_fetch(args, i)));
		else	     /* Don't unwrap any if an any is called for! */
		    args2[cur++].o = vec_fetch(args, i);
	    }
	}
#undef args2
#undef passed_args
}

vec unpack_results(fevalue *rets, vec retTypes)
{
    vec r;
    int i;
    int num_rets = vec_length(retTypes);
    th_assert(initialized, "invocation code not initialized");

    switch (num_rets) {
    case 0: 
      r = results0; break;
    case 1: 
      r = results1; break;
    case 2: 
      r = results2; break;
    default: 
      r = make_vec_Any(num_rets, TRUE); break;
    }

    for (i = 0; i < num_rets; i++) {
	fevalue v = rets[i];
	type t = UNPV(type, vec_fetch(retTypes, i));
	vec_store(r, i, PV(any_as_obj(make_any(t, v))));
    }

    return r;
}

static signal_ method_get_signal(method m, string name)
{
    int num_sigs = vec_length(m->signals);
    signal_ *sigs = UNPV(signal_ *, vec_items(m->signals));
    int i;
    for (i = 0; i < num_sigs; i++) {
	if (string_equal(name, sigs[i]->name)) return sigs[i];
    }
    return 0;
}

#ifndef NDEBUG
static void printExcError(type t, method m)
{
    fail("FATAL ERROR\n"
		 "Method %.50s.%.50s raised the unexpected"
		 " exception %.50s",
		 string_charp(type_name(t)),
		 string_charp(method_name(m)),
		 string_charp(exc->name));
}
#endif

extern rt_result invoke(fevalue self, objtype t, method m, vec args)
{
    int passed_args;
    fevalue *args2;
    ofunc f;
    struct closure dummy;
    obj ret;
    fevalue *mret;
    setup_arguments(&args2, &passed_args, &f, &mret,
		    self, t, m, args, dummy);
#if 0
    {
	int i;
	printf("About to call 0x%x, arguments are:\n", f);
	for (i = 0; i < passed_args; i++) {
	    printf("Argument %d: 0x%lx\n", i, args2[i]);
	}
    }
#endif
    
    /* The arguments are all ready to go at this point. */
    exc = EXC_NONE; /* just to make sure */
    {
	jmp_entry tmp;
	TOP_LEVEL_FAILURE(tmp);
    }
    if (!exc) ret = perform_call(f, args2, passed_args);
	/* Yes, Virginia, "exc" can be non-zero here. */
    free(args2);
    if (exc) {
	rt_result r;
	if (exc != &exc_failure) {
	    signal_ s = method_get_signal(m, exc->name);
#ifndef NDEBUG
	    if (!s) printExcError(objtype_as_type(t), m);
#endif
	    r.u.exceptional.val = unpack_results(exc_value, s->returns);
	} else {
	    assert(get_obj_class((obj)(TH_EXC_VALUE->s)) == String);
	    r.u.exceptional.val = make_vec_Any(1, TRUE);
	    vec_store(r.u.exceptional.val, 0, PV(TH_EXC_VALUE->s));
	}
	r.tag = rt_exceptional;
	r.u.exceptional.exc = exc;
	exc = EXC_NONE;
	return r;
    } else {
	rt_result r;
	int num_rets = vec_length(m->returns);
	r.tag = rt_normal;
	if (num_rets > 1) {
	    r.u.normal = unpack_results(mret, m->returns);
	} else {
	    fevalue v;
            if (num_rets == 1) {
	        r.u.normal = results1;
	    }
	    else {
	        r.u.normal = results0;
	    }
	    if (num_rets == 1) {
		v.o = ret; /* Assume this copies everything... */
		vec_any_store(r.u.normal, 0,
			      make_any(UNPV(type,
					    vec_fetch(m->returns,
						      0)), v));
	    }
	}
	return r;
    }
}

extern rt_result iterate(fevalue self, objtype t, method m, vec args,
			 struct closure cl)
{
    int passed_args;
    fevalue *args2;
    ofunc f;
    obj ret;
    rt_result r;
    setup_arguments(&args2, &passed_args, &f, 0,
		    self, t, m, args, cl);
    exc = EXC_NONE; /* just to make sure */
    ret = perform_call(f, args2, passed_args);
    free(args2);
    if (exc) {
	signal_ s = method_get_signal(m, exc->name);
#ifndef NDEBUG
	if (!s) printExcError(objtype_as_type(t), m);
#endif
	r.tag = rt_exceptional;
	r.u.exceptional.exc = exc;
	r.u.exceptional.val = unpack_results(exc_value, s->returns);
	exc = EXC_NONE;
    } else {
	r.tag = rt_normal;
	r.u.normal = make_vec_Any(0, TRUE);
    }
    return r;
}
