/*  Copyright 1995 Barbara Liskov */

#ifndef _PARSE_TREE_H
#define _PARSE_TREE_H

#include "common/basic.h"
#include "runtime/obj.h"
#include "runtime/disphdr.h"
#include "types/class.h"
#include "types/class_class.h"
#include "types/type.h"
#include "types/method.h"
#include "types/string_class.h"
#include "types/objtype_class.h"
#include "config/vdefs/DEBUG_CONTROLS.h"
#include "common/iter.h"
#include "type_check.h"

// The following classes define the program tree structure that will
// be used by the FE to store a batched control structure.
//
// Class hierarchy:
//             Node
//          /       \            \              \
//    MethodNode   LoopNode     IterNode       IfNode
//
//                  VarNode  
//            /                  \         
//       ValNode              FutureNode
//        /                   /        \
//  BasicValNode    BasicFutureNode   FutureOffsetNode
//
// The actual tree will have the following structure:
// - Down points to the next Node on the stack.
// - Next points to the next node in a sequence of nodes.

static inline 
bool is_basic_type(type t)
{
    return (t==(type)Null || t==(type)Int || t==(type)Bool || 
		t==(type)Char || t==(type)Real);
}

// These are auxiliary functions used by VarNodes.

inline ofunc compute_any_func(obj receiver, method m) {
  //
  // Requires: receiver has been properly offset.
  //           receiver is not a basic value.
  //
  // Effects: returns the function to be called when method m is executed on
  //          receiver.

  dtentry *dte;

  DV _dv = *receiver;
  dte = (dtentry *)(&_dv->get_class) + m->index;

  return dte->f;
}

inline ofunc compute_basic_func(method m) {
  //
  // Requires: m->self_type is a basic type.
  //
  // Effects: returns the function to be called when method m is executed

  dtentry *dte;

  class_ c = (class_) m->self_type;  // ok because self->type is basic.
  dte = (dtentry *)(&c->dh[c->dhsize - 1]->get_class) + m->index;

  return dte->f;
}

// A VarNode holds a "variable".  A VarNode is either a ValNode, in
// which case it contains a constant object, or a FutureNode, in which
// case it holds a future.  The get_val method returns the actual
// object referred to by the VarNode.

// VarNodes are tagged so the actual type can be determined at runtime.
enum VarTag {Val_Node, Future_Node};

class VarNode {
 public:
  VarNode(VarTag tag_) : tag(tag_) {}

  virtual ~VarNode() {}
  // Virtual destructor so that superclass destructors will be called
  // automatically.

  virtual any get_val() = 0;
  // Return the object stored in this node.

  virtual ofunc compute_func(method m) = 0;
  // Return the function corresponding to method m, where
  // "this->get_val()" is the receiver of the call.

#if DEBUG_CONTROLS
  virtual void print_node() = 0;
#endif

  VarTag tag;
};

class ValNode : public VarNode {
 // A ValNode contains a constant object, which has already been
 // offset properly.  As the receiver or argument of a call, its value
 // can be accessed directly.
 public:
  ValNode(any o_, int hdroff = 0) : VarNode(Val_Node)
  { o = (hdroff ? (any)(((DV)o_)+hdroff) : o_); }  // Add the offset */
						   // immediately. 
  virtual ~ValNode() {}

  virtual any get_val() {
    return o;
  }

  virtual ofunc compute_func(method m) {
    return compute_any_func((obj)o, m);
  }

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  any o;              // Actual value
};

class BasicValNode : public ValNode {
  // A BasicValNode contains a constant basic value.
 public:
  BasicValNode(any o_) : ValNode(o_) {}

  // get_val() Inherited from ValNode.

  virtual ofunc compute_func(method m) {
    return compute_basic_func(m);
  }
};

class FutureNode : public VarNode {
  // A FutureNode contains a pointer to an object that is computed by
  // a method call.
 public:
  FutureNode(obj_ref r_) : VarNode(Future_Node), r(r_) {}

  virtual ~FutureNode() {}

  virtual any get_val() {
    return (any)*r;
  }

  virtual ofunc compute_func(method m) {
    return compute_any_func(*r, m);
  }

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  obj_ref r;
};

class FutureBasicNode : public FutureNode {
  // A FutureBasicNode contains a pointer to a basic value object.
 public:
  FutureBasicNode(obj_ref r_) : FutureNode(r_) {}

  virtual ~FutureBasicNode() {}

  // get_val() Inherited from FutureNode.

  virtual ofunc compute_func(method m) {
    return compute_basic_func(m);
  }
};

class FutureOffsetNode : public FutureNode {
  // A FutureOffsetNode is a specialized FutureNode, which contains a
  // non-zero offset to be added to the object before it is used as an
  // argument.
 public:
  FutureOffsetNode(obj_ref r_, int hdroff_) : FutureNode(r_), 
    hdroff(hdroff_) {}

  virtual ~FutureOffsetNode() {}

  virtual any get_val();

  virtual ofunc compute_func(method m);

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  int hdroff;         // Offset computed between actual and expected type
};

// Nodes are tagged so the actual type can be determined at runtime.
enum NodeTag {If_Node, Iter_Node, Loop_Node, Method_Node};

class Node {
  // Base class for parse tree nodes.
 public:
  Node(NodeTag tag_) : tag(tag_), next(0), down(0) {}

  virtual ~Node() { delete next; }
  // Virtual destructor so that all subclasses will still call it when
  // they get destroyed. 

  virtual bool add_node(Node *n) = 0;
  // This method will be used to add a new node to this node in the
  // parse tree.

  virtual void eval_node() = 0;
  // This method is called during execution of the parse tree.

#if DEBUG_CONTROLS
  virtual void print_node() = 0;
#endif

  NodeTag tag;
  Node *next;        // Next level in parse tree.
  Node *down;        // Next node in the stack (see controls.cc).
};

class MethodNode : public Node {
  // A method node represents a method call in the parse tree.
  // A method node must at the very least contain a function to be
  // called at execution time.  This function may have to be computed
  // at run time.
 public:
  MethodNode() : Node(Method_Node) {}

  virtual ~MethodNode() {}

  virtual bool add_node(Node *n) { if (next = n) return TRUE; return FALSE; }

 protected:
  ofunc f;                      // Actual function to be called.
};

// The following are specialized MethodNodes, optimized to speed up
// execution of the method call.

class Method_cell_getNode : public MethodNode {
  // Method_cell_getNode is an optimization to make cell->get run
  // fast.  Rather than computing the function to be called from the
  // receiver, etc, it calls cell_get directly.
 public:

  Method_cell_getNode(VarNode *receiver_, obj_ref result_, 
		      type result_type_) : MethodNode(),
  receiver(receiver_), result(result_), result_type(result_type_),
  not_basic_result(!is_basic_type(result_type)) {}

  virtual ~Method_cell_getNode() { delete receiver; }

  virtual void eval_node();

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode *receiver;
  obj_ref result;
  type result_type;
  bool not_basic_result;
};

class Method_cell_putNode : public MethodNode {
  // Method_cell_putNode is an optimization to make cell->put run
  // fast.  Rather than computing the function to be called from the
  // receiver, etc, it calls cell_put directly.
 public:
  Method_cell_putNode(VarNode *receiver_, VarNode *arg_) : MethodNode(),
  receiver(receiver_), arg(arg_) {}

  virtual ~Method_cell_putNode() { delete receiver; delete arg; }

  virtual void eval_node();

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode *receiver, *arg;
};

class MethodConstantNode : public MethodNode {
  // A MethodConstantNode is a MethodNode who\'s receiver is a
  // constant object.  This means that the function f can be computed
  // at constrution time.
 public:
  MethodConstantNode(ValNode *receiver_, method m) : MethodNode(),
  receiver(receiver_->get_val())  { f = receiver_->compute_func(m); }

  virtual ~MethodConstantNode() {}

 protected:
  any receiver;                 // Constant receiver of method call.
};

class Method0_0_ConstantNode : public MethodConstantNode {
  // A Method0_0_ConstantNode is a MethodConstantNode that has 0
  // arguments and 0 return values.
 public:
  Method0_0_ConstantNode(ValNode *receiver_, method m) 
  : MethodConstantNode(receiver_, m) {}

  virtual ~Method0_0_ConstantNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif
};

class Method1_0_ConstantNode : public MethodConstantNode {
  // A Method1_0_ConstantNode is a MethodConstantNode that has 0
  // arguments and 1 return value.
 public:
  Method1_0_ConstantNode(ValNode *receiver_, method m, obj_ref result_,
			 type result_type_) 
  : MethodConstantNode(receiver_, m), result(result_),
    result_type(result_type_), not_basic_result(!is_basic_type(result_type)) {}

  virtual ~Method1_0_ConstantNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  obj_ref result;                   // Future for result of method call.    
  type result_type;
  bool not_basic_result;
};

class Method0_1C_ConstantNode : public MethodConstantNode {
  // A Method0_1C_ConstantNode is a MethodConstantNode that has 1
  // constant argument and 0 results.
 public:
  Method0_1C_ConstantNode(ValNode *receiver_, method m, ValNode *arg_) 
  : MethodConstantNode(receiver_, m), arg(arg_->get_val()) {}

  virtual ~Method0_1C_ConstantNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  any arg;                      // Single, constant argument.
};

class Method0_1F_ConstantNode : public MethodConstantNode {
  // A Method0_1F_ConstantNode is a MethodConstantNode that has 1
  // future argument and 0 results.
 public:
  Method0_1F_ConstantNode(ValNode *receiver_, method m, FutureNode *arg_) 
  : MethodConstantNode(receiver_, m), arg(arg_) {}

  virtual ~Method0_1F_ConstantNode() { delete arg; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  FutureNode *arg;               // Single, future argument.
};

class Method1_1C_ConstantNode : public Method1_0_ConstantNode {
  // A Method1_1C_ConstantNode is a MethodConstantNode that has 1
  // constant argument and 1 result.
 public:
  Method1_1C_ConstantNode(ValNode *receiver_, method m, ValNode *arg_,
			  obj_ref result_, type result_type_) 
  : Method1_0_ConstantNode(receiver_, m, result_, result_type_),
    arg(arg_->get_val()) {}

  virtual ~Method1_1C_ConstantNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  any arg;                      // Single, constant argument.
};

class Method1_1F_ConstantNode : public Method1_0_ConstantNode {
  // A Method1_1F_ConstantNode is a MethodConstantNode that has 1
  // future argument and 1 result.
 public:
  Method1_1F_ConstantNode(ValNode *receiver_, method m, FutureNode *arg_,
			  obj_ref result_, type result_type_) 
  : Method1_0_ConstantNode(receiver_, m, result_, result_type_), arg(arg_) {}

  virtual ~Method1_1F_ConstantNode() { delete arg; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  FutureNode *arg;              // Single, future argument.
};

class Method1_n_ConstantNode : public Method1_0_ConstantNode {
  // A Method1_n_ConstantNode is a MethodConstantNode that has 1
  // result and n arguments.
 public:
  Method1_n_ConstantNode(ValNode *receiver_, method m, int num_args_,
			 VarNode **args_, obj_ref result_, type result_type_)
  : Method1_0_ConstantNode(receiver_, m, result_, result_type_),
    num_args(num_args_), args(args_) {}

  virtual ~Method1_n_ConstantNode() { delete [] args; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode **args;              // Array of arguments.
  int num_args;
};

class Methodk_n_ConstantNode : public MethodConstantNode {
  // A Methodk_n_ConstantNode is a MethodConstantNode that has k
  // results and n arguments.
 public:
  Methodk_n_ConstantNode(ValNode *receiver_, method m, int num_args_,
			 VarNode **args_, int num_rets_, int
			 *results_, type* result_types_)
  : MethodConstantNode(receiver_, m), num_args(num_args_),
    args(args_), num_rets(num_rets_), results(results_),
    result_types(result_types_) {}

  virtual ~Methodk_n_ConstantNode() { 
    delete [] args;
    delete [] results;
  }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode **args;              // Array of arguments.
  int num_args;
  int *results;                // Array of result futures
  type *result_types;
  int num_rets;
};

class MethodFutureNode : public MethodNode {
  // A MethodFutureNode is a MethodNode who\'s receiver
  // is a future.
 public:
  MethodFutureNode(FutureNode *receiver_, method m_) : MethodNode(),
  receiver(receiver_), m(m_) {}

  virtual ~MethodFutureNode() { delete receiver; }

 protected:
  FutureNode* receiver;                 // Future receiver of method call.
  method m;
};

class Method0_0_FutureNode : public MethodFutureNode {
  // A Method0_0_FutureNode is a MethodFutureNode that has 0
  // arguments and 0 return values.
 public:
  Method0_0_FutureNode(FutureNode *receiver_, method m) 
  : MethodFutureNode(receiver_, m) {}

  virtual ~Method0_0_FutureNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif
};

class Method1_0_FutureNode : public MethodFutureNode {
  // A Method1_0_FutureNode is a MethodFutureNode that has 0
  // arguments and 1 return value.
 public:
  Method1_0_FutureNode(FutureNode *receiver_, method m, obj_ref result_,
			 type result_type_) 
  : MethodFutureNode(receiver_, m), result(result_),
    result_type(result_type_),
    not_basic_result(!is_basic_type(result_type)) {}

  virtual ~Method1_0_FutureNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  obj_ref result;                   // Future for result of method call.    
  type result_type;
  bool not_basic_result;
};

class Method0_1C_FutureNode : public MethodFutureNode {
  // A Method0_1C_FutureNode is a MethodFutureNode that has 1
  // constant argument and 0 results.
 public:
  Method0_1C_FutureNode(FutureNode *receiver_, method m, ValNode *arg_) 
  : MethodFutureNode(receiver_, m), arg(arg_->get_val()) {}

  virtual ~Method0_1C_FutureNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  any arg;                      // Single, constant argument.
};

class Method0_1F_FutureNode : public MethodFutureNode {
  // A Method0_1F_FutureNode is a MethodFutureNode that has 1
  // future argument and 0 results.
 public:
  Method0_1F_FutureNode(FutureNode *receiver_, method m, FutureNode *arg_) 
  : MethodFutureNode(receiver_, m), arg(arg_) {}

  virtual ~Method0_1F_FutureNode() { delete arg; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  FutureNode *arg;               // Single, future argument.
};

class Method1_1C_FutureNode : public Method1_0_FutureNode {
  // A Method1_1C_FutureNode is a MethodFutureNode that has 1
  // constant argument and 1 result.
 public:
  Method1_1C_FutureNode(FutureNode *receiver_, method m, ValNode *arg_,
			  obj_ref result_, type result_type_) 
  : Method1_0_FutureNode(receiver_, m, result_, result_type_),
    arg(arg_->get_val()) {}

  virtual ~Method1_1C_FutureNode() {}

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  any arg;                      // Single, constant argument.
};

class Method1_1F_FutureNode : public Method1_0_FutureNode {
  // A Method1_1F_FutureNode is a MethodFutureNode that has 1
  // future argument and 1 result.
 public:
  Method1_1F_FutureNode(FutureNode *receiver_, method m, FutureNode *arg_,
			  obj_ref result_, type result_type_) 
  : Method1_0_FutureNode(receiver_, m, result_, result_type_), arg(arg_) {}

  virtual ~Method1_1F_FutureNode() { delete arg; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  FutureNode *arg;              // Single, future argument.
};

class Method1_n_FutureNode : public Method1_0_FutureNode {
  // A Method1_n_FutureNode is a MethodFutureNode that has 1
  // result and n arguments.
 public:
  Method1_n_FutureNode(FutureNode *receiver_, method m, int num_args_,
			 VarNode **args_, obj_ref result_, type result_type_)
  : Method1_0_FutureNode(receiver_, m, result_, result_type_),
    num_args(num_args_), args(args_) {}

  virtual ~Method1_n_FutureNode() { delete [] args; }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode **args;              // Array of arguments.
  int num_args;
};

class Methodk_n_FutureNode : public MethodFutureNode {
  // A Methodk_n_FutureNode is a MethodFutureNode that has k
  // results and n arguments.
 public:
  Methodk_n_FutureNode(FutureNode *receiver_, method m, int num_args_,
			 VarNode **args_, int num_rets_, int
			 *results_, type* result_types_)
  : MethodFutureNode(receiver_, m), num_args(num_args_),
    args(args_), num_rets(num_rets_), results(results_),
    result_types(result_types_) {}

  virtual ~Methodk_n_FutureNode() { 
    delete [] args; 
    delete [] results;
  }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 private:
  VarNode **args;              // Array of arguments.
  int num_args;
  int *results;                // Array of result futures
  type *result_types;
  int num_rets;
};

class MethodXNode : public MethodNode {
  // A MethodXNode contains all the information necessary for a method
  // call.
 public:
  MethodXNode(VarNode *s_, method m_, int num_args_, VarNode** args_,
	     int num_rets_, int *results_, type *result_types_) :
          MethodNode(), self(s_), m(m_), num_args(num_args_),
	  args(args_), num_rets(num_rets_), results(results_),
	  result_types(result_types_) {}

  virtual ~MethodXNode() { 
    delete self;
    delete [] args; 
    delete [] results;
  }

  virtual void eval_node();
#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  VarNode *self;
  method m;
  int num_args;
  VarNode** args;
  int num_rets;
  int *results;
  type *result_types;
};

class MethodIterNode : public MethodXNode {
  // A MethodIterNode contains all the information necessary for an
  // iterator method call.
 public:
  MethodIterNode(VarNode *s, method m, int num_args, VarNode** args,
	     int num_rets, int *results, type *result_types) :
          MethodXNode(s, m, num_args, args, num_rets, results,
		      result_types) {}

  virtual ~MethodIterNode() {}

  virtual bool add_node(Node *n) { return FALSE; }

  virtual void eval_node() {}

  MethodNode *specialize(Node *body);
};

class IfNode : public Node {
  // An IfNode represents an if statement at the fe.
 public:
  IfNode() : Node(If_Node), expr(0), then_branch(0), else_branch(0),
  state(Expression) {}

  virtual ~IfNode() {
    delete expr;
    delete condition;
    delete then_branch;
    delete else_branch;
  }

  virtual bool add_node(Node *n);
  virtual void eval_node();

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

  bool begin_then(VarNode *c);
  // This function is used to end the construction of the expression
  // and begin construction of the body of the then branch.  c is the
  // condition for executing the then branch.

  bool begin_else();
  // This function is used to end the construction of the then branch,
  // and begin construction of the else branch.

  Node *optimize();
  // This function returns an optimized node to replace this ifnode in
  // the parse tree.

private:
  MethodNode *expr;
  VarNode *condition;
  Node *then_branch;
  Node *else_branch;

  enum {Expression, ThenBody, ElseBody} state;
  Node *tail;
};

class LoopNode : public Node {
  // A LoopNode represents a loop at the fe.
 public:
  LoopNode() : Node(Loop_Node), expr(0), condition(0), body(0), tail(0),
  state(Expression) {}

  virtual ~LoopNode() {
    delete expr;
    delete condition;
    delete body;
  }

  virtual bool add_node(Node *n);
  virtual void eval_node();

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

  bool begin_body(VarNode *c);
  // This function is used to end the construction of the expression
  // and begin construction of the body of the loop.  c is the
  // condition for executing the loop body.

 private:
  MethodNode *expr;
  VarNode *condition;
  Node *body;

  enum {Expression, Body} state;
  Node *tail;
};

class IterNode : public Node {
  // An IterNode represents a Theta FOR construct, which loops over an
  // iterator.
 public:
  IterNode() : Node(Iter_Node), body(0), tail(0), state(Expression) {}

  virtual ~IterNode() {
    delete iterator;
    delete body;
  }

  virtual void eval_node();
  virtual bool add_node(Node *n);

  void end_body();

#if DEBUG_CONTROLS
  virtual void print_node();
#endif

 protected:
  MethodNode *iterator;
  Node *body;

  enum {Expression, Body} state;
  Node *tail;
};

inline void eval_tree(Node *t) {
  while (t) {
    t->eval_node();
    t = t->next;
  }
}

enum cell_method {GET, PUT, NONE};

MethodNode* create_specialized_method_node(
	VarNode* self, method m, int num_args, VarNode** args, int
	num_rets, int *results, type* ret_types, cell_method cm);

#if DEBUG_CONTROLS

void print_args_list(VarNode**, int);
void print_return_list(int*, int);
void print_tree(Node *t);

#endif

#endif /* _PARSE_TREE_H */
