/*  Copyright 1995 Barbara Liskov */

#include "runtime/obj.h"
#include "common/basic.h"
#include "types/string_class.h"
#include "server.h"
#include "runtime/alloc.h"
#include "common/th_assert.h"
#include "common/other_unix.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <netinet/tcp.h>
#include <stdarg.h>

#define INVALID_FD -1

static int rendezSock = INVALID_FD;
static fd_set readers, writers, excepts;
static bool recompute_bits = TRUE;
static int maxfds;
static int numfds;
static bool pipe_closed;
static struct timeval timeout = {0, 0};
static connection_dispFcn stdDisp;
static obj stdDispEnv;
static void ignore_close(connection c, obj env) {}
static struct closure closeDisp = { (ifunc)ignore_close, 0 };

struct connection_s {
    struct connection_private_s f;
    int inputfd, outputfd;
    char inbuf[BUFSIZ];
    char outbuf[BUFSIZ];
    char *remoteName;
    bool open;
    connection next;
    connection_dispFcn dispatch;
    obj dispatchEnv;
    struct sockaddr_in address;
    int lastInputTime;
};

string greeting = 0;

static connection theConnections = 0;

connection new_connection(int i, int o)
{
    connection c = THOR_MALLOC(struct connection_s);
    c->f.bytes_read = 0;
    c->f.msgsize = 0;
    c->inputfd = i;
    c->outputfd = o;
    c->f.in = fdopen(i, "r");
    c->f.out = fdopen(o, "w");
    setbuf(c->f.in, c->inbuf);
    setbuf(c->f.out, c->outbuf);
    c->open = TRUE;
    c->dispatch = stdDisp;
    c->dispatchEnv = stdDispEnv;
    return c;
}

static void connection_add(connection c)
{
    c->next = theConnections;
    theConnections = c;
}

static void connection_delete(connection c)
{
    connection c0 = theConnections;
    if (c0 == c)
	theConnections = c->next;
    else {
	while (c0 && c0->next != c) c0 = c0->next;
	if (c0) c0->next = c->next;
    }
}

static void catch_pipe(int dummy)
{
    signal(SIGPIPE, catch_pipe);
    pipe_closed = TRUE;
}

static void welcome(connection c)
{
    connection_fwrite(c, string_charp(greeting), string_length(greeting));
    connection_fflush(c);
}

void connection_resynchronize(connection c, char resync_char) {
#define INTMIN(a,b) ((a)<(b) ? (a) : (b))

  static char tmpbuf[BUFSIZ];
  int sz;
  char ch;

  if (c->f.bytes_read < c->f.msgsize) {
    /* Ignore the partially read current message */
    connection_fread(c, tmpbuf, c->f.msgsize - c->f.bytes_read);
  }
  /* Keep looping over messages till the desired one is found */
  while ( (ch = connection_fgetc(c)) != resync_char) {
    int total_ignore_bytes =  c->f.msgsize - c->f.bytes_read;
    for (sz = 0; sz < total_ignore_bytes; sz += BUFSIZ) {
      int ignore_bytes = INTMIN(total_ignore_bytes - sz, BUFSIZ);
      connection_fread(c, tmpbuf, ignore_bytes);
    }
  }
}

/* Return TRUE on failure to flush all output */
bool connection_fflush(connection c)
{
    if (EOF == fflush(c->f.out)) {
	perror("fflush");
	return TRUE;
    } else {
	return FALSE;
    }
}

void connection_close(connection c)
{
    if (!c->open) {
	exc = &exc_not_possible;
	return;
    }

    connection_putc(c, '=');
    connection_fflush(c);

    c->open = FALSE;
    {
	time_t t;
	if (c->inputfd != INVALID_FD) close(c->inputfd);
	if (c->outputfd != INVALID_FD) close(c->outputfd);
	time(&t);
	fprintf(stderr, "Connection to %s closed, %s",
	    c->remoteName,
	    ctime(&t));
	connection_delete(c);
	recompute_bits = TRUE;
	(*(connection_closeFcn)closeDisp.f)(c, closeDisp.env);
    }
}

/* Return 0 if the connection has been closed. */
size_t connection_fwrite(connection c, void const *buf, size_t size)
{
    size_t n = 0;
    pipe_closed = FALSE;
    while (n <= 0) {
      n = fwrite(buf, 1, size, c->f.out);
      if (n <= 0) {
	switch(errno) {
	case EINTR:
	  if (pipe_closed) {
	    fprintf(stderr, "Client closed connection.\n");
	    connection_close(c);
	    return 0;
	  }
	  break; /* try again */
	default:
	  perror("fwrite");
	  /* Quit at this point */
	  return 0;
	}
      }
    }
    return n;
}

static size_t connection_fread_private(connection c, void *buf, size_t size) {
  /* This function reads the data raw from the connection. It does not check
     to see whether the message has ended or not
  */
  size_t n = 0;
  pipe_closed = FALSE;
  while (n <=0) {
    n = fread(buf, 1, size, c->f.in);
    if (n <= 0) {
      switch(errno) {
      case EINTR:
	if (pipe_closed) {
	  fprintf(stderr, "Client closed connection.\n");
	  connection_close(c);
	  return 0;
	}
	break;			/* try again */
      default:
	perror("fread");
	/* Quit at this point */
	return 0;
      }
    }
  }
  return n;
}

size_t connection_fread(connection c, void *buf, size_t size)
{
  size_t n = 0;
  /* Check if current msg has finished and a new msg needs to be fetched */

  if (c->f.bytes_read > c->f.msgsize) {
    /* Only possible due to connection_getc; the difference must be 1 char
       Also increment the _cnt field of the the input file pointer */
    th_assert(c->f.bytes_read == c->f.msgsize+1,
	      "Bytes_read has beeen incremented excessively");
    c->f.in->_cnt++;
    c->f.bytes_read--;
  }
  
  if (c->f.bytes_read == c->f.msgsize) {
    connection_fread_private(c, &c->f.msgsize, sizeof(int));
    c->f.bytes_read = 0;
    /* For sanity check, when a new message is read, the fread must have been
       performed for the command character */
    th_assert(size == 1, "First read from new message is not for a command");
  }
  n = connection_fread_private(c, buf, size);
  c->f.bytes_read += n;
  th_assert(c->f.bytes_read <= c->f.msgsize, "Read spans across messages");
  return n;
}

void connection_fputc(connection c, char ch)
{
    char buf[1];
    buf[0] = ch;
    if (!connection_fwrite(c, buf, 1)) exc = &exc_not_possible;
}

void connection_fputi(connection c, int i)
{
    int buf[1];
    buf[0] = i;
    if (!connection_fwrite(c, buf, sizeof(int))) exc = &exc_not_possible;
}
char connection_fgetc(connection c)
{
    char buf[1];
    if (!connection_fread(c, buf, 1)) exc = &exc_not_possible;
    return buf[0];
}

int connection_fgeti(connection c)
{
    int buf[1];
    if (!connection_fread(c, buf, sizeof(int))) exc = &exc_not_possible;
    return buf[0];
}

void connection_put(connection c, string s)
{
    size_t n = connection_fwrite(c, string_charp(s), string_length(s));
    if (n != 0) return;
    exc = &exc_not_possible;
}

void connection_printf(connection c, char const *format, ...)
{
    va_list args;

    va_start(args, format);
    if (vfprintf(c->f.out, format, args) >= 0) return;
    else
        exc = &exc_not_possible;
    va_end(args);
}

static bool registered_root = FALSE;

void connection_setGreeting(string msg)
{
    if (!registered_root) {
        gc_register_root((obj *)&greeting);
        registered_root = TRUE;
    }
    greeting = msg;
}

bool connection_createSocket(int port)
{
    int *sock = &rendezSock;
    int opt = 1;
    struct sockaddr_in server;

    if (!greeting) greeting = string_new("Welcome.\n");
    
    signal(SIGPIPE, catch_pipe);
    if (*sock >= 0) {
	if (0>close(*sock)) {
	    perror("closing old socket");
	    return FALSE;
	}
    }
    *sock = socket(AF_INET, SOCK_STREAM, 0);
    if (*sock<0) {
	perror("create socket");
	return FALSE;
    }
    if (setsockopt (*sock, SOL_SOCKET, SO_REUSEADDR,
		    &opt, sizeof(opt)) < 0) {
	perror ("setsockopt");
	return FALSE;
    }


    server.sin_family = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;
    server.sin_port = htons(port);


    if (0 > bind(*sock, (struct sockaddr*) &server, sizeof(server))) {
	perror ("binding stream socket");
	return FALSE;
    }
    if (0 > listen(*sock, 5)) {
	perror ("listen");
	return FALSE;
    }
    maxfds = getdtablesize();

/* set up child death catching... */

    return TRUE;
}

int connection_socketPort(void) {
    int size;
    struct sockaddr_in server;

    size = sizeof(server);
    if (getsockname(rendezSock, (struct sockaddr*) &server, &size) < 0) {
	return -1;
    }

    return ntohs(server.sin_port);
}

static void receiveInput(connection c)
{
    /* This procedure assumes BUFSIZ is a power of 2 */
    
    struct timeval tv;
    struct timezone tz;
#if 0
    printf("Reading from %s\n", c->remoteName);
#endif
    if (c->dispatch) {
	(*c->dispatch)(c, c->dispatchEnv, c->f.in);
    } else {
	fprintf(c->f.out, "@ No dispatch function, input ignored\n");
	fflush(c->f.out);
    }
    gettimeofday(&tv,&tz);
    c->lastInputTime = tv.tv_sec;
    if (feof(c->f.in)) {
#if 0
	fprintf(stderr, "End of input from %s\n", c->remoteName);
#endif
	connection_close(c);
	return;
    }
}

static connection connection_create()
{
    struct sockaddr_in where;
    int where_size = sizeof(where);
    int fd;
    connection c;
    memset(&where, 0, sizeof(where));
    fd = accept(rendezSock, (struct sockaddr*) &where, &where_size);

    if (fd < 0) {
	perror("accept");
	return 0;
    }
    
    /* don't delay send to coalesce packets. */
    {
      int val = 1;
      if (0 > setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, 
			 (char*)&val, sizeof(val))) {
	perror ("setsockopt");
	return FALSE;
      }
    }

    c = new_connection(fd,fd);
    connection_add(c);
    c->address = where;
    {
	struct hostent *h = gethostbyaddr((char *)&where.sin_addr.s_addr,
					  where_size, AF_INET);
	if (h) {
	    c->remoteName = strdup(h->h_name);
	} else {
	    char buf[32];
	    long a = ntohl(where.sin_addr.s_addr);
	    sprintf(buf, "%d.%d.%d.%d",
		    (a>>24)&0xff, (a>>16)&0xff, (a>>8)&0xff, a&0xff);
	    c->remoteName = strdup(buf);
	}
    }
    {
	time_t t;
	struct timeval tv;
	struct timezone tz;
	time(&t);
	gettimeofday(&tv, &tz);
	c->lastInputTime = tv.tv_sec;
	fprintf(stderr, "New connection from %s, %s",
		c->remoteName,
		ctime(&t));
    }
    welcome(c);
    return c;
}

static bool outputPending(connection c)
{
#if defined(__osf__) || defined(ultrix)
    FILE *f = c->f.out;
    return (f->_ptr != f->_base);
#else
    th_fail("Don't know how to test for pending output.");
#endif
}

static void recomputeFDsets()
{
  if (recompute_bits) {
    connection c = theConnections;
    numfds = 0;
    FD_ZERO(&readers);
    FD_ZERO(&writers);
    FD_ZERO(&excepts);
    if (rendezSock >= 0) {
      FD_SET(rendezSock, &readers);
      numfds = rendezSock;
    }
    while (c) {
      if (c->inputfd != INVALID_FD) {
	FD_SET(c->inputfd, &readers);
	FD_SET(c->inputfd, &excepts);
      }
      if (c->inputfd != INVALID_FD &&
	  c->inputfd >= numfds)
	numfds = c->inputfd;
      if (c->outputfd != INVALID_FD) {
	FD_SET(c->outputfd, &writers);
	if (c->outputfd >= numfds)
	  numfds = c->outputfd;
      }
      c = c->next;
    }
    recompute_bits = FALSE;
    numfds = numfds + 1;
  }
}

static bool inputPending(connection c)
{
#if defined(__osf__) || defined(ultrix)
    return (c->f.in->_cnt > 0);
#else
    th_fail("Don't know how to test for pending input.");
#endif
}

serverResult connection_serviceRequests()
{
  fd_set readers_copy, writers_copy, exc_copy;
  struct timeval zero;
  serverResult state = all_idle;
  recomputeFDsets();
  assert(numfds <= maxfds);

  /* check for input to distribute */
  memcpy(&readers_copy, &readers, sizeof(readers));
  memcpy(&writers_copy, &writers, sizeof(writers));
  memcpy(&exc_copy, &excepts, sizeof(excepts));
  zero.tv_sec = 0;
  zero.tv_usec = 0;
  switch (select(numfds, 0, &writers_copy, 0, &zero)) {
    case -1:
      if (errno == EINTR) break;
      perror("select");
      exit(2);
    case 0:
      break;
    default:
    {
	connection co = theConnections;
	while (co) {
	    if (outputPending(co) &&
		FD_ISSET(co->outputfd, &writers_copy)) {
		    connection_fflush(co);
	    }
	    co = co->next;
	}
    }
  }
  memcpy(&writers_copy, &writers, sizeof(writers));
  switch (select(numfds, &readers_copy, 0, &exc_copy, &timeout)) {
    case -1:
      if (errno == EINTR) break;
      perror("select");
      exit(2);
    case 0:			/* timeout */
      break;
    default:
      {
	  if (FD_ISSET(rendezSock, &readers_copy))
	    {
		/*	    printf("Checking new connection\n");  */
		(void)connection_create();
		recompute_bits = TRUE;
	    }
	  loop {
	      bool someInput = FALSE;
	      connection co = theConnections;
	      while (co) {
		  if (co->inputfd != INVALID_FD &&
		      (inputPending(co) ||
		       FD_ISSET(co->inputfd, &readers_copy) ||
		       FD_ISSET(co->inputfd, &exc_copy))) {
		      receiveInput(co);
		      someInput = TRUE;
		      FD_CLR(co->inputfd, &readers_copy);
		      FD_CLR(co->inputfd, &exc_copy);
		  }
		  co = co->next;
	      }
	      state = keep_going;
	      if (!someInput) break;
	  }
      }
  }

  return state;
}

void connection_broadcast(string msg)
{
    connection c = theConnections;
    while (c) {
	connection_put(c, msg);
	connection_fflush(c);
	c = c->next;
    }
}

void connection_setTimeout(struct timeval t)
{
    timeout = t;
}

void connection_setStdDisp(struct closure cl)
{
    stdDisp = (connection_dispFcn)cl.f;
    stdDispEnv = cl.env;
}

void connection_setDisp(connection c, struct closure cl)
{
    c->dispatch = (connection_dispFcn)cl.f;
    c->dispatchEnv = cl.env;
}

void connection_setCloseDisp(struct closure cl) {
    closeDisp = cl;
}
