/*  Copyright 1995 Barbara Liskov */

#ifndef _SERVER_H
#define _SERVER_H

#ifdef __cplusplus
extern "C" {
#endif

#include "fe/runtime/obj.h"
#include "common/iter.h"
#include "common/basic.h"
#include <sys/time.h>
#include <stdio.h>

typedef struct connection_s *connection;

typedef enum { keep_going, all_idle } serverResult;
extern struct exception_s exc_not_possible;

typedef void (*connection_dispFcn)(connection, obj env, FILE *f);
typedef void (*connection_closeFcn)(connection, obj env);

/* Overview: A "connection" represents a connection to a client of
 * the FE. Input arriving from a connection is dispatched to a
 * dispatch function attached to the connection. The input may be
 * read from the stdio input stream "f", or through "connection_fgetc"
 * or "connection_fread".
 *    Output may be sent using "connection_fwrite", "connection_fputc",
 * or "connection_printf".
 *    All connections share a common timeout value, an initial
 * dispatch function, and an initial greeting message.
 *    New connections are created automatically when clients connect to
 * a rendezvous socket created through the "createSocket" call.
 * Connections persist until "connection_close" is called on them, or
 * the client closes the connection itself.
 *    The various connection routines will spit messages out
 * to standard error if things go wrong.
 */


/* In the new protocol, the message from the client contains the size and then
   the actual message. Furthermore, the FE should never try to perform a read
   operation that spans messages. The effect is undefined. The server
   abstraction hides the size field from the binary_interface.cc code
*/

extern serverResult connection_serviceRequests();
/* Check all clients for input or deliver outstanding output if
   possible. Return "all_idle" if all clients are idle for the
   current timeout period. Clients receiving input have their
   dispatch functions called.
*/

bool connection_createSocket(int port);
/*
   Try to create a socket to service rendezvous requests, allowing
   clients to connect to the FE.
*/

int connection_socketPort(void);
/*
    Return the port of the rendezvous socket.
*/
   
void connection_setTimeout(struct timeval t);
/* Set the current timeout period. */


void connection_setStdDisp(struct closure);
/* Set the dispatch function for new connections. The closure must
   contain a function of type "connection_dispFcn".
*/

void connection_setDisp(connection c, struct closure);
/* Set the dispatch function for "c". The closure must contain a
   function of type "connection_dispFcn".
*/

void connection_setCloseDisp(struct closure);
/* Set the dispatch function to call when a connection is closed by the
   client or by a call to "connection_close".
   The closure must contain a function of type "connection_closeFcn".
*/

void connection_close(connection c);
/* Close a connection. Signal "not_possible" if the connection has
   already been closed.
*/
			 
void connection_put(connection, string);
/* Send the characters in the string as output to the connection.
   If the connection has been closed, signal "not_possible".
   No pointers to the string object are stored by this procedure.
 */

void connection_fputc(connection, char c);
/* Send "c" to the connection output, or signal "not_possible"
   if closed. The macro "connection_putc" has the same semantics
   and is usually much faster.
 */

void connection_printf(connection c, char const *format, ...);
/* Send the arguments to the connection output as described by a
   printf style format string, or signal "not_possible" if closed.
*/

size_t connection_fwrite(connection c, void const *buf, size_t size);
/* Send size bytes from buf to the connection.  Return 0 if the
   connection has been closed or there is an error while writing
*/

size_t connection_fread(connection c, void *buf, size_t size);
/* Receive size bytes from the connection to buf.
   Return 0 if the connection has been closed  or there is an error
   while reading
*/

char connection_fgetc(connection c);
/* Read a character from "c", or signal "not_possible"
   if closed. The macro "connection_getc" has the same semantics and
   is usually much faster.
*/

void connection_fputi(connection, int i);
/* Send "c" to the connection output, or signal "not_possible"
   if closed.
   */

int connection_fgeti(connection c);
/* Read an integer from "c", or signal "not_possible"
   if closed. */

void connection_resynchronize(connection c, char resync_char);
/* Keep ignroing messages from the client till the first character of the
   message is "resync_char"
*/

bool connection_fflush(connection c);
/* Flush all outstanding output to the underlying output device. This
   routine should be called after "connection_put"s to make sure that
   the data goes out in a timely manner. This is analogous to stdio
   "fflush".

   Return TRUE if the attempt to flush output failed, usually because
   the connection has been closed.
*/

void connection_broadcast(string msg);
/* Send "msg" to every open connection.
 */

void connection_setGreeting(string msg);
/* Set the initial greeting message that is sent to new connections.
 */

/******************************* MACROS *********************************/

#if defined(__osf__) || defined(ultrix)

/*
   The type "struct connection_private_s" exists only for the use of
   some macros defined here. It may not be used by clients.
*/
struct connection_private_s {
    FILE *in, *out;
    int bytes_read, msgsize;
    /* bytes_read indicates the number of bytes from the current message that
       have been read. Size indicates the total size of current message.
       If bytes_read == size, then message has been completely read
    */
};

#define priv(c) ((struct connection_private_s *)(c))
/* "priv" is not for direct use. */

#define connection_getc(c) 					\
    ((--priv(c)->in->_cnt >= 0 && ++priv(c)->bytes_read <= priv(c)->msgsize) ?\
     (int) *(priv(c)->in)->_ptr++: connection_fgetc(c))
/* See "connection_fgetc". */

#define connection_putc(c,x) 						\
	((--priv(c)->out->_cnt < 0) ? (connection_fputc(c, x),0)	\
		  : (int) (*priv(c)->out->_ptr++ = (unsigned char) (x)))
/* See "connection_fputc". */

#endif /* defined(__osf__) || defined(ultrix) */

#ifdef __cplusplus
}
#endif

#endif /* _SERVER_H */
