// Copyright 1995 Barbara Liskov

#include "fe/client/shm_connect.h"

#include "common/basic.h"
#include "common/other_unix.h"
#include "common/th_assert.h"

#include <stdlib.h>
#include <unistd.h>

#include "config/vdefs/DEBUG_SIGS.h"
#include "config/vdefs/DEBUG_SHM.h"
#include "config/vdefs/USE_SCHED.h"

#if USE_SIGS
#include <signal.h>
#include <sys/wait.h>
#include <sys/time.h>

#include "config/vdefs/RENDEZVOUS_SIGNAL.h"
#endif

#if USE_SCHED
extern "C" {
#include <sched.h>
}
#endif

#ifndef NDEBUG
#include <stdio.h>
#endif

#if USE_SIGS

sigset_t mask_on = (0);
  
#define MASK_ON (&mask_on)
#define MASK_OFF (1 << RENDEZVOUS_SIGNAL)

struct timeval tv;

static void handleWakeup(int);

#elif !USE_MSEM && !USE_SCHED

extern "C" unsigned usleep(unsigned);
unsigned int sleeplen = 1;

#endif


ShmClient *ShmServer::asClient() const
{
    return (ShmClient *)this;
}

#define buf1 (buffer)
#define buf2 (&(buffer[SHMBUFSIZ]))
#define bufsize SHMBUFSIZ

void ShmServer::init_ptrs() {
    cptr1 = buf1;
    sptr1 = &buf1[bufsize];
    sptr2 = buf2;
    cptr2 = &buf2[bufsize];
    eor1 = sptr1;
    eow1 = &buf1[bufsize - 1];
    eor2 = cptr2;
    eow2 = &buf2[bufsize - 1];
#if SHM_SENTINELS
    int i;
    for (i = 0; i<bufsize_; i++) {
	buf1[i] = empty;
	buf2[i] = empty;
    }
#endif
}

ShmServer::ShmServer(unsigned bufsize_)
{
    assert(bufsize_ < SHMBUFSIZ);
    bufsize_ = SHMBUFSIZ; // Ignoring the value sent in
    
    init_ptrs();
    immediate_flag = FALSE;

#if USE_SIGS
    spid = 0;
    cpid = 0;

    cblock1 = FALSE;
    cblock2 = TRUE;
    sblock2 = FALSE;
    sblock1 = TRUE;
#endif
#if USE_MSEM
    if (!msem_init(&csema, MSEM_LOCKED))
      th_fail("could not init msemaphore");
    if (!msem_init(&ssema, MSEM_LOCKED))
      th_fail("could not init msemaphore");
#endif
}

#ifndef NDEBUG
void ShmServer::dump(char const *prefix, int bufnum, bool vals)
{
    char cbuffer[1000];
    switch(bufnum) {
      case 1:
        sprintf(cbuffer,
                "%s Buffer 1\n"
                "%s   read pointer at %d, write pointer at %d\n"
                "%s   read length = %d, write length = %d\n",
                prefix,
                prefix, sptr1 - buf1, cptr1 - buf1,
                prefix, can_get(), asClient()->can_put());
        break;
      case 2:
        sprintf(cbuffer,
                "%s Buffer 2\n"
                "%s  read pointer at %d, write pointer at %d\n"
                "%s  read length = %d, write length = %d\n",
                prefix,
                prefix, cptr2 - buf2, sptr2 - buf2,
                prefix, asClient()->can_get(), can_put());
        break;
    }
    ::write(2, cbuffer, strlen(cbuffer));
    if (vals) {
	int k = 0;
	for (k = 0; k<bufsize; k++) {
	    int val = ((bufnum == 1)?buf1:buf2)[k];
#if SHM_SENTINELS 
	    if (val != empty)
#endif
		fprintf(stderr, "%4d:%-6d ", k, val);
#if SHM_SENTINELS 
	    else
		fprintf(stderr, "%4d:EMPTY  ", k, val);
#endif
	    if (k % 5 == 4) putc('\n', stderr);
	}
	putc('\n', stderr);
    }
}

int const *ShmServer::peek(int bufnum, int *len)
{
    int const *ret;
    switch (bufnum) {
	case 1:
	    ret = sptr1;
	    if (cptr1 > sptr1)
		*len = cptr1 - sptr1;
	    else
		*len = (buf1 + bufsize) - ret;
	    break;
	case 2:
	    ret = cptr2;
	    if (sptr2 > cptr2)
		*len = sptr2 - cptr2;
	    else
		*len = (buf2 + bufsize) - ret;
	    break;
    }
    return ret;
}

#endif

inline void ShmServer::wait_for_read(int const *r, bool quit_on_flag)
{
    loop {
	if (quit_on_flag) {
	    if (immediate_flag == FALSE) break;
	} else {
	    if (r != cptr1) break;
	}
#if USE_SIGS || USE_MSEM
	sblock1 = TRUE;
#if USE_SIGS
	tv.tv_sec = 3600; tv.tv_usec = 0;
#endif
	wakeupClient();
	/* We're the server here, so we wake up the client. */
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Server going to sleep\n");
#endif
#if USE_SIGS
	if (tv.tv_sec) select(0,0,0,0,&tv);
#endif
#if USE_MSEM
	msem_lock(&ssema, 0);
#endif
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Server awoke\n");
#endif
#else /* !USE_MSEM && !USE_SIGS */
#if USE_SCHED
	sched_yield();
#else
	usleep(sleeplen);
#endif
#endif
    }
}

inline void ShmClient::wait_for_read(int const *r)
{
    loop {
	if (r != sptr2) break;
#if USE_SIGS || USE_MSEM
	/* We're the client here, so we wake up the server. */
	cblock2 = TRUE;
#if USE_SIGS
	tv.tv_sec = 3600; tv.tv_usec = 0;
#endif
	wakeupServer();
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Client going to sleep\n");
#endif
#if USE_SIGS
	if (tv.tv_sec) select(0,0,0,0,&tv);
#endif
#if USE_MSEM
	msem_lock(&csema, 0);
#endif
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Client awoke\n");
#endif
#else /* !USE_MSEM && !USE_SIGS */
#if USE_SCHED
	sched_yield();
#else
	usleep(sleeplen);
#endif
#endif
    }
}

void ShmServer::wait_for_immediate_flag_reset() {
    // Block until the buffers have been cleaned up.
    // Resynchronization is done after the exception has been sent. Else the
    // client may block waiting for some result

    wait_for_read(NULL, TRUE);
    th_assert(immediate_flag == FALSE, "Wait for read returned too early");
}

int ShmServer::full_get()
{
#define W cptr1
#define E eor1
#define R sptr1
    int const *r = R;
    if (r == &buf1[bufsize]) r = &buf1[0]; /* wrap before check */
    wait_for_read(r, FALSE);
    int x = *r++;
#if SHM_SENTINELS
    ((int *)r)[-1] = empty;
#endif
    int *w = W;
    /* must snapshot because of race condition */
    if (!w) abort();
    if (r <= w)
      E = w;
    else
      E = &buf1[bufsize];
    R = r;
    return x;
}
#undef R
#undef W
#undef E


int ShmClient::full_get()
{
#define W sptr2
#define E eor2
#define R cptr2
    int const *r = R;
    if (r == &buf2[bufsize]) r = &buf2[0]; /* wrap before check */
    wait_for_read(r);
    int x = *r++;
#if SHM_SENTINELS
    ((int *)r)[-1] = empty;
#endif
    int *w = W;
    /* must snapshot because of race condition */
    if (r <= w)
      E = w;
    else
      E = &buf2[bufsize];
    R = r;
    return x;
}
#undef R
#undef W
#undef E

inline void ShmClient::wait_for_write(int *w)
{
    loop {
	if (sptr1 != w) break;
#if USE_SIGS || USE_MSEM
	/* We're the client here, so we wake up the server. */
	cblock1 = TRUE;
#if USE_SIGS
	tv.tv_sec = 3600; tv.tv_usec = 0;
#endif
	wakeupServer();
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Client going to sleep\n");
#endif
#if USE_SIGS
	if (tv.tv_sec) select(0,0,0,0,&tv);
#endif
#if USE_MSEM
	msem_lock(&csema, 0);
#endif
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Client awoke\n");
#endif
#else /* !USE_MSEM && !USE_SIGS */
#if USE_SCHED
	sched_yield();
#else
	usleep(sleeplen);
#endif
#endif
    }
}

inline void ShmServer::wait_for_write(int *w)
{
    loop {
	if (cptr2 != w) break;
#if USE_SIGS || USE_MSEM
	/* We're the server here, so we wake up the client. */
	sblock2 = TRUE;
#if USE_SIGS
	tv.tv_sec = 3600; tv.tv_usec = 0;
#endif
	wakeupClient();
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Server going to sleep\n");
#endif
#if USE_SIGS
	if (tv.tv_sec) select(0,0,0,0,&tv);
#endif
#if USE_MSEM
	msem_lock(&ssema, 0);
#endif
#if !NDEBUG && DEBUG_SIGS
	fprintf(stderr, "Server awoke\n");
#endif
#else /* !USE_MSEM && !USE_SIGS */
#if USE_SCHED
	sched_yield();
#else
	usleep(sleeplen);
#endif
#endif
    }
}

void ShmClient::full_put(int x)
{
#define R sptr1
#define W cptr1
#define E eow1

    // If immediate_flag has been set, we perform "dummy" writes so that
    // the client does not block on a full send buffer (when this flag is set,
    // the server is not reading anything)
    if (immediate_flag) return;

    int *w = W;
#if SHM_SENTINELS && !defined(NDEBUG)
    if (*w != empty) { dump("OOPS cf", 1, TRUE); abort(); }
#endif
    *w++ = x; /* There is always a free spot under the write pointer */
    wait_for_write(w); /* must check before the wrap */
    if (w == &buf1[bufsize]) w = &buf1[0];
    int const *r = R; /* must snapshot because of race condition */
    if (w >= r) 
      E = &buf1[bufsize - 1];
    else
      E = r - 1;
    W = w;
}
#undef R
#undef W
#undef E

void ShmServer::full_put(int x)
#define R cptr2
#define W sptr2
#define E eow2
{
    int *w = check_ptr(W);
#if SHM_SENTINELS && !defined(NDEBUG)
    if (*w != empty) { dump("OOPS sf", 2, TRUE); abort(); }
#endif
    *w++ = x; /* There is always a free spot under the write pointer */
    wait_for_write(w); /* must check before the wrap */
    if (w == &buf2[bufsize]) w = &buf2[0];
    int const *r = R; /* must snapshot because of race condition */
    if (w >= r) 
      E = &buf2[bufsize - 1];
    else
      E = r - 1;
    W = w;
}
#undef R
#undef W
#undef E

void ShmServer::read(int *buf, int n)
{
#if !SHM_SENTINELS
    int len = can_get();
    if (n <= len) {
	int const *r = start_get(n);
	while (n--) *buf++ = raw_get(r++);
	finish_get(r);
    } else
#endif
	while (n--) *buf++ = get();
}

void ShmServer::write(int const *buf, int n)
{
#if !SHM_SENTINELS
    int len = can_put();
    if (n <= len) {
	int *w = start_put(n);
	while (n--) raw_put(w++, *buf++);
	finish_put(w);
    } else 
#endif
	while (n--) put(*buf++);
}

void ShmClient::read(int *buf, int n)
{
#if !SHM_SENTINELS
    int len = can_get();
    if (n <= len) {
	int const *r = start_get(n);
	while (n--) *buf++ = raw_get(r++);
	finish_get(r);
    } else
#endif
	while (n--) *buf++ = get();
}

void ShmClient::write(int const *buf, int n)
{
#if !SHM_SENTINELS
    int len = can_put();
    if (n <= len) {
	int *w = start_put(n);
	while (n--) raw_put(w++, *buf++);
	finish_put(w);
    } else
#endif
	while (n--) put(*buf++);
}

void ShmClient::resynchronize_buffers() {
    th_assert(immediate_flag, "Resynchnozation of buffers with flag FALSE");
    // Server is blocked, so there are no race conditions
    
    init_ptrs();     // Clear both buffers
    immediate_flag = FALSE;
}

bool ShmClient::looksOk()
{
    return ((sptr2 >= buf2) && (sptr2 < buf2 + bufsize) &&
        (cptr2 > buf2) && (cptr2 <= buf2 + bufsize) &&
        (sptr1 > buf1) && (sptr1 <= buf1 + bufsize) &&
        (cptr1 >= buf1) && (cptr1 < buf1 + bufsize));
}

#if USE_SCHED
void beFIFO()
{
    struct sched_param param;
    param.sched_priority = SCHED_PRIO_USER_MAX;
    sched_setscheduler(getpid(), SCHED_FIFO, &param);
}
#endif

void ShmServer::beServer()
{
#if USE_SIGS
    spid = getpid();

    struct sigaction action;
    action.sa_handler = handleWakeup;
    action.sa_mask = 0;
    action.sa_flags = 0;
    sigaction(RENDEZVOUS_SIGNAL, &action, 0);
#endif
#if USE_SCHED
    beFIFO();
#endif
}

void ShmClient::beClient()
{
#if USE_SIGS
    cpid = getpid();

    struct sigaction action;
    action.sa_handler = handleWakeup;
    action.sa_mask = 0;
    action.sa_flags = 0;
    sigaction(RENDEZVOUS_SIGNAL, &action, 0);
#endif
#if USE_SCHED
    beFIFO();
#endif
}

#if USE_SIGS
static void handleWakeup(int dummy)
{
    tv.tv_sec = 0;
    tv.tv_usec = 0;
}
#endif

void ShmServer::wakeupClient()
{
#if USE_SIGS || USE_MSEM
#if USE_SIGS
    if (!cpid) {
#if !NDEBUG && DEBUG_SIGS
        fprintf(stderr, "No client to wake up\n");
#endif
        return;
    }
#endif
    if (cblock1) {
        /*
           We are the server. The client is waiting for the server to
           read something, so cptr1 is stuck. Therefore, we can assume
           that both cptr1 and sptr1 are constant until we send the signal
           to the client.

           Note that the appropriate "block" flag must be cleared before
           the signal is sent, else there's a race condition. Once the
           signal is sent, all bets are off.
           
        */
        int *next = cptr1 + 1;
        if (next == buf1 + bufsize) next = buf1;
        if (next != sptr1) {
            cblock1 = FALSE;
#if !NDEBUG && DEBUG_SIGS
            fprintf(stderr, "Signalling client (%d)\n", cpid);
#endif
#if USE_SIGS
            kill(cpid, RENDEZVOUS_SIGNAL);
#endif
#if USE_MSEM
            msem_unlock(&csema, 0);
#endif
        }
    } else if (cblock2) {
        /* The client is waiting for the server to write something. */
        if (cptr2 != sptr2) {
            cblock2 = FALSE;
#if !NDEBUG && DEBUG_SIGS
            fprintf(stderr, "Signalling client (%d)\n", cpid);
#endif
#if USE_SIGS
            kill(cpid, RENDEZVOUS_SIGNAL);
#endif
#if USE_MSEM
            msem_unlock(&csema, 0);
#endif
        }
    }
#endif
}

void ShmClient::wakeupServer()
{
#if USE_SIGS || USE_MSEM
#if USE_SIGS
    if (!spid) {
#if !NDEBUG && DEBUG_SIGS
        fprintf(stderr, "No server to wake up\n");
#endif
        return;
    }
#endif
    if (sblock2) {
        /* See comments in "wakeupClient".*/
        
        int *next = sptr2 + 1;
        if (next == buf2 + bufsize) next = buf2;
        if (next != cptr2) {
            sblock2 = FALSE;
#if !NDEBUG && DEBUG_SIGS
            fprintf(stderr, "Signalling server (%d)\n", spid);
#endif
#if USE_SIGS
            kill(spid, RENDEZVOUS_SIGNAL);
#endif
#if USE_MSEM
            msem_unlock(&ssema, 0);
#endif
        }
    } else if (sblock1) {
        /* The server is waiting for the client to write something. */
        if (sptr1 != cptr1) {
            sblock1 = FALSE;
#if !NDEBUG && DEBUG_SIGS
            fprintf(stderr, "Signalling server (%d)\n", spid);
#endif
#if USE_SIGS
            kill(spid, RENDEZVOUS_SIGNAL);
#endif
#if USE_MSEM
            msem_unlock(&ssema, 0);
#endif
        }
    }
#endif
}

#if USE_MSEM
void ShmServer::remove_semaphores()
{
	msem_remove(&ssema);
	msem_remove(&csema);
}
#endif
