/*  Copyright 1995 Barbara Liskov */

#ifndef _SHM_CONNECT_H
#define _SHM_CONNECT_H

#include <sys/types.h>
#include <stdlib.h>
#include "common/basic.h"
#include "config/vdefs/NDEBUG.h"
#include "config/vdefs/USE_MSEM.h"
#include "config/vdefs/USE_SIGS.h"
#include "config/vdefs/USE_SCHED.h"
#include "config/vdefs/SHMBUFSIZ.h"
#include "config/vdefs/DEBUG_SHM.h"
#include "config/vdefs/SHM_SENTINELS.h"

#if USE_MSEM
extern "C" {
#include <sys/mman.h>
}
#endif

/*
   "ShmServer" is an object that lives in shared memory. Therefore, it
   cannot have any virtual methods. A "ShmServer" can also be viewed as
   a "ShmClient". The client and server are completely symmetrical; the
   names `client' and `server' are chosen in order to have names for
   the two sides.

   This implementation assumes that "int"'s are read and written from memory
   atomically.

   A "ShmServer" object is usually created using an explicit "in-place"
   constructor, where the constructor is invoked directly on a piece of
   memory to initialize. A "ShmClient" is produced using the 
   "asClient" member function.

   The server and client processes then invoke "beServer" and "beClient"
   on their respective ends, after which they can use the "get", "put",
   "read", and "write" methods with abandon.
*/

#if SHM_SENTINELS
int const empty = 0xa5a5a5a5;
#endif

class ShmClient;

class ShmServer {

  public:
  
    ShmServer(unsigned bufsize = SHMBUFSIZ);
        /* Create a "ShmServer" that uses input and output buffers
           containing "bufsize" items. "bufsize" must be in the range
           0 ... "SHMBUFSIZ".
	*/

    ShmClient *asClient() const;
        /* Obtain the "ShmClient" for this server */

    void write(int const buf[], int len);
        /* Write "len" items from "buf" */

    void read(int buf[], int len);
        /* Read "len" items into "buf" */

    int *check_ptr(int const *ptr) {
      /* Check whether the pointer is actually pointing into
	 the shared memory buffer. This is used to prevent the client
	 from making the server overwrite non-shared data. Return "ptr"
	 if it is okay, 0 if not.
	 */
	
	unsigned long df = ((unsigned long)(ptrdiff_t)
	    ((char *)ptr - (char *)&buffer[0])) < sizeof(buffer);
	if (df) return (int *) ptr;
	return 0;
// 	  return (((unsigned long)(ptrdiff_t)
// 	    ((char *)ptr - (char *)&buffer[0])) < sizeof(buffer))
// 	      ? (int *)ptr : 0;
    }

    void put(int x) {
        /* Write one item to the client, assuming that the system will
	   trap a write to address 0. */
	unsafe_put(check_ptr(sptr2), x);
    }

    int get() {
        /* Read one item from the client. */
	int const *ptr = sptr1;
        if (ptr != eor1) {
	  int x = *ptr;
#if SHM_SENTINELS
	  *(int *)ptr = empty;
#endif
          sptr1 = ptr + 1;
	  return x;
        } else
          return full_get();
    }

    int can_put() const { return eow2 - sptr2; }
        /* Return a number of items that can be written into the buffer
           without blocking. The actual number of such items is at least
           this large.*/

    int can_get() const { return eor1 - sptr1; }
        /* Return a number of items that can be written into the buffer
           without blocking. The actual number of such items is at least
           as large. */

    void beServer();
        /* Register this process's intent to be the server. The server
           may not be unblocked by "wakeupServer" unless this call is
           made. */

    void wakeupClient();
        /* Wake up the client if the client is blocked on a read or
           write, yet is now able to perform the operation. May be
	   called at any time to transfer control to the client. */

    void wait_for_immediate_flag_reset();
    // effects: Wait for the client to reset it to FALSE. The client will
    //          resynchronize the buffers also, i.e. clean them


#ifndef NDEBUG
    void dump(char const *prefix, int bufnum, bool dumpvals = FALSE);
        /* Dump a description of the state of the server in buffer
	   "bufnum". */
    
    int const *peek(int buffer, int *len);
	/* Provide a pointer into buffer "buffer" which contains at least
	   "*len" contiguous items to be read. The buffer may actually contain
	   more than "*len" items. */
#endif  

/* THE RAW INTERFACE: when you feel the need for speed. Real bad. And you
    want to put or get a bunch of items at once. Be careful. */

    int *start_put(int n) { return check_ptr(sptr2); }
    void raw_put(int *ptr, int x) { *ptr = x; }
    void finish_put(int *ptr) { sptr2 = ptr; }

/* How to use raw_put to write items val_0 ... val_n-1:

    int n;
    ...
    int len = s->can_put();
    if (n <= len) {                    // fast path
	int *ptr = s->start_put(n);
	s->raw_put(ptr, val0);
	s->raw_put(ptr+1, val1);
	...
	s->raw_put(ptr+n-1, valnm1);
	s->finish_put(ptr + n);        // notice increment by n here
    } else {                           // slow path
	s->put(val0);
	s->put(val1);
	...
	s->put(valnm1);
    }
*/

    int const *start_get(int n)
	{ return check_ptr(sptr1); }
    int raw_get(int const *ptr) { return *ptr; }
    void finish_get(int const *ptr) { sptr1 = ptr; }

/* How to use raw_get to read items val_0 ... val_n-1 :

    int n; // number of items we want to read
    ...
    int len = s->can_get();
    if (n <= len) {
	int *ptr = s->start_get(n);
	val0 = s->raw_get(ptr);
	val1 = s->raw_get(ptr+1);
	...
	valnm1 = s->raw_get(ptr+n-1, valnm1);
	s->finish_get(ptr + n);
    } else {
	val0 = s->get();
	val1 = s->get();
	...
	valnm1 = s->get();
    }
*/

/* The slow versions. Maybe useful if you _know_ it's going to be slow
   anyway. */
    void full_put(int x);
    int full_get();
    void set_immediate_flag() { immediate_flag = TRUE;}
    // effects: sets the immediate_flag to TRUE

#if USE_MSEM
    void remove_semaphores();
#endif

  protected:
  
    void unsafe_put(int *ptr, int x) {
    /* Write "x" into the buffer, assuming that "sptr2" cannot be
       trusted and "ptr" is its true value.
       I.e., don't bother to check for bad write pointers.
       Applications should almost always use put() instead.
       */
        if (ptr != eow2) {
#if SHM_SENTINELS && !defined(NDEBUG)
          if (*ptr != empty) { dump("OOPS s", 2, TRUE); abort(); }
#endif
	  *ptr = x;
	  sptr2 = ptr + 1;
        } else
          full_put(x);
    }   

    void wait_for_write(int *w);
    void wait_for_read(int const *r, bool quit_on_flag);
    // effects: Wait till there is data to read if quit_on_flag is FALSE
    //          If it is TRUE, then wait till immediate_flag is FALSE

    void init_ptrs();
    // effects: Initialize the pointers cptr[12], sptr[12], eo[rw][12]
    //          Also if SHM_SENTINELS is TRUE, initialize buffers

    int *cptr1, *sptr2;
    int const *sptr1, *cptr2;
    int const *eor1, *eor2;
    int const *eow1, *eow2;
    bool immediate_flag;
    // Set by the FE when it wants to sends an "immediate" exception 
    // to the FE

#if USE_SIGS
    pid_t cpid, spid;
#endif
#if USE_MSEM
    msemaphore csema, ssema;
#endif
    
    
#if USE_SIGS || USE_MSEM
    int cblock1, cblock2, sblock1, sblock2;
#endif
        /*
           boolean values to indicate whether the client or server are
           blocked waiting for various pointers to move. A "cblock"
           variable is only set by the client, and only reset by the
           server. "sblock" variables are symmetrical.
        */

    int buffer[SHMBUFSIZ * 2];
        /*
           The instance variable "buffer" comes last so that the whole
           structure fits onto a single page if "bufsize" is not too
           large. The two buffers are placed at 0 and SHMBUFBIZ in
	   this buffer.

       The picture for buffer 1 is the following (buffer 2 is the same,
       but client and server exchange places).  The "eo[rw][12]" point just
       past the end of the readable or writable part of the buffer.

\begin{verbatim}
    
         * = contains data
         - = empty

          |******-----------********|      case 1       
                 ^          ^       ^
               cptr1      sptr1    eor1
			  eow1

	  In case 1, eow1 lies between cptr1 and sptr1-1

          |------***********--------|      case 2
                 ^          ^       ^
                 sptr1      cptr1   eow1
		       eor1
	  In case 2, eor1 lies between sptr1 and cptr1    

\end{verbatim}

       Note that the buffer slot immediately preceding the read pointer
       in either buffer is always empty. This allows the case where the
       reader is waiting for the writer to be disambiguated from when
       the writer is waiting for the reader.

       If the reader is waiting for the writer, then the read and write
       pointers for the buffer point to the same place. If the writer is
       waiting for the reader, then the write pointer points to the location
       just before the read pointer.

       The "full_get" and "full_put" routines are called whenever
       the simple increment and load/store won't work.
    */
};

class ShmClient : private ShmServer {
    friend class ShmServer;
  public:
    bool is_immediate_flag() {return immediate_flag;};
    // effects: Return TRUE if the FE has sent an immediate exception to the
    //          client


    void resynchronize_buffers();
    // requires: immediate_flag is TRUE
    // effects: Clear the buffer in which the client sends data to the FE
    //          Clear the buffer in which the FE sends data to the client
    //          Clears the immediate_flag so that the FE can resume work
    
    void write(int const *buf, int len);
    void read(int *buf, int len);
    void put(int x) {
	int *ptr = cptr1;
        if (ptr != eow1) {
#if SHM_SENTINELS && !defined(NDEBUG)
          if (*ptr != empty) { dump("OOPS c", 1, TRUE); abort(); }
#endif
	  *ptr = x;
	  cptr1 = ptr + 1;
        } else
          full_put(x);
    }

    int get() {
	int const *ptr = cptr2;
        if (ptr != eor2) {
	  int x = *ptr;
#if SHM_SENTINELS
	  *(int *)ptr = empty;
#endif
	  cptr2 = ptr + 1;
	  return x;
        } else
          return full_get();
    }

    int can_put() { return eow1 - cptr1; }
        /* Return a number of items that can be written into the buffer
           without blocking. */
    int can_get() { return eor2 - cptr2; }
        /* Return a number of items that can be written into the buffer
           without blocking. */

    bool looksOk();
      /* provide a non-definitive check on whether the data structure
         seems to be properly situated in memory and obeys its
         constraints. Typically, this needs to be called from client
         side and is only used for debugging purposes.
         */
      
    void beClient();
        /* Register the intent of this process to be the client. The client
           will not be unblocked by "wakeupClient" unless this call is made. */
    
    void wakeupServer();
        /* See "ShmServer::wakeupClient". */
    
/* THE RAW INTERFACE. See comments in ShmServer. */
    int *start_put(int n) { return cptr1; }
    void raw_put(int *ptr, int x) { *ptr = x; }
    void finish_put(int *ptr) { cptr1 = ptr; }
    int const *start_get(int n) { return cptr2; }
    int raw_get(int const *ptr) { return *ptr; }
    void finish_get(int const *ptr) { cptr2 = ptr; }

/* The slow versions. Maybe useful if you _know_ it's going to be slow
   anyway. */
    void full_put(int x);
    int full_get();
  protected:
    void wait_for_write(int *w);
    void wait_for_read(int const *r);
    
  private:
    /* No member variables are allowed, because a client object is
       really the same as a server object. */
};

#if !USE_SIGS && !USE_MSEM && !USE_SCHED
extern unsigned sleeplen;
/* The amount of time to go to sleep for when blocked, in microseconds.
   Probably should not be changed. */
#endif

#endif
