// Copyright 1995 Barbara Liskov

#include "config/vdefs/DEBUG_CONTROLS.h"
#include "type_check.h"
#include "handle.h"
#include "future.h"
#include "runtime/except.h"
#include "runtime/disphdr.h"
#include "fe_config.h"

implementOpenHashMap(FutureTypeTable, int, type_table_val*, hash_int, comp_int)

static FutureTypeTable *tt;

void init_type_table(int size) {
// Create and initialize a new type table.
// Must be called before any operations are performed on the table!
  tt = new FutureTypeTable(size);
}

void destroy_type_table() {
// Free up storage used by type table.
  FutureTypeTable::Bindings *map = new FutureTypeTable::Bindings(tt);

  while (map->ok()) {
    delete map->val();            // Delete the type_table_val struct.
    map->next();
  }
  delete tt;
}

void signal_type_error() {

//#if DEBUG_CONTROLS
  fprintf(stderr, "Signalling type error!\n");
//#endif

  SET_EXC(exc_type_error);  // XXX Need to fix up all result futures!
}

type find_type(int f, obj_ref &r) {
  // Find the type of the object referred to by f.

  handle_val tmp;
  type res = 0;
  type_table_val* ttv;

  if (f >= 0) {

#if DEBUG_CONTROLS
    fprintf(stderr, "find_type: Illegal future: %d\n", f);
#endif

    signal_type_error();
    return res;
  }

  bool found = tt->fetch(-f, ttv);  // Find the entry in the type table.

  if (!found) {
    // If f is not found in the type_table, find the object in the
    // future table.  If it\'s not there, then signal type error.

    handle_val tmp = FEConf->ft->future_to_any(f);
    CATCH_EXC(exc_not_possible) {

#if DEBUG_CONTROLS
      fprintf(stderr, "find_type: Argument type not found: %d\n", f);
#endif

      signal_type_error();
      return res;
    }
#ifdef __GNUC__
    res = handle_val_get_type(tmp);
#else
    res = tmp.get_type();
#endif

    // Enter a mapping from f to curr_type in the type_table to
    // simplify future look_ups.  Allocate a new obj_ref for f and
    // initialize it with the object in tmp.

    ttv = new type_table_val();

    ttv->t = res;
    ttv->r = any_as_obj(tmp.val);
    
    tt->store(-f, ttv);
  }

  r = &(ttv->r);
  return ttv->t;
}

int type_check(type at, type et) {
  // Make sure that the apparant type (at) is a subtype of the expected
  // type (et).
  //
  // Effects: returns the appropriate offset to be added to the object
  //          of type at for it to maskerade as an et.

#if DEBUG_CONTROLS
  fprintf(stderr, "Type checking.  Apparant type: %s, Expected type: %s\n",
	  char_type(at), char_type(et));
#endif

  if (at == et)                // The simple case.
    return 0;
    
  int hdroff = hdr_offset(type_as_objtype(at), type_as_objtype(et));
  CATCH {

#if DEBUG_CONTROLS
    fprintf(stderr, "Type error: Argument not a subtype of expected type\n");
    fprintf(stderr, "apparant type: %s, expected type: %s\n", 
	    char_type(at), char_type(et));
#endif
    
    signal_type_error();
  }
  return 0;
}


void enter_result(int f, type t) {
  
#if DEBUG_CONTROLS
  fprintf(stderr, "Type entering future: %d, type: %s\n", f, char_type(t));
#endif

  // Since we are using assign once semantics, we allow the type of h
  // to be determined once and only once!

  if (tt->contains(-f)) {

#if DEBUG_CONTROLS
  fprintf(stderr, "Type mapping already exists for future %d\n", f);
#endif

    signal_type_error();
    return;
  }
  else {
    type_table_val *ttv;

    ttv = new type_table_val();

    ttv->t = t;
    ttv->r = 0;
    
    tt->store(-f, ttv);
    return;
  }
}

#if DEBUG_CONTROLS
void print_type_table() {

  FutureTypeTable::Bindings *map = new FutureTypeTable::Bindings(tt);

  while (map->ok()) {
    fprintf(stderr, "future: %d, type: %s\n", -map->key(),
	    char_type((type)map->val()->t));
    map->next();
  }
}
#endif
