/*  Copyright 1995 Barbara Liskov */

#ifndef _TYPE_CHECK_H
#define _TYPE_CHECK_H

#include "runtime/obj.h"
#include "common/hashfuncs.h"
#include "common/openhashmap.h"
#include "config/vdefs/DEBUG_CONTROLS.h"

// The type table maps futures to types.  The type_table keeps track
// of what type was assigned to a future when it was first returned by
// a method.  This is used to verify that the future has the correct
// type when it is later used as an argument or receiver.

// The type table also maps futures to object references.  This
// is used to pass the results of previous calls as arguments to
// subsequent ones.  This is done to by-pass the future table during
// the excecution of a Batched Control Structures because future table
// lookups are more expensive and unecessary in this case.

#if DEBUG_CONTROLS
#include "stdio.h"
#include "types/type.h"
#include "types/string_class.h"

inline const char* char_type(type t) {
  return string_charp(type_name(t));
}
#endif

typedef obj* obj_ref;

struct type_table_val {
  type t;
  obj r;
};

// Declare the hash table implementing the type table.
declareOpenHashMap(FutureTypeTable, int, type_table_val*, hash_int, comp_int)

void init_type_table(int size);
// Create and initialize a new type table of size.
// Must be called before any operations are performed on the table!

void destroy_type_table();
// Free up storage used by type table.

type find_type(int f, obj_ref &or);
// Determine the type of future f.  First look for it in type_table.
// If it\'s not there, look in the future table.  If it\'s not
// there, signal a type_error.
//
// Requires: f is a future (f < 0).
//
// Modifies: or, type_table.
//
// Effects: If it finds the type in the future table, it creates a new
//          entry for it in the type_table.  The r field in that entry will
//          point to the object it found in the future table.
//          If succesful, or will point to r.

int type_check(type at, type et);
// Make sure that the apparant type (at) is a subtype of the expected
// type (et).
//
// Effects: returns the appropriate offset to be added to the object
//          of type at for it to maskerade as an et.

void enter_result(int f, type t);
// Generate a new reference for the object that f will point to (r).
// Create a new mapping between f and (t, r).  If a mapping already exists
// for f, signal type_error.

#if DEBUG_CONTROLS
void print_type_table();
#endif

#endif /* _TYPE_CHECK_H */
