/* Copyright 1995 Barbara Liskov */

%{
#include "my_string.h"
#include "theta.tab.h"
#include <ctype.h>

char cmp_file_input()
{
	return input();
	}

void cmp_file_unput(char c)
{
	unput(c);
	}

#undef input
#undef unput
#undef output

#define output(c)
#define input() cmp_lex_input()
#define unput(c) cmp_lex_unput(c)

#define MAX_STRING 16384
#define DEBUG_FLAG 0

int lexstr(char *s);
int lexchar(char *c);
int  lexhex(char *s);
char *slower(char *s);

extern void yyerror();

void yydocomment();

#define text ((char *)yytext)

%}

special [_]
S [ \t\r]
letter [A-Za-z]
decimal [0-9]
octal [0-7]
uptohex [0-9a-fA-F]
decint {decimal}+
id (({letter}|{special})({letter}|{decimal}|{special})*)
charac (\'([^\'\n\\]|{bscombos})\')
string (\"([^\"\n\\]|{bscombos})*\")
bscombos (\\\\|\\[ntrfbv]|\\\"|\\\'|\\{octal}{octal}{octal})
uptohexnum ({decint}_{uptohexstuff})
uptohexstuff ({uptohex}+)
real (({decint}\.{decint}({exp}?))|(\.{decint}({exp}?))|({decint}{exp})|({decint}\.{exp}))
unreal ({decint}\./[ \t\r\n~*+\-\/<=>,.()[\]&|])
exp ([Ee][+-]?{decint})
       
comment %[^\n]*
handle (#{intstr})
dot (\.)

%%

{S}+            ;
{comment}       ;
"&"		{ return CAND; }
"|"		{ return COR; }
"("		{ return LPAREN; }
")"		{ return RPAREN; }
"{"		{ return LCURLY; }
"}"		{ return RCURLY; }
"["		{ return LBRACK; }
"]"		{ return RBRACK; }
"~"		{ return TILDE; }
"*"		{ return STAR; }
"**"		{ return EXP; }
"+"		{ return PLUS; }
"-"		{ return MINUS; }
"^"		{ return CARET; }
"\."		{ return DOT; }
"\.\.\."		{ return DOTS; }
","		{ return COMMA; }
"/"		{ return DIV; }
"//"		{ return MOD; }
"||"		{ return CONCAT; }
":"		{ return COLON; }
":="		{ return ASSIGN; }
"<"		{ return LT; }
"<="		{ return LE; }
"="		{ return EQ; }
"~="		{ return NEQ; }
">"		{ return GT; }
">="		{ return GE; }
[bB][eE][gG][iI][nN]		{ return BEGIN_; }
[bB][iI][nN][dD]		{ return BIND; }
[bB][rR][eE][aA][kK]		{ return BREAK; }
[cC][lL][aA][sS][sS]		{ return CLASS; }
[cC][oO][nN][sS][tT]"_"[iI][vV][aA][rR][sS]		{ return CONST_IVARS; }
[cC][oO][nN][tT][iI][nN][uU][eE]		{ return CONTINUE; }
[dD][oO]		{ return DO; }
[eE][lL][sS][eE]		{ return ELSE; }
[eE][lL][sS][eE][iI][fF]		{ return ELSEIF; }
[eE][nN][dD]		{ return END; }
[eE][xX][cC][eE][pP][tT]		{ return EXCEPT; }
[eE][xX][iI][tT]		{ return EXIT; }
[fF][aA][lL][sS][eE]		{ return FALSE_; }
[fF][oO][rR]		{ return FOR; }
[hH][aA][sS]		{ return HAS; }
[hH][iI][dD][eE][sS]		{ return HIDES; }
[iI][fF]		{ return IF; }
[iI][mM][pP][lL][eE][mM][eE][nN][tT][sS]		{ return IMPLEMENTS; }
[iI][nN]		{ return IN; }
[iI][nN][hH][eE][rR][iI][tT][sS]		{ return INHERITS; }
[iI][tT][eE][rR]		{ return ITER; }
[mM][aA][kK][eE]		{ return MAKE; }
[mM][aA][kK][eE][sS]		{ return MAKES; }
[mM][oO][dD][uU][lL][eE]		{ return MODULE; }
[nN][eE][wW]"_"[oO][bB][jJ][eE][cC][tT]		{ return NEWOBJECT; }
[nN][iI][lL]		{ return NIL; }
[oO][nN][eE][oO][fF]		{ return ONEOF; }
[oO][pP]		{ return OP; }
[oO][tT][hH][eE][rR][sS]		{ return OTHERS; }
[pP][rR][oO][cC]		{ return PROC; }
[pP][rR][oO][vV][iI][dD][eE][sS]		{ return PROVIDES; }
[rR][eE][cC][oO][rR][dD]		{ return RECORD; }
[rR][eE][sS][iI][gG][nN][aA][lL]		{ return RESIGNAL; }
[rR][eE][tT][uU][rR][nN]		{ return RETURN; }
[rR][eE][tT][uU][rR][nN][sS]		{ return RETURNS; }
[sS][eE][lL][fF]		{ return SELF; }
[sS][iI][gG][nN][aA][lL]		{ return SIGNAL; }
[sS][iI][gG][nN][aA][lL][sS]		{ return SIGNALS; }
[sS][tT][rR][uU][cC][tT]		{ return STRUCT; }
[tT][aA][gG][cC][aA][sS][eE]		{ return TAGCASE; }
[tT][hH][eE][nN]		{ return THEN; }
[tT][rR][uU][eE]		{ return TRUE_; }
[tT][yY][pP][eE]		{ return TYPE; }
[tT][yY][pP][eE][cC][aA][sS][eE]		{ return TYPECASE; }
[vV][aA][rR][iI][aA][nN][tT]		{ return VARIANT; }
[wW][hH][eE][nN]		{ return WHEN; }
[wW][hH][eE][rR][eE]		{ return WHERE; }
[wW][hH][iI][lL][eE]		{ return WHILE; }
[yY][iI][eE][lL][dD]		{ return YIELD; }
[yY][iI][eE][lL][dD][sS]		{ return YIELDS; }
{id}            { yylval.s = string_new(slower(text));
		  return ID; }
{decint}        { yylval.i = atoi(text);
		  return INT; }
{real}          { yylval.r = (real)atof(text); return REAL; }
{unreal}        { yylval.r = (real)atof(text); return REAL; }
{uptohexnum}	{ return(lexhex(text)); }
{string}	{ return(lexstr(text)); }
{charac}	{ return(lexchar(text+1)); }
.		{ return LexErr; }

%%

int is_octal(char c)
{
    return (c>='0' && c<='7')?TRUE:FALSE;
}

char *slower(char *s)
{
char *c;
	for (c = s; *c; c++) {
		/* _tolower appears to be broken on osf 1.2 7/15/94 dwc */
		*c = tolower(*c);
		}
	return s;
	}

/* 1/27/95 dcurtis; added mapping for \b, \v; removed \f mapping;
		    added \p (to \f) mapping;
		    fixed 3 digit octal computation
*/

char dochar(char *s, int *size, int *err)
{
    *size = 1;
    *err = 0;
    if (!s[0]) {
	*err = 1;
	return '?';
	}
    if (s[0] != '\\') {
	return s[0];
	}
    else {
	*size = 2;
	switch (s[1]) {
		case 0: {
			*err = 1;
			return '?';
			}
		case 'n': return '\n';
		case 'r': return '\r';
		case 'f': return '\f';
		case 't': return '\t';
		case 'v': return '\v';
		case 'b': return '\b';
		case '"': return '\"';
		case '\'': return '\'';
		case '\\': return '\\';
		case '0': case '1': case '2': case '3':
		case '4': case '5': case '6': case '7': {
		    *size = 4;
		    if (is_octal(s[2]) & is_octal(s[3])) {
			int temp = (64 * (s[1] - '0') +
				       8 * (s[2] - '0') +
					   (s[3] - '0')) & 0xff;
			return temp;
			}
		    else {
			*err = 1;
			return '?';
			}
		    }
			
		default: {
			*err = 1;
			return '?';
			}
	}
    }
}

int lexchar(char *s)
{
    int temp;
    int err;
    yylval.c = dochar(s, &temp, &err);
    if (err) {
	yyerror("Illegal character");
	return LexErr;
	}
    return CHAR;
}

char lexedstr[MAX_STRING];
int lexstr(char *s)
{
    char *cp = lexedstr;
    int err;
    s++;
    for (;;) {
	char c = *s;
	int skip, cc;
	if (c == '"') { 
    		int len = cp-lexedstr;
		*cp = 0; 
		yylval.s = string_newn(lexedstr, len); 
		return STRING;
		}
	cc = dochar(s, &skip, &err);
	if (err) {
		yyerror("Illegal character in string)");
		return LexErr;
		}
	*cp++ = (char)cc;
	s += skip;
    }
}

int lexhex(char *num)
{
  char *val, *base, *check;
  int res, b;

  val = base = num;
  while (*val != '_') val++;
  *val = 0; val++;
  if (DEBUG_FLAG)
	fprintf(stderr,"Base = %s, Value = %s\n", base, val);
  b = atoi(base);
  if ((b > 16) || (b < 2))
	{yyerror("Illegal integer literal (unrecognized base value)");
	 return LexErr;}
  if (((res = strtol(val, &check, b)) == 0) && (val != "0"))
	{yyerror("Illegal integer literal");
	 return LexErr;}
  if (*check != 0)
	{yyerror("Illegal integer literal");
	 return LexErr;}
  yylval.i = res;
  return INT;
}

void yydocomment()
{
   char c;
   while (1) {
      c = yyinput();
      if (c == '\n' || c == 0) {
#ifdef FLEX_SCANNER
        yylineno++;
#endif
	return;
      }
   }
}

extern void cmp_err(char *msg, int lineno);

void yyerror(char *msg)
{
    /* fprintf(stderr, "line %d: %s\n", yylineno, msg); */
    cmp_err(msg, yylineno);
}

#ifndef yywrap

/* Keep new flex happy */
int yywrap() {
    return 1;
}

#endif
