// Copyright 1995 Barbara Liskov

#include "parse.h"
#include "types/class_class.h"
#include "types/any.h"
#include "types/vec.h"
#include "types/ptype.h"
#include "type_chk.h"
#include "types/type.h"
#include "types/type_def.h"
#include "types/class.h"
#include "types/objtype_class.h"
#include "types/method.h"
#include "types/vec_instns.h"
#include "types/param.h"
#include "types/param_class.h"
#include "cg.h"

void remove_equate(Equate *eq, TypeCheckObj *tco);
void process_equate(Equate *eq, TypeCheckObj *tco);
void remove_teq(TypeEquate *teq);
void resolve_teq(TypeCheckObj *tco);
void save_teq(TypeEquate *teq);
void cmp_err_check_point();
void cmp_err_return_to_checkpoint();


#define DEBUG 0

void handle_equates(ParseNode *pn, bool add, TypeCheckObj *tco)
{
    switch (pn->tag()) {
	case ParseNode::BodyT: {
		Body *b = (Body *)pn;
		ParseNodeList *eqs = b->get_equates();
		if (add) process_equates(eqs, tco);
		else remove_equates(eqs, tco);
		break;
		}
	case ParseNode::ModuleT: {
	    Module *m = (Module *)pn;
	    switch (m->tag()) {
	    case Module::SpecModuleT: {
		SpecModule *sm = (SpecModule *)pn;
		ParseNodeList *specs = sm->get_specs();
		if (add) {
			process_equates(specs, tco);
			}
		else {
			remove_equates(specs, tco);
			}
		break;
		}
	case Module::ImplModuleT: {
		ImplModule *im = (ImplModule *)pn;
		ParseNodeList *eqs = im->get_equates();
		ParseNodeList *impls = im->get_impls();
		if (add) {
			process_equates(eqs, tco);
			process_equates(impls, tco);
			}
		else {
			remove_equates(eqs, tco);
			remove_equates(impls, tco);
			}
		break;
		}
		}
		break;
		}
	case ParseNode::EquateT: {
	    Equate *eq = (Equate *)pn;
		if (add) {
			process_equate(eq, tco);
			}
		else {
			remove_equate(eq, tco);
			}
		break;
		}
	case ParseNode::ImplEltT: {
	    ImplElt *ie = (ImplElt*)pn;
	    switch (ie->tag()) {
	    case ImplElt::ClassDefT: {
		ClassDef *cd = (ClassDef *)pn;
		ParseNodeList *eqs = cd->get_equates();
		ParseNodeList *elts = cd->get_classElts();
		if (add) {
			process_equates(eqs, tco);
			process_equates(elts, tco);
			}
		else {
			remove_equates(eqs, tco);
			remove_equates(elts, tco);
			}
		break;
		}
	    }
	    break;
	    }
	default: {
		sprintf(cmp_err_buf, "Time to fix handle equates");
		cmp_err(cmp_err_buf, pn->get_line());
		if (DEBUG) pn->print(2);
		}
	    }
	}


void process_equate(Equate *eq, TypeCheckObj *tco)
{
Environment *e = tco->get_env();

	switch (eq->tag()) {
	    case Equate::TypeEquateT: {
		TypeEquate *teq = (TypeEquate *)eq;
		string nm = teq->get_id()->get_id();
		bool exists = FALSE;
		if (e->look_up(nm)) exists = TRUE;
		TypeSpec *ts = teq->get_typ();
		set_node_type(ts, ts, tco, FALSE);
		TypeInterface *ti = ts->get_type();
		if (ti->tag() == TypeInterface::NullT) {
			save_teq(teq);
			return;
			}
		if (exists && !tco->allow_redefs) {
			sprintf(cmp_err_buf, "Redefinition of %.50s occurred",
				string_charp(nm));
			cmp_err(cmp_err_buf, ts->get_line());
			tco->redef_count++;
			}
		else {
			if (!exists) {
				e = e->add_type_binding(nm, ti);
    				tco->set_env(e);
				resolve_teq(tco);
				}
			}
		break;
		}
	    case Equate::ExprEquateT: {
		ExprEquate *ceq = (ExprEquate *)eq;
		Expr *ex = ceq->get_expr();
		string nm = ceq->get_id()->get_id();
		switch (ex->tag()) {
		    case Expr::LiteralT: {
			e = e->add_var_binding(nm, ex->get_type());
    			tco->set_env(e);
			break;
			}
		    default: {
			cmp_err("Expression too complicated for Theta v0",
				ex->get_line());
			}
		    }
		break;
		}
	    }
   }

void process_equates(ParseNodeList *l, TypeCheckObj *tco)
{
Environment *e = tco->get_env();

    if (l) for (Pix p = l->first(); p; l->next(p)) {
	ParseNode *pn = (ParseNode *)(*l)(p);
	switch (pn->tag()) {
		case ParseNode::EquateT: {
			Equate *eq = (Equate *)pn;
			process_equate(eq, tco);
			break;
			}
		case ParseNode::SpecEltT: {
			SpecElt *se = (SpecElt*) pn;
			if (se->tag() == SpecElt::SpecEquateT) {
				SpecEquate *speceq = (SpecEquate *)se;
				Equate *eq = speceq->get_equate();
				process_equate(eq, tco);
				}
			break;
			}
		}
      	}
    }

#define MAX_TEQS 100
TypeEquate *saved_teqs[MAX_TEQS];
int nteqs = 0;

void save_teq(TypeEquate *teq)
{
   if (nteqs >= MAX_TEQS) {
	th_fail("type_util.cc: MAX_TEQS exceeded");
	}
   saved_teqs[nteqs] = teq;
   nteqs++;
   }

void resolve_teq(TypeCheckObj *tco)
{
TypeEquate *teq;
bool keep_going = TRUE;

 // prepare to ignore any errors during this processing
 cmp_err_check_point();
 while (keep_going) {
   keep_going = FALSE;
   for (int i = 0; i < nteqs; i++) {
	teq = saved_teqs[i];
	TypeSpec *ts = teq->get_typ();
	set_node_type(ts, ts, tco, FALSE);
	TypeInterface *ti = ts->get_type();
	if (ti->tag() != TypeInterface::NullT) {
		string nm = teq->get_id()->get_id();
		Environment *e = tco->get_env();
		e = e->add_type_binding(nm, ti);
    		tco->set_env(e);
		keep_going = TRUE;
		remove_teq(teq);
		break;
		}
	}
   }
 cmp_err_return_to_checkpoint();
 }

void remove_teq(TypeEquate *teq)
{
   for (int i = nteqs-1; i >= 0; i--) {
	if (saved_teqs[i] == teq) {
		saved_teqs[i] = saved_teqs[nteqs-1];
		nteqs--;
		return;
		}
	}
   }

void remove_equate(Equate *eq, TypeCheckObj *tco)
{
Environment *e = tco->get_env();

	switch (eq->tag()) {
	    case Equate::TypeEquateT: {
		TypeEquate *teq = (TypeEquate *)eq;
		string nm = teq->get_id()->get_id();
		e = e->remove_binding(nm);
    		tco->set_env(e);
		remove_teq(teq);
		break;
		}
	    case Equate::ExprEquateT: {
		ExprEquate *ceq = (ExprEquate *)eq;
		Expr *ex = ceq->get_expr();
		string nm = ceq->get_id()->get_id();
		switch (ex->tag()) {
		    case Expr::LiteralT: {
			e = e->remove_binding(nm);
    			tco->set_env(e);
			}
		    }
		break;
		}
	    }
    }

void remove_equates(ParseNodeList *l, TypeCheckObj *tco)
{
    if (l) for (Pix p = l->first(); p; l->next(p)) {
	ParseNode *pn = (ParseNode *)(*l)(p);
	switch (pn->tag()) {
		case ParseNode::EquateT: {
			Equate *eq = (Equate *)pn;
			remove_equate(eq, tco);
			break;
			}
		case ParseNode::SpecEltT: {
			SpecElt *se = (SpecElt*) pn;
			if (se->tag() == SpecElt::SpecEquateT) {
				SpecEquate *speceq = (SpecEquate *)se;
				Equate *eq = speceq->get_equate();
				remove_equate(eq, tco);
				}
			break;
			}
		}
      	}
   }

void instantiate_tagged(TaggedTypeSpec *ts, string full_nm, int line, 
							TypeCheckObj *tco)
{
bool one = FALSE;
bool rec = FALSE;
bool str = FALSE;
bool var = FALSE;

	Environment *e = tco->get_env();
	string nm = ts->get_name()->get_name()->get_id();
	// must be oneof,record,?variant,struct,?maybe
	one = string_equal(nm, string_const("oneof"));
	rec = string_equal(nm, string_const("record"));
	if (!one && !rec && !str && !var) {
		sprintf(cmp_err_buf,
			"Tagged Type %s not implemented yet", nm->chars);
		cmp_err(cmp_err_buf, ts->get_line());
		return;
		}
	// while env maintains visible symbols and nested scope stuff
	// 	instns caches instantiations to make equal work...
	Environment *instns = tco->get_instns();
	NameBinding *bind = instns->look_up(full_nm);
	if (bind) {
		// put it back into the environment
		TypeInterface *ti = bind->get_type();
		e = e->add_type_binding(full_nm, ti);
		// only works for records...
		if (rec) {
			string mnm = string_concat(full_nm, 
					string_const("_MAKE"));
			bind = instns->look_up(mnm);
			if (bind) e = e->add_type_binding(mnm, ti);
			}
		tco->set_env(e);
		return;
		}
	objtype nt = new_objtype();
	nt->name = full_nm;
	// probably should check for redefinition...
	// do methods
	ParseNodeList *fields = ts->get_fields();
	int nfields = count_lhs(fields);
	int nmeth = nfields * 2 + 3;
	vec v = make_vec_Method(nmeth, FALSE);
	nt->methods_ = v;
	type t = objtype_as_type(nt);
	method m = new_copy_method(t);
        vec_store(v, 0, PV(m));
	m = new_equal_method(t);
        vec_store(v, 1, PV(m));
	m = new_similar_method(t);
        vec_store(v, 2, PV(m));
	int i = 3;
	int tag = 0;
	for (Pix p = fields->first(); p ; fields->next(p)) {
		Field *f = (Field *)(*fields)(p);
		TypeSpec *typ = f->get_typeSpec();
		set_node_type(typ, typ, tco, TRUE);
		ParseNodeList *ids = f->get_ids();
		for (Pix p2 = ids->first(); p2 ; ids->next(p2)) {
			Id *id = (Id *)(*ids)(p2);
			if (one) m = new_is_method(t, id->get_id(), tag);
			if (rec) m = new_get_method(t, id->get_id(), tag, typ);
        		vec_store(v, i, PV(m));
			i++;
			if (one) m = new_value_method(t, id->get_id(), tag, typ);
			if (rec) m = new_set_method(t, id->get_id(), tag, typ);
        		vec_store(v, i, PV(m));
			i++; tag++;
			if (one) {
				m = new_one_var_make_method(t, id->get_id(),
					tag, typ);
				// stick it in the environment
				TypeInterface *ti = new TypeInterface(m, 0);
				e = e->add_var_binding(m->name, ti);
				instns = instns->add_var_binding(m->name, ti);
				}
			}
		}
	if (rec) {
		m = new_rec_str_make_method(t, nfields, fields);
		TypeInterface *ti = new TypeInterface(m, 0);
		e = e->add_var_binding(m->name, ti);
		instns = instns->add_var_binding(m->name, ti);
		}
	// do supertypes
	vec st = make_vec_simple(Type, 1);
        vec_store(st, 0, PV(Any));
	nt->supertypes_ = st;

	// stick it in the environment
	TypeInterface *ti = nt?new TypeInterface(objtype_as_type(nt)):
					new TypeInterface();
	e = e->add_type_binding(full_nm, ti);
	instns = instns->add_type_binding(full_nm, ti);
	tco->set_env(e);
	tco->set_instns(instns);
	}

method new_copy_method(type nt)
{
	method m = new_method();
	m->name = string_const("copy");
	vec v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(nt));
	m->arguments = v;
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_equal_method(type nt)
{
	method m = new_method();
	m->name = string_const("equal");
	vec v = make_vec_simple(Type, 2);
	vec_store(v, 0, PV(nt));
	vec_store(v, 1, PV(nt));
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(class_as_type(Bool)));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_similar_method(type nt)
{
	method m = new_method();
	m->name = string_const("similar");
	vec v = make_vec_simple(Type, 2);
	vec_store(v, 0, PV(nt));
	vec_store(v, 1, PV(nt));
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(class_as_type(Bool)));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_is_method(type nt, string nm, int tag)
{
	method m = new_method();
	m->name = string_concat(string_const("is_"), nm);
	vec v = make_vec_simple(Type, 0);
	// vec_store(v, 0, PV(nt));
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(class_as_type(Bool)));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_value_method(type nt, string nm, int tag, TypeSpec *typ)
{
	method m = new_method();
	m->name = string_concat(string_const("value_"), nm);
	vec v = make_vec_simple(Type, 0);
	// vec_store(v, 0, PV(nt));
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(typ->get_type()->get_type()));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 1);
	signal_ s = new_signal();
	s->name = string_const("wrong_tag");
	s->returns = make_vec_simple(Type, 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_one_var_make_method(type t, string nm, int tag, TypeSpec *typ)
{
	method m = new_method();
	m->name = string_concat(type_as_objtype(t)->name, string_const("_MAKE_"));
	m->name = string_concat(m->name, nm);
	vec v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(typ->get_type()->get_type()));
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(t));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = t;
	return m;
	}

method new_rec_str_make_method(type t, int nfields, ParseNodeList *fields)
{
	method m = new_method();
	m->name = string_concat(type_as_objtype(t)->name, string_const("_MAKE"));
	vec v = make_vec_simple(Type, nfields);
	int i = 0;
	for (Pix p = fields->first(); p ; fields->next(p)) {
		Field *f = (Field *)(*fields)(p);
		TypeSpec *typ = f->get_typeSpec();
		ParseNodeList *ids = f->get_ids();
		for (Pix p2 = ids->first(); p2 ; ids->next(p2)) {
			// Id *id = (Id *)(*ids)(p2);
			// stick in each field type
			vec_store(v, i, PV(typ->get_type()->get_type()));
			i++;
			}
	    	}
	m->arguments = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(t));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = t;
	return m;
	}

method new_get_method(type nt, string nm, int tag, TypeSpec *typ)
{
	method m = new_method();
	m->name = nm;
	vec v = make_vec_simple(Type, 0);
	m->arguments = v;
	m->extra_args = v;
	v = make_vec_simple(Type, 1);
	vec_store(v, 0, PV(typ->get_type()->get_type()));
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}

method new_set_method(type nt, string nm, int tag, TypeSpec *typ)
{
	method m = new_method();
	m->name = string_concat(string_const("set_"), nm);
	vec v = make_vec_simple(Type, 1);
	// vec_store(v, 0, PV(typ->get_type()->get_type()));
	m->arguments = v;
	v = make_vec_simple(Type, 0);
	m->extra_args = v;
	m->returns = v;
	m->signals = make_vec_simple(class_as_type(Signal), 0);
	m->parameterized = FALSE;
	m->spec = string_empty();
	m->index = 0;
	m->self_type = nt;
	return m;
	}


vec methods_from_def(ParseNode *def, type ct, TypeCheckObj *tco)
{
	int line = def->get_line();
	int size = 0;
	int index = 0;
	vec v = make_vec_Method(0, FALSE);
	bool parmd_defn = FALSE;
	ParseNodeList *l = 0;
	switch (def->tag()) {
	  case ImplElt::ClassDefT: {
		ClassDef *cd = (ClassDef *)def;
		ParseNodeList *parms = cd->get_parms();
		if (parms && parms->length()) parmd_defn = TRUE;
		l = cd->get_classElts();
		if (l) size = count_meths(l);
		// Handle Ivars
		l = cd->get_decl();
		if (l) size += count_fcns(l);
		v = make_vec_Method(size, FALSE);
		index = 0;
		if (l) for (Pix p = l->first(); p ; l->next(p)) {
			ParseNode *pn = (*l)(p);
			switch (pn->tag()) {
				case ParseNode::DeclT: {
				  Decl *d = (Decl *)pn;
			  	  if (d->tag() != Decl::ImplDeclT) continue;
				  ImplDecl *imd = (ImplDecl *)d;
				  Id *id = imd->get_id();
				  TypeSpec *ts = imd->get_typeSpec();
				  TypeInterface *ti = 0;
				  set_node_type(ts, ts, tco, TRUE);
				  ti = ts->get_type();
				  Id *imdget = imd->get_get();
				  if (imdget) {
					method m = new_method();
					m->name = imdget->get_id();
					dupe_method(v, index, m->name, line);
					if (parmd_defn) m->parameterized = TRUE;
					else m->parameterized = FALSE;
					m->index = 0;
					m->self_type = ct;
					m->spec = string_const("null");
					m->arguments = make_vec_simple(Type,0);
					m->extra_args = make_vec_simple(Type,0);
					m->returns = make_vec_simple(Type, 1);
			                vec_store(m->returns, 0, 
						PV(ti->get_type()));
					m->signals = make_vec_simple(
						class_as_type(Signal), 0);
					vec_store(v, index, PV(m));
					index++;
					}
				  Id *imdset = imd->get_set();
				  if (imdset) {
					method m = new_method();
					m->name = imdset->get_id();
					dupe_method(v, index, m->name, line);
					if (parmd_defn) m->parameterized = TRUE;
					else m->parameterized = FALSE;
					m->index = 0;
					m->self_type = ct;
					m->spec = string_const("null");
					m->arguments = make_vec_simple(
						class_as_type(Formal), 1);
					formal f = new_formal();
					f->name = id->get_id();
					f->t = ti->get_type();
					vec_store(m->arguments, 0, PV(f));
					m->extra_args = make_vec_simple(Type,0);
					m->returns = make_vec_simple(Type, 0);
					m->signals = make_vec_simple(
						class_as_type(Signal), 0);
					vec_store(v, index, PV(m));
					index++;
					}
				}
			}
		  }
		l = cd->get_classElts();
		break;
		}
	  case ParseNode::SpecEltT: {
		SpecElt *se = (SpecElt *)def;
		TypeIntf *ti = (TypeIntf *)se;
		l = ti->get_methods();
		if (l) size = count_meths(l);
		v = make_vec_Method(size, FALSE);
		ParseNodeList *parms = ti->get_parms();
		if (parms && parms->length()) parmd_defn = TRUE;
		break;
		}
	}
	if (l) for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		switch (pn->tag()) {
			case ParseNode::RoutineIntfT: {
				RoutineIntf *ri = (RoutineIntf *)pn;
				method m = method_from_signature(
						ri->get_signature(), tco);
				m->name = ri->get_id()->get_id();
				m->self_type = ct; // ?appropriate
				if (parmd_defn) m->parameterized = TRUE;
				else m->parameterized = FALSE;
				dupe_method(v, index, m->name, line);
				vec_store(v, index, PV(m));
				index++;
				break;
				}
			case ParseNode::MethodOrOpDefT: {
				MethodOrOpDef *mood = (MethodOrOpDef *)pn;
				RoutineIntf *ri = mood->get_routineDef()
							->get_routineIntf();
				method m = method_from_signature(
					ri->get_signature(), tco);
				m->name = ri->get_id()->get_id();
				m->self_type = ct;
				if (parmd_defn) m->parameterized = TRUE;
				else m->parameterized = FALSE;
				dupe_method(v, index, m->name, line);
				if (mood->get_IsConstr()) {
					vec rets = make_vec_simple(Type, 1);
					vec_store(rets, 0, PV(ct));
					m->returns = rets;
					}
				vec_store(v, index, PV(m));
				index++;
				break;
				}
		default: {
			// This is normal (equates)
			// cmp_err("Losing in methods_from_def", -1);
			}
		   }
		}
	return v;
	}

method method_from_signature(Signature *signa, TypeCheckObj *tco)
{
      method m = new_method();
      m->parameterized = FALSE;
      m->index = 0;
      m->spec = string_const("null");
      m->name = string_empty();

      // Process Arguments
      if (DEBUG) printf("ARGS\n");
      int index = 0;
      ParseNodeList *args = signa->get_args();
      vec formals;
      if (args == NULL) {
	formals = make_vec_simple(class_as_type(Formal), 0);
	}
      else {
        if (DEBUG) print_parsenode_list(args, 2);
        formals = make_vec_simple(class_as_type(Formal),
  					count_lhs(signa->get_args()));
        ParseNodeList la = *args;
        index = 0;
        for (Pix pa = la.first(); pa ; la.next(pa)) {
          Decl *arg = (Decl *)la(pa);
  	  switch (arg->tag()) {
  	    case Decl::RegDeclT: {
  	       RegDecl *r = (RegDecl *)arg;
	       TypeSpec *ts = r->get_typeSpec();
	       set_node_type(ts, r, tco, FALSE);
  	       TypeInterface *ti = r->get_type();
               ParseNodeList l = *(r->get_ids());
                 for (Pix p = l.first(); p ; l.next(p)) {
  			Id *id = (Id*)l(p);
  			if (DEBUG) id->print(3);
  			string nm = id->get_id();
  			formal f = new_formal();
  			f->name = nm;
			// A little simplistic...
  			f->t = ti->get_type();
  			vec_store(formals, index, PV(f));
			index++;
			// RESET_EXC
			// env = env->add_var_binding(nm, ti);
			// CATCH_EXC(exc_duplicate) {
		          //  sprintf(cmp_err_buf, "Duplicate name: %.50s",
			    //        string_charp(id->get_id()));
		           // cmp_err(cmp_err_buf, id->get_line());
			   // }
			}
		break;
		}
	    }
	  }
	}
      m->arguments = formals;
      m->extra_args = make_vec_simple(Type,0);

      // Process Returns
      if (DEBUG) printf("RETS\n");
      ParseNodeList *rets = signa->get_returns();
      index = 0;
      vec returns;
      if (rets == NULL) {
      		returns = make_vec_simple(Type, 0);
		}
      else {
      	if (DEBUG) print_parsenode_list(rets,2);
	returns = make_vec_simple(Type, rets->length());
	ParseNodeList la = *rets;
	for (Pix pa = la.first(); pa ; la.next(pa)) {
	  TypeSpec *ret = (TypeSpec *)la(pa);
          set_node_type(ret, ret, tco, TRUE);
	  TypeInterface *ti = ret->get_type();
	  vec_store(returns, index, PV(ti->get_type()));
	  index++;
	  } // end for
	}
      m->returns = returns;

      // Process Signals
      if (DEBUG) printf("SIGS\n");
      ParseNodeList *sigs = signa->get_signals();
      index = 0;
      vec signals;
      if ( sigs == NULL ) {
      	signals = make_vec_simple(class_as_type(Signal), 0);
	}
      else {
        if (DEBUG) print_parsenode_list(sigs, 2);
        signals = make_vec_simple(class_as_type(Signal), sigs->length());
        ParseNodeList la = *sigs;
        for (Pix pa = la.first(); pa ; la.next(pa)) {
      	  signal_ s = new_signal();
          Exception *sg = (Exception *)la(pa);
	  s->name = sg->get_id()->get_id();
	  int index2 = 0;
          vec values;
          ParseNodeList *vals = sg->get_typeSpec();
	  if (vals == NULL) {
          	values = make_vec_simple(Type, 0);
		}
	  else {
            values = make_vec_simple(Type, vals->length());
            for (Pix p = vals->first(); p ; vals->next(p)) {
		// ParseNode *pn = (ParseNode)(*vals)(p);
		TypeSpec *ts = (TypeSpec *)(*vals)(p);
		set_node_type(ts, ts, tco, TRUE);	// possibly simplistic
		type v = ts->get_type()->get_type();
		vec_store(values, index2, PV(v));
		index2++;
		} // end type for
	    }
	  s->returns = values;
	  vec_store(signals, index, PV(s));
	  index++;
	  } // end sig for
        }
        m->signals = signals;
        // signa->type_ = new TypeInterface(m, 0);
        // if (DEBUG) printf("m %lX sig %lX\n", m, signa->type_);
	// this->set_current_method(m);
    
    switch (signa->tag()) {
      
    case Signature::ProcSigT: {
      m->iter = FALSE;
        break;
    }
    case Signature::IterSigT: {
      m->iter = TRUE;
      break;
    }
    }

	return m;
	}
vec superclass_from_inherits(Inherit *inh, TypeCheckObj *tco) {
	if (!inh) return make_vec_simple(Type, 0);
	TypeSpec *ts = inh->get_classSpec();
	set_node_type(ts, ts, tco, TRUE);
	type t = ts->get_type()->get_type();
	if (t) {
		vec v = make_vec_simple(Type, 1);
		vec_store(v, 0, PV(t));
		}
	else return make_vec_simple(Type, 0);
	}

// Should handle renamings...
vec supertypes_from_intf(TypeIntf *tin, TypeCheckObj *tco)
{
vec v;

	ParseNodeList *sprs = tin->get_supers();
	if (sprs == NULL) {
		// should avoid storing any if type is any, but
		//	seems unlikely so far
      		v = make_vec_simple(Type, 1);
		vec_store(v, 0, PV(Any));
      		return v;
		}
	v = make_vec_simple(Type, sprs->length());
	int i = 0;
	for (Pix p = sprs->first() ; p ; sprs->next(p)) {
		SuperInfo *si = (SuperInfo *)(*sprs)(p);
		TypeSpec *ts = si->get_typespec();
		set_node_type(ts, si, tco, FALSE);
		// may need work...
		vec_store(v, i, PV(si->get_type()->get_type()));
		i++;
		}
	return v;
	}

vec supertypes_from_intfnm(string s) {
	// cmp_err("Time to implement supertypes_from_intf.", -1);
	return (make_vec_simple(Type, 0));
	}

vec formals_from_decls(ParseNodeList *l, TypeCheckObj *tco)
{
	if (!l) return (make_vec_simple(Type, 0));
	int size = count_lhs(l);
	vec v = make_vec_simple(class_as_type(Formal), size);
	int index = 0;
	for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		formal f = new_formal();
		switch (pn->tag()) {
			case ParseNode::DeclT: {
			  Decl *d = (Decl *)pn;
			  switch (d->tag()) {
				case Decl::ImplDeclT: {
					formal f = new_formal();
			  		ImplDecl *imd = (ImplDecl *)d;
					TypeSpec *ts = imd->get_typeSpec();
					set_node_type(ts, d, tco, TRUE);
					TypeInterface *ti = d->get_type();
			  		Id *id = imd->get_id();
			  		f->name = id->get_id();
			  		f->t = ti->get_type();
					vec_store(v, index, PV(f));
					index++;
					break;
					}
				case Decl::RegDeclT: {
			  		RegDecl *rd = (RegDecl *)d;
					TypeSpec *ts = rd->get_typeSpec();
					set_node_type(ts, d, tco, TRUE);
					TypeInterface *ti = d->get_type();
					ParseNodeList *ids = rd->get_ids();
					for (Pix q = ids->first(); q; 
							ids->next(q)) {
						Id *id = (Id *)(*ids)(q);
						formal f = new_formal();
			  			f->name = id->get_id();
						f->t = ti->get_type();
						vec_store(v, index, PV(f));
						index++;
						}
					break;
					}
			  	}
			  }
			}
		}
	return v;
	}

ParseNodeList *tintf_get_method_rets(ParseNode *tpn, string m_name)
{
ParseNodeList *rets = 0;

    switch (tpn->tag()) {
      case ParseNode::SpecEltT: {
      SpecElt *se = (SpecElt*)tpn;
      switch (se->tag()) {
	case SpecElt::TypeIntfT: {
	TypeIntf *tif = (TypeIntf *)se;
	ParseNodeList *meths = tif->get_methods();
	if (!meths) return (NULL);
	for (Pix p = meths->first() ; p ; meths->next(p)) {
		RoutineIntf *ri = (RoutineIntf *)(*meths)(p);
		if (! string_equal(m_name, ri->get_id()->get_id())) continue;
		Signature *sg = ri->get_signature();
		rets = sg->get_returns();
		}
	break;
	}
	/*
	case SpecElt::EtypeIntfT: {
	EtypeIntf *etif = (EtypeIntf *)se;
	ParseNodeList *opers = etif->get_operations();
	if (!opers) return (NULL);
	for (Pix p = opers->first() ; p ; opers->next(p)) {
		RoutineIntf *ri = (RoutineIntf *)(*opers)(p);
		if (! string_equal(m_name, ri->get_id()->get_id())) continue;
		Signature *sg = ri->get_signature();
		rets = sg->get_returns();
		}
	break;
	}
	*/
	} // end switch
	break;
	} // end SpecElT
	} // end other switch
	return rets;
	}

TypeInterface *pn_to_ti(ParseNode *tpn, Environment *env)
{
TypeInterface *ti = 0;

	printf("pn to ti\n");
	if (DEBUG) tpn->print(2);
	switch (tpn->tag()) {
	case ParseNode::SpecEltT: {
		SpecElt *se = (SpecElt *) tpn;
		switch (se->tag()) {
		/*
		case SpecElt::EtypeIntfT: {
		EtypeIntf *et = (EtypeIntf *)tpn;
     		NameBinding *bind = env->look_up (et->get_id()->get_id());
      		if (bind == 0) {
			sprintf(cmp_err_buf, "No binding found for %.50s (SE).",
				et->get_id()->get_id()->chars);
			cmp_err(cmp_err_buf, et->get_line());
		      }
		      else if (bind->tag() != NameBinding::TypeBindingT) {
			sprintf(cmp_err_buf, "Naming conflict; %.50s found, but not as variable.",
				et->get_id()->get_id());
			cmp_err(cmp_err_buf, et->get_line());
		      }
		      else 
			ti = bind->get_type();
		break;
		} // end etype
		*/
		case SpecElt::TypeIntfT: {
		TypeIntf *tif = (TypeIntf *)tpn;
     		NameBinding *bind = env->look_up (tif->get_id()->get_id());
      		if (bind == 0) {
			sprintf(cmp_err_buf, "No binding found for %.50s (SE).",
				tif->get_id()->get_id()->chars);
			cmp_err(cmp_err_buf, tif->get_line());
		      }
		      else if (bind->tag() != NameBinding::TypeBindingT) {
			sprintf(cmp_err_buf, "Naming conflict; %.50s found, but not as variable.",
				tif->get_id()->get_id());
			cmp_err(cmp_err_buf, tif->get_line());
		      }
		      else 
			ti = bind->get_type();
		break;
		} // end typeintf
		} // end switch
		break;
		} // end spec elt
	case ParseNode::TypeSpecT :{
		TypeSpec *ts = (TypeSpec *)tpn;
		switch (ts->tag()) {
		case TypeSpec::SimpleTypeSpecT : {
			SimpleTypeSpec *sts = (SimpleTypeSpec *)ts;
			ParseNode *field = sts->get_name();
			switch (field->tag()) {
			case ParseNode::TypeNameT: {
				TypeName *tn = (TypeName *)field;
				string nm = tn->get_name()->get_id();
     				NameBinding *bind = env->look_up (nm);
		      		if (bind == 0) {
					sprintf(cmp_err_buf, 
						"No binding found for %.50s (TC)\n", 
						nm->chars);
					cmp_err(cmp_err_buf, tpn->get_line());
		      		}
		      		else if (bind->tag() != NameBinding::TypeBindingT) {
					sprintf(cmp_err_buf,
						"Naming conflict; %.50s found, but not as type\n",
						nm->chars);
					cmp_err(cmp_err_buf, tpn->get_line());
					}
		      			else ti = bind->get_type();
				break;
		        }
			// case ParseNode::TypeObjectT: {
			// 	TypeObject *bar = (TypeObject *)field;
			// 	ti = bar->get_leaf_type();
			// 	break;
			// 	}
			default: {
				cmp_err("Losing in pn_to_ti: not a tn or to.",
						tpn->get_line());
				}
			} // end switch
			break;
			} // end sts
		default: {
			cmp_err("Losing in pn_to_ti: not a simple.",
					tpn->get_line());
			}
		} // end switch
		break;
		} // end typespec
	default: {
		cmp_err("Losing in pn_to_ti: not a typespec.", tpn->get_line());
		}
		} // end switch
	return ti;
	}

Decl *find_ivar_decl(string m_name, TypeCheckObj *tco)
{
	ClassDef *cd = (ClassDef *)tco->get_current_class();
	ParseNodeList *dl = cd->get_decl();
	return find_ivar_in_dl(m_name, dl);
	}

Decl *find_ivar_in_dl(string m_name, ParseNodeList *dl)
{
	if (!dl) return 0;
	for (Pix p = dl->first(); p ; dl->next(p)) {
		Decl *d = (Decl *)(*dl)(p);
		switch (d->tag()) {
			case Decl::RegDeclT: {
				RegDecl *r = (RegDecl *)d;
				ParseNodeList *ids = r->get_ids();
				for (Pix p = ids->first(); p ; ids->next(p)) {
					Id *id = (Id *)(*ids)(p);
					if (string_equal(id->get_id(), m_name))
						return d;
					}
				break;
			}
			case Decl::ImplDeclT: {
				ImplDecl *im = (ImplDecl *)d;
				if (string_equal(im->get_id()->get_id(), m_name))
					return d;
				break;
			}
			case Decl::VarArgsDeclT: {
			cmp_err("find_ivar_decl: varargsdecl unexpected.", -1);
			break;
			}
			} // end switch
		} // end for
	return 0;
	}

bool same_deftype(ParseNode *pn1, ParseNode *pn2, Environment *env)
{
TypeInterface *et1_ti = 0;
TypeInterface *et2_ti = 0;

	cmp_err("Time to implement same_deftype.", -1);
	return FALSE;

	/*	
	et1_ti = pn_to_ti(pn1, env);
	et2_ti = pn_to_ti(pn2, env);
	if (!et1_ti || !et2_ti) return FALSE;
	EtypeIntf *tpn1 = (EtypeIntf *)et1_ti->get_intf();
	EtypeIntf *tpn2 = (EtypeIntf *)et2_ti->get_intf();
	string nm1 = tpn1->get_id()->get_id();
	string nm2 = tpn2->get_id()->get_id();
	// tpn->print(1);
	// tpn = et2_ti->get_intf();
	// tpn->print(1);
	return string_equal(nm1, nm2);
	*/
	}

void set_node_type(TypeSpec *ts, ParseNode *pn, TypeCheckObj *tco, bool verbose)
{
string nm = ts2nm(ts, pn->get_line());

    if (string_equal(nm, string_empty())) return;
    bool ok = find_type(nm, pn, tco->get_env(), FALSE);
    if (ok) return;
    switch (ts->tag()) {
	case TypeSpec::ParamTypeSpecT: {
		ParamTypeSpec *ps = (ParamTypeSpec *)ts;
		// see if we can do the instantiation and live happily
		instantiate_pts(ps, nm, pn->get_line(), tco);
        	find_type(nm, pn, tco->get_env(), TRUE);
		break;
	}
	case TypeSpec::TaggedTypeSpecT: {
		TaggedTypeSpec *tts = (TaggedTypeSpec *)ts;
		// see if we can do the instantiation and live happily
		instantiate_tagged(tts, nm, pn->get_line(), tco);
        	find_type(nm, pn, tco->get_env(), TRUE);
		break;
	}
    default: {
	// do it again to get error message...
        find_type(nm, pn, tco->get_env(), TRUE);
	}
    }
}

void instantiate_pts(ParamTypeSpec *ps, string full_nm, int line, 
							TypeCheckObj *tco)
{
	Environment *e = tco->get_env();
	string nm = ps->get_name()->get_name()->get_id();
	TypeInterface *save_ti = ps->get_type();
    	bool ok = find_type(nm, ps, e, FALSE);
	if (!ok) { sprintf(cmp_err_buf,
			"Type Generator %s not found", nm->chars);
		   cmp_err(cmp_err_buf, ps->get_line());
		   return;
		}
	type t = ps->get_type()->get_type();
	ps->type_ = save_ti;
	if (!t) {
		   sprintf(cmp_err_buf,
			"Insufficient type information for %s", nm->chars);
		   cmp_err(cmp_err_buf, ps->get_line());
		   return;
		}
	objtype ot = type_as_objtype(t);
	if ((ot->kind != PTYPE_KIND && ot->kind != PCLASS_KIND)) {
		   sprintf(cmp_err_buf,
			" %s found but is not a type generator", nm->chars);
		   cmp_err(cmp_err_buf, ps->get_line());
		   return;
		}
	// needs review: seems a little suspicious: perhaps create a new ptype,
	//	rather than mutating this ptype...
	ptype pt = type_as_ptype(t);
	ParseNodeList *l = ps->get_actualParms();
	vec v = tis2vec(l, ps->get_line(), tco);
	if (!v) return;	// new 5/4/95
	RESET_EXC
	type nt = 0;
	if (v) nt = instn_as_type(ptype_instantiate(pt, v));
	CATCH { sprintf(cmp_err_buf, "Unable to instantiate: %.50s",
			    string_charp(full_nm));
		    cmp_err(cmp_err_buf, ps->get_line());
		nt = 0;
		return;	// new 5/4/95
		}
	TypeInterface *ti = nt?new TypeInterface(nt):new TypeInterface();
	e = e->add_type_binding(full_nm, ti);
	tco->set_env(e);
}

vec tis2vec(ParseNodeList *l, int line, TypeCheckObj *tco)
{
vec v;
	
	if (!l) return 0;
	v = make_vec_simple(Type, l->length());
	int i = 0;
	for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		type t = 0;
		switch (pn->tag()) {
			case ParseNode::ActualParmT: {
				ActualParm *ap = (ActualParm *)(*l)(p);
				TypeSpec *ts = ap->get_typeSpec();
				set_node_type(ts, ap, tco, TRUE);
				t = ap->get_type()->get_type();
				if (!t) { sprintf(cmp_err_buf,
					" No type found for %s ", 
					string_charp(ts2nm(ts, ap->get_line())));
		   			cmp_err(cmp_err_buf, ap->get_line());
					return 0;
				}
				vec_store(v, i, PV(t));
				i++;
				break;
			}
			// The following clause may not be necessary
			case ParseNode::ParmT: {
				Parm *pm = (Parm *)(*l)(p);
				ParseNodeList *idl = pm->get_ids();
				for (Pix p2 = idl->first(); p2; idl->next(p2)) {
					Id *id = (Id *)(*idl)(p2);
    					find_type(id->get_id(), pm, tco->get_env(), TRUE);
					t = pm->get_type()->get_type();
					vec_store(v, i, PV(t));
					i++;
					}
				break;
			}
			default: {
				cmp_err("Unexpected Node in tis2vec", pn->get_line());
				if (DEBUG) pn->print(2);
				}
			}
		}
	return v;
	}

string ts2nm(TypeSpec *ts, int line)
{
string nm = string_empty();

    switch (ts->tag()) {
      case TypeSpec::SimpleTypeSpecT: {
	SimpleTypeSpec *st = (SimpleTypeSpec *)ts;
	nm = st->get_name()->get_name()->get_id();
        break;
      }
      case TypeSpec::ParamTypeSpecT: {
	ParamTypeSpec *ps = (ParamTypeSpec *)ts;
	nm = ps->get_name()->get_name()->get_id();
	ParseNodeList *l = ps->get_actualParms();
	nm = string_concat(nm, parms2nm(l));
        break;
      }
      case TypeSpec::TaggedTypeSpecT: {
		TaggedTypeSpec *tts = (TaggedTypeSpec *)ts;
		nm = tts->get_name()->get_name()->get_id();
		ParseNodeList *fields = tts->get_fields();
		nm = string_concat(nm, fields2nm(fields));
        break;
      }
      case TypeSpec::RoutineTypeSpecT: {
		RoutineTypeSpec *rts = (RoutineTypeSpec *)ts;
		Signature *si = rts->get_signature();
		nm = string_const("PROC");
		// needs work
        break;
      }
    }
    return nm;
}

string fields2nm(ParseNodeList *l)
{
string nm = string_empty();
string trail = string_empty();

	bool first = TRUE;
	for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		if (pn->tag() != ParseNode::FieldT) {
			if (DEBUG) pn->print(3);
			}
		Field *f = (Field *)(*l)(p);
		ParseNodeList *idl = f->get_ids();
		for (Pix p2 = idl->first(); p2 ; idl->next(p2)) {
			ParseNode *pn2 = (*idl)(p2);
			Id *id = (Id *)pn2;
			if (first) {
				nm = string_concat(nm, string_const("_OF_"));
				first = FALSE;
			}
			else nm = string_concat(nm, string_const("_AND_"));
			nm = string_concat(nm, id->get_id());
			}
		}
	return nm;
	}

/* may not be doing the right thing w.r.t. equated names */
string parms2nm(ParseNodeList *l)
{
string nm = string_empty();

	bool first = TRUE;
	for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		string trail = string_empty();
		switch (pn->tag()) {
		  case ParseNode::ActualParmT: {
			ActualParm *ap = (ActualParm *)(*l)(p);
			trail = ts2nm(ap->get_typeSpec(), ap->get_line());
			break;
			}
		  case ParseNode::ParmT: {
			Parm *pm = (Parm *)(*l)(p);
			ParseNodeList *ids = pm->get_ids();
			// sort of sleazy
			Id *id = (Id *)ids->front();
			trail = id->get_id();
			break;
			}
		  default: {
			cmp_err("parms2nm losing", pn->get_line());
				if (DEBUG) pn->print(3);
			return nm;
			}
		}
		if (first) {
			nm = string_concat(nm, string_const("_OF_"));
			first = FALSE;
			}
		else nm = string_concat(nm, string_const("_AND_"));
		nm = string_concat(nm, trail);
		// needs more work to cover renamings
		}
	return nm;
	}

bool find_type(string s, ParseNode *pn, Environment *env, bool verbose)
{
	NameBinding *bind = env->look_up (s);
      	if (bind == 0) {
		if (verbose) sprintf(cmp_err_buf, 
			"No binding found for %.50s (FT).",
			string_charp(s));
		if (verbose) cmp_err(cmp_err_buf, pn->get_line());
		return FALSE;
	      }
	      else if (bind->tag() != NameBinding::TypeBindingT) {
		if (verbose) sprintf(cmp_err_buf, 
			"Naming conflict; %.50s found, but not as type (FT).",
			string_charp(s));
		if (verbose) cmp_err(cmp_err_buf, pn->get_line());
		return FALSE;
	      }
	      else {
		pn->type_ = bind->get_type();
		return TRUE;
	      }
	}


TypeInterface *get_type(string s, Environment *env)
{
	NameBinding *bind = env->look_up (s);
      	if (bind == 0) return 0;
        else if (bind->tag() != NameBinding::TypeBindingT) return 0;
	     else return bind->get_type();
	}

void find_var(string s, ParseNode *pn, Environment *env, bool verbose)
{
	NameBinding *bind = env->look_up (s);
      	if (bind == 0) {
		if (verbose) {
			sprintf(cmp_err_buf, 
				"No binding found for %.50s (FV).",
				string_charp(s));
			cmp_err(cmp_err_buf, pn->get_line());
			}
	      	}
	else if (bind->tag() != NameBinding::VarBindingT) {
		if (verbose) sprintf(cmp_err_buf, 
			"Naming conflict; %.50s found, but not as var (FV).",
			string_charp(s));
		if (verbose) 
			cmp_err(cmp_err_buf, pn->get_line());
	      	}
	      else pn->type_ = bind->get_type();
	}

void find_var_or_type(string s, ParseNode *pn, Environment *env, bool verbose)
{
	NameBinding *bind = env->look_up (s);
      	if (bind != 0) pn->type_ = bind->get_type();
	else {
		if (verbose) {
			sprintf(cmp_err_buf, 
				"No binding found for %.50s (FVT).",
				string_charp(s));
			cmp_err(cmp_err_buf, pn->get_line());
			}
	      }
	}

type get_one_type(TypeInterface *ti, int lineno)
{
	switch (ti->tag()) {
		case TypeInterface::SingleT: {
			return(ti->get_type());
			break;
			}
		case TypeInterface::MultipleT: {
			vec v  = ti->get_mult();
			if (vec_length(v) != 1) {
				sprintf(cmp_err_buf,"Mult TI not 1 but %d.",
					vec_length(v));
				cmp_err(cmp_err_buf, lineno);
				return 0;
				}
			else return(UNPV(type, vec_fetch(v, 0)));
			break;
			}
		}
	return 0;
	}

// Report an error if a member of idl is not in the current class defn.

void check_methods(ParseNodeList *idl, ClassDef *cd, Environment *env)
{
	if (idl) for (Pix p = idl->first(); p ; idl->next(p)) {
		Id *id = (Id *)(*idl)(p);
		check_method(id, cd, env);
		} // end for
	return;
	}

// Report an error if id is not in the current class defn.

void check_method(Id *id, ClassDef *cd, Environment *env)
{
	// Check ivars
	ParseNodeList *l = cd->get_decl();
	if (l) for (Pix p = l->first(); p ; l->next(p)) {
		Decl *d = (Decl*)(*l)(p);
		switch (d->tag()) {
		    case Decl::ImplDeclT: {
			ImplDecl *imp = (ImplDecl *)d;
			Id *gm = imp->get_get();
			Id *sm = imp->get_set();
			if (gm && 
				string_equal(id->get_id(), 
					imp->get_id()->get_id())) return;
			if (sm && 
				string_equal(id->get_id(), 
					string_concat(string_const("get_"), 
					imp->get_id()->get_id()))) return;
			}
		    }
		}
	// Check methods
	l = cd->get_classElts();
	if (l) for (Pix p = l->first(); p ; l->next(p)) {
		ClassElt *ce = (ClassElt *)(*l)(p);
		switch (ce->tag()) {
		    case ClassElt::MethodOrOpDefT: {
			MethodOrOpDef *mopd = (MethodOrOpDef *)ce;
			string nm = mopd->get_routineDef()->
					get_routineIntf()->get_id()->get_id();
			if (string_equal(nm, id->get_id())) return;
			}
		    }
		}
	// Check inherited methods w/renames
	Inherit *inh = cd->get_inherits();
	if (inh ) {
	TypeSpec *ts = inh->get_classSpec();
	switch (ts->tag()) {
		case TypeSpec::SimpleTypeSpecT: {
			SimpleTypeSpec *sts = (SimpleTypeSpec *)ts;
			TypeName *tn = sts->get_name();
			TypeInterface *ti = get_type(tn->get_name()
					->get_id(), env);
			switch (ti->tag()) {
      			    case TypeInterface::SingleT: {
      			      fevalue tgm_ret[2];
		              type t = ti->get_type();
		    	      RESET_EXC
		    	      getMethod(type_as_objtype(t), 
					tgm_ret, id->get_id());
		              CATCH {
		    	      exc = EXC_NONE;
			      return;
        			      }
		             else break;
			     }
			    } // end ti switch
			} // end sts case
		default: {
			cmp_err("Time to extend check_method", -1);
			}
		} // end ts switch
	}
	sprintf(cmp_err_buf,"Method %s not found in %s", 
			string_charp(id->get_id()),
			string_charp(cd->get_classId()->get_id()));
	cmp_err(cmp_err_buf, cd->get_line());
	return;
	}

struct check_env{
	objtype def;
	objtype typ;
	int line;
	};

void check_nth_method(struct check_env *env, method m);

void check_def_against_type(type tdef, type ttyp,  int line, Environment *env)
{
	 if (ttyp == 0) { // sprintf(cmp_err_buf,"Unable to find type for class");
                        // cmp_err(cmp_err_buf, line);
	 		return;
	 		}
	objtype typ = type_as_objtype(ttyp);
	objtype def = type_as_objtype(tdef);

	// temporarily turn off supertype of class
	vec save_sprs = def->supertypes_;
	def->supertypes_ = make_vec_simple(Type, 0);

	// loop through type's methods
	struct check_env cenv;
	cenv.def = def;
	cenv.typ = typ;
	cenv.line = line;
	struct closure cl;
	cl.env = &cenv;
	cl.f = (ifunc)check_nth_method;
	objtype_all_methods_(ttyp, cl);

	// restore 
	def->supertypes_ = save_sprs;
}

void check_nth_method(struct check_env *env, method m)
{
	string m_name = method_name(m);
	// see if class has that method
	fevalue tgm_ret[2];
    	RESET_EXC
	getMethod(env->def, tgm_ret, m_name);
	// if not, holler
	CATCH {
		sprintf(cmp_err_buf,"Method %s not found in class %s"
			" but required by type %s", 
			string_charp(m_name), 
			string_charp(env->def->name), 
			string_charp(env->typ->name));
		cmp_err(cmp_err_buf, env->line);
		  }
	else {
		method om = (method)tgm_ret[0].o;
		check_rtn_against_rtn_spec(om, m, env->line);
		}
}

int count_parms(ParseNodeList *parms)
{
int count = 0;

	if (parms) {
		for (Pix p = parms->first() ; p ; parms->next(p)) {
		  Parm *parm = (Parm *)(*parms)(p);
		  ParseNodeList *ids = parm->get_ids();
		  if (ids) for (Pix p2 = ids->first() ; p2 ; ids->next(p2)) {
			Id *nth = (Id *)(*ids)(p2);
			count++;
			}
		  }
	  }
	return count;
}

int count_meths(ParseNodeList *l)
{
int size = 0;

	if (l) for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		switch (pn->tag()) {
			case ParseNode::MethodOrOpDefT: {
			  size++;
			  break;
			  }
			case ParseNode::RoutineIntfT: {
			  size++;
			  break;
			  }
			}
		}
	return size;
	}

void check_rtn_against_rtn_spec(method m1, method m2, int line)
{
	if (!m1 || !m2) {
		sprintf(cmp_err_buf,"Method definition(s) missing");
		cmp_err(cmp_err_buf, line);
		return;
		}
	string m1_name = m1->name;
	string m2_name = m2->name;
// same # of args, check subtyping per arg
	vec m1_args = m1->arguments;
	vec m2_args = m2->arguments;
	int m1_size = m1_args? vec_length(m1_args) : 0;
	int m2_size = m2_args? vec_length(m2_args) : 0;
	if (m1_size != m2_size) {
			sprintf(cmp_err_buf,"Wrong number of arguments in "
				"matching method %s implementation to spec",
				string_charp(m1_name));
			cmp_err(cmp_err_buf, line);
		}
	else {
		for (int i = 0; i < m1_size; i++) {
			formal f1 = UNPV(formal, vec_fetch(m1_args, i));
			formal f2 = UNPV(formal, vec_fetch(m2_args, i));
			if (!f1->t) {
				sprintf(cmp_err_buf,"No type information "
				  "for argument %s in impl method %s",
					string_charp(f1->name), 
					string_charp(m1_name));
				cmp_err(cmp_err_buf, line);
				continue;
				}
			if (!f2->t) {
				sprintf(cmp_err_buf,"No type information "
				  "for argument %s in spec method %s",
					string_charp(f2->name), 
					string_charp(m1_name));
				cmp_err(cmp_err_buf, line);
				continue;
				}
			if (!isSubtype(f2->t, f1->t)) {
				sprintf(cmp_err_buf,"Argument %d type mismatch "
				  "in matching method %s implementation to spec",
					i+1, string_charp(m1_name));
				cmp_err(cmp_err_buf, line);
				}
			}
		}
// same # of rets, check subtyping per ret
	vec m1_rets = m1->returns;
	vec m2_rets = m2->returns;
	m1_size = m1_rets? vec_length(m1_rets) : 0;
	m2_size = m2_rets? vec_length(m2_rets) : 0;
	if (m1_size != m2_size) {
			sprintf(cmp_err_buf,"Wrong number of returns in "
				"matching method %s implementation to spec",
				string_charp(m1_name));
			cmp_err(cmp_err_buf, line);
		}
	else {
		for (int i = 0; i < m1_size; i++) {
			type t1 = UNPV(type, vec_fetch(m1_rets, i));
			type t2 = UNPV(type, vec_fetch(m2_rets, i));
			if (!t1 || !t2 || !isSubtype(t1, t2)) {
				sprintf(cmp_err_buf,"Return value %d type mismatch "
				  "in matching method %s implementation to spec",
					i+1, string_charp(m1_name));
				cmp_err(cmp_err_buf, line);
				}
			}
		}

// same or fewer # of sigs, subset of names
// same # of vals per sig, check subtyping per val
	vec m1_sigs = m1->signals;
	vec m2_sigs = m2->signals;
	m1_size = m1? vec_length(m1_sigs) : 0;
	m2_size = m2? vec_length(m2_sigs) : 0;
	if (m1_size > m2_size) {
			sprintf(cmp_err_buf,"Wrong number of signals in"
				" matching method %s implementation to spec",
				string_charp(m1_name));
			cmp_err(cmp_err_buf, line);
		}
	else {
		for (int i = 0; i < m1_size; i++) {
		   signal_ s1 = UNPV(signal_, vec_fetch(m1_sigs, i));
		   string m1_sname = s1->name;
		   vec m1_svec = s1->returns;
		   int m1_scount = m1_svec? vec_length(m1_svec) : 0;
		   int found = 0;
		   for (int j = 0; j < m1_size; j++) {
		        signal_ s2 = UNPV(signal_, vec_fetch(m2_sigs, j));
		        string m2_sname = s2->name;
			if (string_equal(m2_sname, m1_sname)) {
				found = 1;
		   		vec m2_svec = s2->returns;
		   		int m2_scount = m2_svec? vec_length(m2_svec) : 0;
				if (m2_scount != m1_scount) {
				   sprintf(cmp_err_buf,"Signal value count mismatch "
				  "in matching method %s implementation to spec"
				  " signal name %s",
					string_charp(m1_name), 
					string_charp(m1_sname));
				  cmp_err(cmp_err_buf, line);
				  break;
					}
			      for (int k = 0; k < m1_scount; k++) {
				type t1 = UNPV(type, vec_fetch(m1_svec, k));
				type t2 = UNPV(type, vec_fetch(m2_svec, k));
				if (!t2) {
					sprintf(cmp_err_buf,"Type info missing"
						" for value %d in spec for"
						" signal %s", k, 
						string_charp(m1_sname));
					cmp_err(cmp_err_buf, line);
					continue;
					}
				if (!t1) {
					sprintf(cmp_err_buf,"Type info missing"
						" for value %d in impl for"
						" signal %s", k, 
						string_charp(m1_sname));
					cmp_err(cmp_err_buf, line);
					continue;
					}
			        if (!isSubtype(t1, t2)) {
			      	      sprintf(cmp_err_buf,"Signal value %d type mismatch "
				        "in matching method %s implementation to spec"
				        " signal name %s",
				        i+1, string_charp(m1_name), 
					string_charp(m1_sname));
				      cmp_err(cmp_err_buf, line);
				break;
				}
			      break;
			      }
			   }
			else continue;
			}
		   if (!found) {
			sprintf(cmp_err_buf,"Signal not found "
			  "in matching method %s implementation to spec"
			  " signal name %s",
				string_charp(m1_name), string_charp(m1_sname));
			cmp_err(cmp_err_buf, line);
			}
		   }
		}
	}

ParseNodeList *dl2tl(ParseNodeList *dl)
{
ParseNodeList *l = new ParseNodeList();

	for (Pix p = dl->first(); p ; dl->next(p)) {
		Decl *d = (Decl *)(*l)(p);
		switch (d->tag()) {
		   case Decl::RegDeclT: {
		   RegDecl *rd = (RegDecl *)d;
			TypeSpec *ts = rd->get_typeSpec();
			// TypeInterface *ti = pn_to_ti(ts);
			ParseNodeList *idl = rd->get_ids();
			for (Pix i = idl->first(); i; idl->next(i)) {
				Id *id = (Id *)(*idl)(i);
				// l->append(ti);
				}
			}
		  }
		}
	return l;
	}

int count_fcns(ParseNodeList *l)
{
int size = 0;

	if (l) for (Pix p = l->first(); p ; l->next(p)) {
		ParseNode *pn = (*l)(p);
		switch (pn->tag()) {
			case ParseNode::DeclT: {
			  Decl *d = (Decl *)pn;
			  if (d->tag() != Decl::ImplDeclT) continue;
			  ImplDecl *imd = (ImplDecl *)d;
			  Id *imdget = imd->get_get();
			  if (imdget) size++;
			  Id *imdset = imd->get_set();
			  if (imdset) size++;
			  }
			}
		}
	return size;
}

// check previous definition of m_name in vec
//	Note: vec only partially built: use size!
void dupe_method(vec v, int size, string m_name, int line)
{
	for (int i = 0; i < size; i++) {
		method m = UNPV(method, (vec_fetch(v, i)));
		if (string_equal(m_name, m->name)) {
			sprintf(cmp_err_buf,"Duplicate method name %s ",
				string_charp(m_name));
			cmp_err(cmp_err_buf, line);
			}
		}
	}

bool exported(string nm, TypeCheckObj *tco)
{
	ImplModule *im = (ImplModule *)tco->get_current_module();
	ParseNodeList *exports = im->get_exports();
	if (exports) for (Pix p = exports->first(); p ; exports->next(p)) {
		Id *id = (Id *)(*exports)(p);
		if (string_equal(nm, id->get_id())) return TRUE;
		}
	return FALSE;
	}

// needs a better implementation someday...
string string_const(char *s)
{
	return string_new(s);
	}

vec parms2vec(ParseNodeList *l, ptype pt, ParseNodeList *rl, int count, 
				TypeCheckObj *tco)
{
vec v = make_vec_Param(count, FALSE);

	int i = 0;
	for (Pix p = l->first(); p ; l->next(p)) {
		Parm *parm = (Parm *)(*l)(p);
		ParseNodeList *ids = parm->get_ids();
		if (ids) for (Pix p2 = ids->first() ; p2 ; ids->next(p2)) {
			Id *id = (Id *)(*ids)(p2);
			param pm = new_param();
			pm->name = id->get_id();
			pm->ptype_ = pt;
			vec_store(v, i, PV(pm));
			Environment *e = tco->get_env();
			e = e->add_type_binding(id->get_id(),
				new TypeInterface(param_as_type(pm)));
			tco->set_env(e);
			i++;
			}
		}
	// first loop through restricts counting restricts per param
	//	and noting any restricts not corresponding to parms
	i = 0;
	if (rl) for (p = rl->first(); p ; rl->next(p)) {
		Restriction *r = (Restriction *)(*rl)(p);
		string rnm = r->get_TypeId()->get_id();
		bool found = FALSE;
		for (i = 0; i < count ; i++) {
			param ithpm = UNPV(param, vec_fetch(v, i));
			if (string_equal(rnm, ithpm->name)) {
				found = TRUE;
				ithpm->methods_ = (vec)((long)ithpm->methods_ +
					r->get_operations()->length());
				}
			}
		if (!found) {
			sprintf(cmp_err_buf, "Where clause for %s "
				"corresponds to no existing parameter",
				string_charp(rnm));
			cmp_err(cmp_err_buf, r->get_line());
			}
		i++;
		}
	// now loop through creating method vecs and filling them in
	for (i = 0 ; i < count ; i++) {
		param ithpm = UNPV(param, vec_fetch(v, i));
		long mcount = (long)ithpm->methods_;
		ithpm->methods_ = make_vec_Method(mcount, FALSE);
		// now loop through filling in method vecs
		int j = 0;
		if (rl) for (p = rl->first(); p ; rl->next(p)) {
			Restriction *r = (Restriction *)(*rl)(p);
			string rnm = r->get_TypeId()->get_id();
			if (string_equal(rnm, ithpm->name)) {
				ParseNodeList *ol = r->get_operations();
				for (Pix p2 = ol->first(); p2 ; ol->next(p2)) {
					RoutineIntf *ri = 
						(RoutineIntf *)(*ol)(p2);
					Signature *sig = ri->get_signature();
					method m = method_from_signature(sig,
						tco);
					m->name = ri->get_id()->get_id();
					m->self_type = ptype_as_type(pt);
					vec_store(ithpm->methods_, j, PV(m));
					j++;
					}
				}
			}
		}
	return v;
	}
/*
			Id *nth = (Id *)(*ids)(p2);
			objtype pnt = new_objtype();
			pnt->name = nth->get_id();
			// needs where clauses in methods
			pnt->methods_ = make_vec_Method(0, FALSE);
			pnt->supertypes_ = make_vec_Type(0, FALSE);
*/

bool check_type_against_supertypes(objtype nt, int line, Environment *env)
{
int size = 0;

	type t = objtype_as_type(nt);
	if (nt->supertypes_) size = vec_length(nt->supertypes_);
	for (int i = 0; i < size ; i++) {
		type st = UNPV(type, vec_fetch(nt->supertypes_, i));
		if (!st || !(isSubtype(t, st))) {
			sprintf(cmp_err_buf,"Type %s is not a subtype of"
				" type %s",
				string_charp(nt->name),
				st? string_charp(type_name(st)) : "NONE");
			cmp_err(cmp_err_buf, line);
			return FALSE;
			}
		}
//	vec save_sprs = nt->supertypes_;
//	nt->supertypes_ = make_vec_simple(Type, 0);
//	for (i = 0; i < size ; i++) {
//		objtype ost = type_as_objtype(UNPV(type, 
//				vec_fetch(save_sprs, i)));
//		if (ost == Any) continue;
//		int n = ost->methods_? vec_length(ost->methods_): 0;
//		for (int j = 0 ; j < n; j++ ) {
//			method m = UNPV(method, vec_fetch(ost->methods_, j));
//			string m_name = method_name(m);
//			fevalue tgm_ret[2];
//			RESET_EXC;
//			getMethod(nt, tgm_ret, m_name);
//			CATCH {
//			    sprintf(cmp_err_buf,"Type %s is missing method %s"
//				" required by its supertype %s",
//				string_charp(nt->name), 
//				string_charp(m_name),
//				string_charp(ost->name));
//			    cmp_err(cmp_err_buf, line);
//			    nt->supertypes_ = save_sprs;
//			    return FALSE;
//			    }
//			method tm = (method) tgm_ret[0].o;
//			check_rtn_against_rtn_spec(tm, m, line);
//			}
//		}
//	nt->supertypes_ = save_sprs;
	return TRUE;
	}

/* 
   The purpose of collect_specs is to take a raw parse tree and
   collect all the spec_elts into a single spec_mod, so that
   forward references can be handled. 
*/
ParseNodeList *collect_specs(ParseNodeList *pnl)
{
ParseNodeList *new_pnl = new ParseNodeList();
ParseNodeList *new_specs = new ParseNodeList();

        for (Pix p = pnl->first(); p ; pnl->next(p)) {
                Module *m = (Module *)(*pnl)(p);
                switch (m->tag()) {
                        case Module::SpecModuleT: {
                                SpecModule *sm = (SpecModule *)m;
				ParseNodeList *specs = sm->get_specs();
                                if (specs) new_specs->append(specs->front());
                                break;
                                }
                        }
                }

        new_pnl->append(new SpecModule(new_specs));

        for (p = pnl->first(); p ; pnl->next(p)) {
                Module *m = (Module *)(*pnl)(p);
                switch (m->tag()) {
                        case Module::ImplModuleT: {
                                ImplModule *im = (ImplModule *)m;
                                new_pnl->append(im);
                                break;
                                }
                        }
                }
        return new_pnl;
        }

static string same_object_nm = 0;
static string x_nm = 0;
static string y_nm = 0;
TypeInterface *make_same_object_interface(type ct)
{
method m = new_method();

	if (same_object_nm == 0) {
		same_object_nm = string_new("same_object");
		x_nm = string_new("x");
		y_nm = string_new("y");
		}
	m->index = 0;
	m->iter = FALSE;
	m->parameterized = FALSE;
	m->name = same_object_nm;
	vec formals = make_vec_simple(class_as_type(Formal), 2);
	formal f = new_formal();
	f->name = x_nm;
	f->t = ct;
	vec_store(formals, 0, PV(f));
	f = new_formal();
	f->name = y_nm;
	f->t = objtype_as_type(Any);
	vec_store(formals, 1, PV(f));
	m->arguments = formals;
        m->extra_args = make_vec_simple(Type,0);

	vec rets = make_vec_simple(Type, 1);
	vec_store(rets, 0, PV(Bool));
	m->returns = rets;

	vec sigs = make_vec_simple(class_as_type(Signal), 0);
	m->signals = sigs;
        m->spec = string_const("null");

	TypeInterface *ti = new TypeInterface(m, 0);
	return ti;
	}
