#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common/fail.h"
#include "common/fe_or_msg.h"
#include "common/hostname.h"
#include "common/other_unix.h"
#include "common/client_fe.h"
#include "common/th_assert.h"
#include "fe_config.h"
#include "config/vdefs/SHM.h"
#include "client/handle.h"

fe_config FEConf;

static char usage[] = "Usage: %s [options below]\n"
		      "-s<cachesize>    cachesize in Kbytes\n" 
		      "-b<objsize>      average object size in bytes\n"
		      "-o<[host]:port>  OR location\n"
		      "-h<OR_hostname>  name of host running OR\n"
		      "-p<OR_port>      number of port where OR is listening\n"
                      "-P<FE_port>      number of port where FE will listen\n"
		      "-a               run in background\n"
                      "-c<Theta file>   compile early-Theta file\n"
  		      "-C<Theta file>   compile Theta file\n"
                      "-v<Theta file>   create veneer interface\n"
                      "-m<number>       maximum # of objects to prefetch\n"
                      "-D<debug_level>  Print debugging info\n"
                      "-i               Allow SIGIO to be delivered to FE\n"
		      "-l<language>	language to use for veneer interface\n"
		      "			Currently supported languages are "
		      			"'C-binary', 'C-ascii', 'C++' and 'Erlang'"
;


bool config_process_options (int argc, char* argv[]) {
  int opt;
  FEConf = new fe_config_s;
  int cachesize, objsize, portnum, length;
  char *hostname, *id, *fname;

  FEConf->cachesize = FE_DEFAULT_CACHE_SIZE;
  FEConf->avg_obj = FE_DEFAULT_AVERAGE_OBJ_SIZE;

  FEConf->OR_spec_initialized = FALSE;
  FEConf->OR_spec = new sockaddr_in;
  FEConf->OR_spec->sin_family = AF_INET;
  FEConf->OR_spec->sin_addr.s_addr = INADDR_ANY;
  FEConf->OR_spec->sin_port = htons(OR_DEFAULT_FE_PORT);

  FEConf->allow_sigio = FALSE;
  FEConf->FE_port = 0;
  FEConf->debug_level = 0;
  FEConf->slave = TRUE;
  FEConf->srcfile = 0;
  FEConf->theta_srcfile = 0;
  FEConf->veneer_srcfile = 0;
  FEConf->veneer_language = 0;
  FEConf->runInBackground = FALSE;
  FEConf->shm = FALSE;
  FEConf->shm_bufsize = 256;
  FEConf->max_prefetch = -1;      // Not a legitimate value

  FEConf->ht = new handle_table();
  FEConf->ft = new future_table();

  /* Have to reset "optind", which may have been set by using 
     "getopt" for the client options.  

     This will in turn cause problems if something tries processing
     FE options before this code is called.

     This is a good example of how global state can get you into trouble.
  */
  optind = 1;

  loop {
    if ((opt = getopt(argc, argv, "s:b:o:h:p:iaP:c:C:v:l:m:SD:")) != EOF) {
      switch (opt) {
      case 's':
	cachesize = atoi(optarg);
	if (cachesize > 0)
	  FEConf->cachesize = (cachesize * 1024);
	else {
	  warn("Couldn't interpret %s as a cache size", optarg);
	  return FALSE;
	}
	break;
      case 'b':
	objsize = atoi(optarg);
	if (objsize > 0)
	  FEConf->avg_obj = objsize;
	else {
	  warn("Couldn't interpret %s as an object size", optarg);
	  return FALSE;
	}
	break;
      case 'o':
	if (! findport(optarg, OR_DEFAULT_FE_PORT, FEConf->OR_spec)) {
	  warn("%s: invalid OR location", optarg);
	  return FALSE;
	}
	FEConf->OR_spec_initialized = TRUE;
	break;
      case 'h':
	warn("Option -h is obsolete.  Please switch to the -o option.");
	if (! findhost(optarg, &(FEConf->OR_spec->sin_addr))) {
	  warn("%s: invalid host name");
	  return FALSE;
	}
	break;
      case 'p':
	warn("Option -p is obsolete.  Please switch to the -o option.");
	portnum = atoi(optarg);
	if (portnum <= 0) {
	  warn("Couldn't interpret %s as a port number", optarg);
	  return FALSE;
	}
	FEConf->OR_spec->sin_port = htons(portnum);
	FEConf->OR_spec_initialized = TRUE;
	break;
      case 'P':
	portnum = atoi(optarg);
	if (portnum > 0) {
	  FEConf->FE_port = portnum;
	  FEConf->slave = FALSE;
	}
	else {
	  warn("Couldn't interpret %s as a port number", optarg);
	  return FALSE;
	}
	break;
      case 'a':
	FEConf->runInBackground = TRUE;
	break;
      case 'i':
	FEConf->allow_sigio = TRUE;
	break;
      case 'c':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        FEConf->srcfile = fname;
        break;
      case 'C':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        FEConf->theta_srcfile = fname;
        break;
      case 'v':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        FEConf->veneer_srcfile = fname;
        break;
      case 'l':
        FEConf->veneer_language = new char[strlen(optarg)];
	strcpy(FEConf->veneer_language, optarg);
        break;
      case 'm':
      {
        int number = atoi(optarg);
        if (number >= 0) {
          FEConf->max_prefetch = number;
          break;
        } 
        else {
          warn("Couldn't interpret %s as a non-negative integer", optarg);
          return FALSE;
        }
      }
      case 'S':
#if !SHM
	th_fail("Not configured to use shared memory");
#else
	FEConf->shm = TRUE;
#endif
	break;
      case 'D':
	FEConf->debug_level = atoi(optarg);
        break;
      default:
	warn(usage, argv[0]);
	return FALSE;
      }
    }
    else
      break;
  }

  if (! FEConf->OR_spec_initialized) {
	/* Look in environment */
	char* spec = getenv("THOR");
	if (spec != NULL) {
	    if (! findport(spec, OR_DEFAULT_FE_PORT, FEConf->OR_spec)) {
		    warn("%s: invalid OR location", spec);
		    return FALSE;
	    }
	    FEConf->OR_spec_initialized = TRUE;
	}
    }
    return TRUE;
  return TRUE;
}
