#include "common/basic.h"
#include "common/th_assert.h"
#include "common/Monitor.h"
#include "common/other_unix.h"

#include "config/vdefs/SHM.h"
#include "config/vdefs/COMPILER.h"
#include "config/vdefs/PROF_CLIENT_CONTROL.h"

#if !SHM
#include "client/ascii_interface.h"
#include "client/server.h"
#else
#include "client/binary_interface.h"
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

static ShmServer *server;

#endif

static void destroy_shared_mem(void *addr, int shmid);

#include "init_fe.h"
#include "fe_config.h"
#include "client/handle.h"
#include "cache/cache.h"
#include "types/str.h"
#include "boot/lex_input.h"

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

static void shutdown_fe(connection c, obj env) {
    exit(0);
}
static struct closure shutdown_cl = { (ifunc)shutdown_fe, 0 };
extern void finish_fe();
bool allow_meta;
int meta_size_alloc = 0;

#if SHM
static void *operator new(size_t s, void *addr)
{
    return addr;
}

static int shmid;
static void *mem;

static void destroy_shared_mem(void *addr, int shmid) {
    // This boolean is used to determine whether the shared memory has been
    // destroyed or not 

    static bool shm_present = TRUE;
    if (!shm_present)
	return;
#if !COMPILER
     fprintf(stderr, "FE: destroying shared memory region\n");
#endif
     shm_present = FALSE;
     server->remove_semaphores();
     shmdt((char *)addr);
     shmctl(shmid, IPC_RMID, 0);
}
#endif /* SHM */

static void catch_sig(int dummy)
{
    fprintf(stderr, "FE: caught signal, cleaning up.\n");
#if SHM
    destroy_shared_mem(mem, shmid);
#endif
    exit(EXIT_FAILURE);
}

void main(int argc, char *argv[])
{
    extern void catch_sigio(int dummy);

#if PROF_CLIENT_CONTROL
    stop_monitor();   // Turn off monitoring.
#endif

    allow_meta = TRUE;
    init_fe(argc, argv);
    atexit(finish_fe);
    signal(SIGHUP, catch_sig);
    signal(SIGTERM, catch_sig);
    signal(SIGSEGV, catch_sig);
    // Set up the handler for SIGIO (used for handling messages
    // from a server)
    signal(SIGIO, catch_sigio);

#if !SHM
    {
	/*
	   Time to start up the main loop. Create the server and
	   begin servicing requests.
	*/
	struct timeval tv;
	tv.tv_sec = 20;
	tv.tv_usec = 0;
	connection_setTimeout(tv);
	if (!connection_createSocket(FEConf->FE_port)) {
	    fprintf(stderr, "Could not start server.\n");
	    exit(1);
	}

	if (FEConf->slave) {
	    /* Running as slave.  Tell caller the real port number */
	    int port_number = connection_socketPort();
	    if (port_number < 0) {
		perror("finding socket port number");
		exit(1);
	    }
	    printf("%d\n", port_number);
	    fflush(stdout);

	    /* Set-up the FE to die when any connection closes */
	    connection_setCloseDisp(shutdown_cl);
	}

	fprintf(stderr, "Server started.\n");
	if (FEConf->runInBackground && fork() != 0) { exit(EXIT_SUCCESS); }
	connection_setStdDisp(ascii_cl);
	connection_setGreeting(string_new("0 Welcome to Thor FE version 0.01 :"
					  "  Type \"help\" for help.\n"));
	loop {
	    switch (connection_serviceRequests()) {
	      case keep_going:
	        break;
	      case all_idle:
		// handle_allow_performance_tuning();
                cache_force_gc(); 
		break;
	    }
	}
    }
#else
   {
         shmid = shmget(FEConf->FE_port, sizeof(ShmServer), IPC_CREAT | IPC_EXCL | 0777);
         if (shmid < 0) {
             fprintf(stderr, "FE: Couldn't get shared memory\n");
             exit(EXIT_FAILURE);
         }

	  // Must be mapped in the same address in clientXXXXX
	  mem = shmat(shmid, 0, 0);
	  if (-1 == (ptrdiff_t)mem) {
             fprintf(stderr, "FE: Couldn't attach shared memory\n");
             exit(EXIT_FAILURE);
         }
#if !COMPILER
         fprintf(stderr, "FE: Attached shared memory region %d at 0x%lx\n", shmid,
                 mem);
#endif
         server = new(mem) ShmServer(FEConf->shm_bufsize);
         server->beServer();
	 fprintf(stdout, "%d\n", shmid);
	 fflush(stdout);
	 initDispatch();
#if !COMPILER
	 fprintf(stderr, "FE: Entering dispatch loop\n");
#endif
	 binarySingleOp(server, NULL, NULL);
	 destroy_shared_mem(server, shmid);
	 exit(EXIT_SUCCESS);
   }
#endif
}
