/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _ALLOC_H
#define _ALLOC_H

#include "config/vdefs/WATCH_ALLOC.h"
#include <stddef.h>
#include "runtime/obj.h"
#include "common/basic.h"
#include "common/th_assert.h"

#ifdef __cplusplus
extern "C" {
#endif

    obj cache_oalloc(size_t);

#ifdef __cplusplus
}
#endif

/* See "cache/cache.h". Replicated here to break dependencies. */

#if WATCH_ALLOC
#include <stdio.h>
#define SHOW_LINE(kind) fprintf(stderr, "**ALLOC** %s %s:%d\n", (kind), __FILE__, __LINE__)
#endif

/*
   NEW(rep):
   Allocate memory. This macro will be replaced someday,
   but until then almost all allocation should be done through it.
*/

#define NEW_(rep) ((rep *)cache_oalloc(sizeof(rep)))
#if WATCH_ALLOC
#define NEW(rep) (SHOW_LINE("new"), NEW_(rep))
#else
#define NEW(rep) NEW_(rep)
#endif


/*
    NEWSTRING(len): 
    Allocate space for a string.  To use this macro, you must include
    "string_class.h". You probably don't want to use this macro anyway.
*/

#define NEWSTRING_(len) ((string)cache_oalloc(sizeof(struct string_s)+(size_t)(len)))
#if WATCH_ALLOC
#define NEWSTRING(len) (SHOW_LINE("newstring"), NEWSTRING_(len))
#else
#define NEWSTRING(len) NEWSTRING_(len)
#endif

/*
    NEWVEC(len):
    Allocate space for a "vec".  To use this macro, you must include
    "vec_class.h". You almost certainly don't want to use this macro anyway.
*/

#define NEWVEC_(len)  \
    ((vec)cache_oalloc((sizeof(struct vec_s) - sizeof(fevalue)) + \
		 (size_t)(len * sizeof(fevalue))))
#if WATCH_ALLOC
#define NEWVEC(len) (SHOW_LINE("newvec"), NEWVEC_(len))
#else
#define NEWVEC(len) NEWVEC_(len)
#endif

/* The following allocation macros have been added for meta-objects.
   Currently, the metaobjects are simply being allocated using malloc.
   Later, there will either be a seperate heap for them or they will be
   managed as an old generation in a generational garbage collection scheme
*/
extern int meta_size_alloc;

#define META_MALLOC(len) (th_assert(allow_meta, "Meta malloc disallowed"), meta_size_alloc += len, calloc(1, len))

#define NEW_META(rep) ((rep *)META_MALLOC(sizeof(rep)))

#define NEW_META_VEC(len) \
                ((vec)META_MALLOC((sizeof(struct vec_s) - sizeof(fevalue)) + \
		 (size_t)(len * sizeof(fevalue))))

#define NEW_META_STRING(len) ((string)META_MALLOC(sizeof(struct string_s)+(size_t)(len)))

/* A global that is used to indicate to various creation procedures whether
   they should create the object in the normal heap or the meta object heap
   It is declared in init_fe.cc
*/

extern bool normal_heap;
extern bool allow_meta;

/*
    THOR_FREE(obj):
    Deallocate memory.  Is a NOP.
*/

#define THOR_FREE(obj) 


/* 
    THOR_MALLOC(rep):
    Allocate non-object storage.  Maybe this will be done by a 
    mechanism other than malloc in the future, but it shouldn't
    be allowed to confuse the gc.
*/

#define THOR_MALLOC_(rep) ((rep *)malloc(sizeof(rep)))
#if WATCH_ALLOC
#define THOR_MALLOC(rep) (SHOW_LINE("malloc"), THOR_MALLOC_(rep))
#else
#define THOR_MALLOC(rep) THOR_MALLOC_(rep)
#endif


/* 
    MFREE(p):
    Free non-object storage allocated by THOR_MALLOC.  Should probably
    check first to make sure that the pointer isn't in the cache.
*/

#define MFREE(p) free(p)


#endif /* _ALLOC_H */
