/* Copyright Barbara Liskov, MIT 1996 */

#include "common/network.h"
#include "common/Timer.h"
#include "common/transaction.h"
#include "common/ros.h"
#include "common/mdebug.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/or_set.h"
#include "common/xrefs.h"
#include "common/uids.h"
#include "common/fe_or_msg.h"
#include "common/tstampgen.h"
#include "common/unparser.h"
#include "common/th_assert.h"

#include "cache/cache.h"
#include "cache/net.h"

#include "runtime/commit.h"
#include "runtime/transinfo.h"
#include "runtime/trans_set.h"
#include "fe_config.h"

static Tstamp_Generator gen;
static OR_set empty_or_set;  // To send to participants 

extern fe_num* fe_id;   // identifier of this FE

static commit_result commit_rpc(TransactionSet *tset, OR_num coordinator,
				TransEnv* tenv);
// modifies	tenv->xrefs, tenv->uids
// effects	tenv is the environment data required by the network
//              dispatching routines. tenv->nobjs is the pointer to
//              the set of newly persistent objects
//              Send transaction to be committed to ORs.
//		Wait for the commit reply.  If the transaction commits
//		successfully, fill in tenv->xrefs and tenv->uids of the newly
//		persistent objects in the passed in arrays.  These xrefs
//		and uids are filled in in the same order as the new objects
//		were inserted into the new object set.
//              XXX This will break with multiple ORs, since there are many
//              new object sets.

static commit_result wait_for_read_reply(OR_set *participants, TransEnv* tenv); 
// requires     Current transaction is read-only everywhere.
// modifies     participants
// effects      Waits for replies from given participants for the current
//              read-only transaction, and returns the result of the 
//              transaction.  On exit, participants is empty.
//              tenv is the environment structure required by the
//              network dispatching routine

void abort_trans() {
    Fe_Trans->abort_transaction(FALSE);
}

extern Timer make_timer, send_timer, recv_timer;
extern Timer cumm_make_timer, cumm_send_timer, cumm_recv_timer;
extern float make_time, send_time, recv_time;

commit_result commit_trans() {

    // Generate the object sets, and find coordinator.
    // If transaction has aborted then return
    if (Fe_Trans->must_abort()) {
	Fe_Trans->abort_transaction(FALSE);
	return STALE_READ_ABORT;
    }
    TransactionSet *tset = new TransactionSet(Fe_Trans->predicted_size());
    OR_num coordinator;
    cores* nobjs = new cores;

    cumm_make_timer.start();
    make_timer.reset(); make_timer.start();
    bool need_to_send_to_or =
	Fe_Trans->make_transaction(tset, &coordinator, nobjs);
    make_timer.stop(); make_time = make_timer.elapsed();
    cumm_make_timer.stop();
    cumm_send_timer.start();
    send_timer.reset(); send_timer.start();

    Xrefs* xrefs = new Xrefs;
    Uids*  uids  = new Uids;
    TransEnv tenv;
    tenv.nobjs = nobjs;
    tenv.xrefs = xrefs;
    tenv.uids = uids;
    tenv.committed = TRUE;

    commit_result res;
    if (need_to_send_to_or) {
	// Perform the commit RPC
	res = commit_rpc(tset, coordinator, &tenv);
    } else {
	send_timer.stop();
	cumm_send_timer.stop();
	Fe_Trans->complete_transaction(&tenv);
	res = COMMITTED;
    }

    delete nobjs;
    delete tset;
    delete xrefs;
    delete uids;

    return res;
}

static commit_result commit_rpc(TransactionSet *tset, OR_num coordinator,
				TransEnv* tenv)
{
    OR_num or_num;
    Network *net_coord;  // Network connection to coordinator
    fe_message msg;
    OR_set participants; // Set gets deleted when this procedure exits
    Transaction *trans;

    // Find time stamp for this transaction
    th_assert( fe_id != 0, "Uninitialized FE number");
    Tid tid = Tid(*fe_id, gen.generate());

    // Send appropriate prepare message to each participant
    tset->get_participants(&participants);
    OR_set::Elements gen(&participants);

    while (gen.get(or_num)) {
	trans = tset->get_transaction(or_num);
	th_assert(trans != 0, "Participant without matching transaction!");
	    
	Network *net = cache_get_OR_conn (or_num);
	if (net == 0) {
	    fprintf(stderr, "Couldn't find connection "
		    "to participant %d.\n", or_num);
	    fflush(stderr);
	    return FAILED_ABORT;  /* XXX Maybe we should recover... */
	}

	// Set Tid, coordinator, and participants	
	trans->tid = tid;
	trans->coordinator = coordinator;
	if (or_num == coordinator) {
	    net_coord = net;
	    trans->participants = &participants;
	    msg.msgtype = FE_PREPARE_COORD;
	}
	else {
	    // Participants don't need to know participant set
	    trans->participants = &empty_or_set;
	    msg.msgtype = FE_PREPARE_PART;
	}

	if (FEConf->debug_level > 1) {
	    unparser unp(0);
	    trans->unparse(&unp);
	}

	// Send the commit message
	if (! (msg.encode(net) && trans->encode(net) && net->flush())) {
	    perror("Commit message failed");
	    return FAILED_ABORT;
	}
    }

    // While we're waiting for reply, do something useful
    Fe_Trans->sent_transaction(tset, &participants);

    // Delete the individual transactions
    OR_set::Elements gen2(&participants);

    while (gen2.get(or_num))  {
	trans = tset->get_transaction(or_num);
	delete trans->ros;
	delete trans->mos;
	delete trans->nos;
	delete trans;
    }

    // If transaction is read-only everywhere, we will get votes directly
    // from participants.

    tenv->read_only = (coordinator == 0);

    send_timer.stop(); send_time = send_timer.elapsed();
    cumm_send_timer.stop();
    cumm_recv_timer.start();
    recv_timer.reset(); recv_timer.start();

    if (tenv->read_only) {
	if (FEConf->debug_level > 1) {
	    fprintf (stderr, "Transaction is read only everywhere.\n");
	}
	commit_result ans = wait_for_read_reply(&participants, tenv);
	recv_timer.stop(); recv_time = recv_timer.elapsed();
	cumm_recv_timer.stop();
	return ans;
    }

    // GET the commit/abort message from the coordinator
    OR_desc od;
    bool aborted;
    ubits32 msgtype[] = { OR_COMMITTED, OR_STALEABORT, OR_FAILABORT };
    od.or = coordinator;
    od.msgtype = msgtype;
    od.size = sizeof(msgtype)/sizeof(ubits32); // Should be 3 
    bool ok = read_from_net(&od, 1, FE_BLOCK, &aborted, FALSE,
			    tenv, WAIT_COMMIT);
    if (ok == FALSE)
	return FAILED_ABORT;
    recv_timer.stop(); recv_time = recv_timer.elapsed();
    cumm_recv_timer.stop();
    return (tenv->committed) ? COMMITTED: STALE_READ_ABORT;
}

static commit_result wait_for_read_reply(OR_set *participants, TransEnv* tenv) {
    OR_num or_num;
    // XXX This code has to be debugged (the new communcation style)

    // Create a OR_desc for waiting on all the participants
    ubits32 msgtype[] = { OR_READ_OK, OR_STALEABORT, OR_FAILABORT};
    int msgtype_size = sizeof(msgtype)/sizeof(ubits32);
    int ods_size = participants->size();
    OR_desc* ods = new OR_desc[ods_size];
    OR_set::Elements gen(participants);
    int i = 0;
    while(gen.get(or_num)) {
	ods[i].or = or_num;
	ods[i].msgtype = msgtype;
	ods[i].size = ods_size;
	i++;
    }
    bool aborted;
    // Note: The code will wait for all the participants before replying
    bool ok = read_from_net(ods, ods_size, FE_BLOCK, &aborted,
			    FALSE, tenv, WAIT_COMMIT);
    delete [] ods;
    if (ok == FALSE)
	return FAILED_ABORT;
    return (tenv->committed)? COMMITTED: STALE_READ_ABORT;
}
