/* Copyright Barbara Liskov, MIT 1995, 1996 */

#include <stdlib.h>
#include "objtype_class.h"

/* This file implements the header layout algorithm described in
   "Bidirectional Object Layout for Separate Compilation,"
   Andrew C. Myers, submitted to OOPSLA '95.
*/

dh_template type_dh_template(objtype t);
dh_template class_dh_template(class c);

dh_template get_dh_template(objtype t)
/*
  Returns the dh_template for type t. 
  Computes it, if not already computed.
*/
{
    if (0 != t->dht) return t->dht;
    int kind = type_kind(t);

    switch(kind) {
	case PRIMITIVE_KIND:
	    abort();
	case OBJECT_KIND:
	case INSTN_KIND:
	    return type_dh_template(t);
	case CLASS_KIND:
	case CLASS_INSTN_KIND:
	    return class_dh_template(BUMP(t, class, t->hdr.methods));
    }
}

bool same_method(method m1, method m2)
/*
    Return whether method "m1" is really the same method that "m2" is.
*/
{
    if (m1 == m2) return TRUE;
    objtype t1 = method_self_type(m1);
    objtype t2 = method_self_type(m2);
}

bool dv_match(objtype t1, objtype t2, dv_template dv1, dv_template dv2)
/* Report whether the dispatch vector "dv1" is embedded within 
   "dv2", where "dv1" is a dispatch vector of type "t1",
   and "dv2" is a dispatch vector of type "t2".
*/
{
    int i, n1 = vec_length(dv1), n2 = vec_length(dv2);
    if (n1 > n2) return FALSE;
    for (i = 0; i < n1; i++) {
	if (!same_method(t1, t2, UNPV(method, vec_fetch(dv1, i)),
			 UNPV(method, vec_fetch(dv2, i)))) return FALSE;
    }
    return TRUE;
}

/* Report whether the type header for "t1" can be embedded within that
   of "t2". Does not report the offset to the embedded header, though it
   could. */
bool lth(objtype t1, objtype t2)
{
    dh_template dh1 = type_dh_template(t1);
    dh_template dh2 = type_dh_template(t2);

    int L1 = vec_length(dh1), L2 = vec_length(dh2);

/* test various offsets at which the t1 header might be embedded within
   the t2 header. */

    for (int i = 0; i <= L2 - L1; i++) {
	bool compatible = TRUE;
	for (int j = 0; compatible && j < L1; j++) {
	    if (!dv_match(t1, t2, dv_fetch(dh1, j), dv_fetch(dh2, i + j))) {
		compatible = FALSE;
	    }
	}
	if (compatible) return TRUE;
    }
    return FALSE;
}

vec /* [objtype] */ topological_sort(vec /* [objtype] */ types) {
/*
   Sort the types according to the "header containment" relation, "lth". This
   is basically an insertion sort, but it needs to be a little more
   careful than usual because not all elements are comparable.
*/
    int i,j,n;
    objtype *ts = VEC_ITEMS(types, objtype);
    n = vec_length(types);
    for (i = 1; i < n; i++) {
	objtype *t = ts[i];
	for (j = 0; j < i; j++) if (lth(ts[j], t)) break;
	int k;
	for (k = i; k > j; k--) ts[k] = ts[k-1];
	ts[j] = t;
    }
}

dh_template type_dh_template(objtype t)
{
    vec sts = vec_copy(t->supertypes_);

    sts = topological_sort(sts);

    objtype *sts = (objtype *)vec_items(t->supertypes_);
    int nsts = vec_length(t->supertypes_);

    
  /* (1) compute the size of dh */
  int dht_size = 0;
  objtype sct = 0;
  if (t->kind == CLASS_KIND) {
    class_ c = FORCE(t, class_, Class);
    if (vec_length(c->superclass)) {
      class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
      sct = class_as_objtype(sc);
      dht_size += vec_length(get_dh_template(class_as_objtype(sc)));
    }
  }
  for (int i=0; i<nsts; i++) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dht_size += vec_length(get_dh_template(sts[i]));
  }
  if (0 == dht_size) dht_size = 1;

  /* (2) Create the dh template. */
  dh_template dht = make_vec_Any(dht_size);
  t->dht = dht;

  /* (3) fill up the dh */
  int dht_index = 0;
  int total_methods = 0;
  /* (3.a) copy the dh of each supertype, except that identical to sct */
  for (i=nsts-1; i>=0; i--) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dh_template st_dht = sts[i]->dht;
    int st_dht_size = vec_length(st_dht);
    dv_template *st_dvts = (dv_template *)vec_items(st_dht);
    for (int j=0; j<st_dht_size; j++) 
      vec_store(dht, dht_index++, (obj) st_dvts[j]);
    total_methods += vec_length(st_dvts[st_dht_size-1]);
  }
  /* (3.b) copy the dh of the superclass */
  // XXX it should be possible to avoid this in most cases!
  if (sct) {
    dh_template st_dht = sct->dht;
    int st_dht_size = vec_length(st_dht);
    dv_template *st_dvts = (dv_template *)vec_items(st_dht);
    for (int j=0; j<st_dht_size; j++) 
      vec_store(dht, dht_index++, (obj) st_dvts[j]);
    total_methods += vec_length(st_dvts[st_dht_size-1]);
  }
  th_assert ((dht_index == 0 && dht_size == 1) || (dht_index == dht_size), 
	     "computation of dht_size");

  /* (3.c) replace the core (last) dv of t by a concatenation of all methods
     in the core dv's of t's superclass and supertypes, 
     followed by t's local methods. */ 
  total_methods += vec_length(t->methods_);
  dv_template core = make_vec_Any(total_methods);
  vec_store(dht, dht_size-1, (obj) core);
  int core_index = 0;
  /* (3.c.1) concat the methods of each supertype, except that = sct */
  for (i=0; i<nsts; i++) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dh_template st_dht = sts[i]->dht;
    dv_template st_core = (dv_template)vec_fetch(st_dht, vec_length(st_dht)-1);
    any *st_methods = (any *)vec_items(st_core);
    int nst_methods = vec_length(st_core);
    for (int j=0; j<nst_methods; j++) 
      vec_store(core, core_index++, (obj) st_methods[j]);
  }
  /* (3.c.2) concat the methods of the superclass */
  if (sct) {
    dh_template st_dht = sct->dht;
    dv_template st_core = (dv_template)vec_fetch(st_dht, vec_length(st_dht)-1);
    any *st_methods = (any *)vec_items(st_core);
    int nst_methods = vec_length(st_core);
    for (int j=0; j<nst_methods; j++) 
      vec_store(core, core_index++, (obj) st_methods[j]);
  }
  /* (3.c.3) concat the local methods */
  any *local_methods = (any *)vec_items(t->methods_);
  int nlocal_methods = vec_length(t->methods_);
  for (int j=0; j<nlocal_methods; j++) 
    vec_store(core, core_index++, (obj) local_methods[j]);
  th_assert (core_index == total_methods, "computation of total_methods");

  return dht;
}

dh_template class_dh_template(class c)
{
}
