/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _DISCOVER_FIX_H
#define _DISCOVER_FIX_H

#include "config/vdefs/EDGE_MARKING.h"

#include "runtime/obj_class.h"

/* These are the macros that ensure that objects are actually present
   and available for field accesses.  Their implementations depend
   on how the system is doing swizzling.

   Both FIX and DISCOVER take arguments (s,t). "s" is some lvalue that
   contains the pointer to be fixed. Its C type must be "t". After the macro
   call, "s" will contain the updated pointer, as appropriate.

   Here's when to use them:

   1. DISCOVER

      WARNING: This macro is currently underused in translated Theta code.
      Don't use existing code as a good example of when you should use it;
      you should probably use it more.

      When you first refer to a field of an object that contains 
      another object, you must DISCOVER it.  For example, assume that 
      "self" has a field "x" that you know is a T object, and you
      (or the compiler) has written the following code which is the
      first time in the method that you've referred to x:

      T foo = self.x;

      Instead you (or the compiler) should write

      DISCOVER(self.x, T)
      T foo = self.x;

      Note that you should never DISCOVER "self" in a method, since
      the currently-executing object is guaranteed to be a non-surrogate.
      You should not discover any of the other arguments of the method
      either.

      See also DISCOVER_FAST, below.

  2. FIX 

     WARNING: This macro is currently used way too much in translated
     Theta code.  It should be used only rarely.

     When you first refer to an object whose state you don't know,
     for purposes of comparing pointers, or for direct access to
     the object's fields, you FIX it. Since you always know the state
     of "self", the most common use of FIX is to access fields of
     a secondary method argument that has been type-cased down to the
     type of "self".

     For example, assume
     that the method being implemented is a comparison between "self" and
     some other object "y". Instead of writing 

     return self.id == y.id;

     you would write

     FIX(y, T)
     return self.id == y.id;

     Note that FIX _only_ applies to field accesses and pointer comparisons.
     If you are invoking a method on an object, you do not FIX it first.
     Since most field accesses are to "self", FIX is rarely used.

     As with DISCOVER, you should never FIX "self" in a method,
     since the currently-executing object is guaranteed to be a non-surrogate.
     You may need to FIX arguments to the method, however.

  2a. FIX_FAST

     WARNING: This macro is basically obsolete.

     If you happen to know that the object type T has only a single 
     implementation, then you can write the previous example as

     FIX_FAST(y, T, T_methods)
     return self.id == y.id;

     However, you will only know this information (and what name to 
     use for "T_methods") if you are inside the implementation of T.

  2b. FIX + DISCOVER
     It is reasonable to FIX an object and DISCOVER its fields:

     FIX(y, T)
     DISCOVER(self.x)
     DISCOVER(y.x)
     compare(self.x, y.x);

     However, it is never necessary to both FIX and DISCOVER the same
     entity.  It is wasteful to do so, since FIX and DISCOVER are defined
     in terms of each other (depending on the swizzling technique used)
     so that 

     FIX(x)
     DISCOVER(x)

     either becomes two FIXes or two DISCOVERs, but in either case
     simply wastes time.

*/

#if EDGE_MARKING
#define FIX_FAST(s,t,m) FIX(s, t)
#else /* usual node marking */
#define FIX_FAST(s,t,m) { STATS(stats->fixes++); \
                     if (*((obj)(s)) != (DV)&(m)) \
			  s = (t)get_address((obj)(s)); }
#endif
/*
   The "FIX_FAST" macro takes a variable "s" of type "t", where "t" has
   only one implementation, and a dispatch header size of 1.  "m" must
   the corresponding dispatch vector for that implementation. "FIX_FAST"
   checks whether "s" refers to a surrogate, and if so, changes its
   value to refer to a real object of type "t". 

   This macro is basically obsolete.
*/

#if EDGE_MARKING 
#define FIX(s,t) DISCOVER(s, t)
#else 
#define FIX(s,t) { STATS(stats->fixes++); \
                     if (IS_SPECIAL((obj)(s))) \
                        s = (t)get_address((obj)(s)); }
#endif
/*
   The "FIX" macro has the same effect as "FIX_FAST", but it does not
   require that "t" has only one implementation. Also, no knowledge
   about the dispatch table is needed. However, "FIX" is slower than
   "FIX_FAST".

   If there is a possibility that an object is a surrogate, either
   "FIX_FAST" or "FIX" must be called on it before any fields of the
   object are accessed. There is no need to call "FIX" on an object on
   which only method calls are being performed.

   "FIX" treats the variable "s" as an lvalue, potentially overwriting
   it with the real object pointer. Therefore, if "s" is an object
   field, further accesses to the object will not result in swizzling.

   See also "DISCOVER", below. Once "FIX" has been used, a "DISCOVER"
   is not needed. 
*/

#if EDGE_MARKING
#define DISCOVER(s,t) { STATS(stats->discovers++); 			\
                        if (IS_MARKED_POINTER(s)) 			\
		            s = (t)cache_fetch_pointer((obj)(s), 0);}
#else
#define DISCOVER(s,t) 
#endif
/*
   The "DISCOVER" macro is similar to "FIX", but it must be used before
   any method calls are performed on an object. If fields of the object
   are also going to be accessed, the "FIX" method should be used instead.

   For performance reasons, it is a good idea to call "DISCOVER" at
   the first point where it is clear that the object referred to by a field
   will have methods invoked on it. Thus, the field can be overwritten
   with a valid pointer that further calls to "DISCOVER" will quickly
   skip over.
*/

#if EDGE_MARKING
#define DISCOVER_FAST(s,t,tmp) 						\
    (STATS_EXPR(stats->discovers++),					\
     (tmp) = (s),							\
     (IS_MARKED_POINTER(tmp) ?						\
	 ((s) = (t)cache_fetch_pointer((obj)(s), 0)) : (tmp)))
#else
#define DISCOVER_FAST(s,t,tmp) s
#endif
/*
   The "DISCOVER_FAST" macro is a faster substitute for "DISCOVER".  It
   is an expression that returns the discovered pointer.

   The extra argument "tmp" must be a variable of type "t".
   Its value will be undefined after the macro call.

   For example,

       DISCOVER(s,t)
       x = f(s);

   becomes

       t tmp;
       ...
       x = f(DISCOVER_FAST(s,t,tmp));

*/

#endif
