/* Copyright Barbara Liskov, MIT 1996 */

/* Stubs for dispatch header functions */

#include "runtime/disphdr.h"
#include "runtime/surr.h"
#include "common/iter.h"
#include "runtime/alloc.h"
#include "runtime/force.h"
#include "runtime/stats.h"
#include "types/type.h"
#include "types/class_class.h"
#include "types/vec_class.h"
#include "types/vec_instns.h"
#include "types/string_class.h"
#include "types/method.h"
#include "types/class_instn.h"
#include "types/ptype.h"
#include "common/th_assert.h"
#include "cache/gc_register.h"
#include "cache/swiz.h"
#include "config/vdefs/ADAPT_PREFETCH.h"
#include <stdio.h>
#include <stdlib.h>

extern "C" {
void *memcpy(void*, const void*, size_t);
#include "types/pclass.h"
}

#define DHENTRY_SIZE sizeof(DV)


#if ADAPT_PREFETCH
// Global variable that holds the statistic on the number of objects used.
// Defined in net.cc
extern long p_nused;
#define INCR_NUSED() p_nused++
#else
#define INCR_NUSED()
#endif


/* Special dispatch headers are provided for the following. */
enum ObjKind {EMPTY_SURR, FULL_SURR, UNSWIZZLED};

/* The procedure "inst_obj_hdr" is implemented in "obj.c". */

#if 0
void fill_surr_hdr (surr s) {
    objtype t = (*s)->t;
    int dhsize = hdr_size(t);
    th_assert(dhsize > 0, "Nonsense size for dispatch header");
    memcpy(s, t->full_surr_DH, dhsize * DHENTRY_SIZE);
}
#endif

#define SURR_METHOD(N) \
obj surr_method_##N(obj o, obj arg1, obj arg2, obj arg3, obj arg4, obj arg5, \
                    obj arg6, obj arg7, obj arg8, obj arg9) \
{   surr s = o; \
    core c; \
    DV __dv; \
    dtentry *dte; \
    obj (*f)(core, ...); \
    o = cache_fetch(s, 0); \
    th_assert(!(o == s), "Fetched object same as surrogate"); \
    STATS(stats->empty_surrogate_uses++) \
    __dv = *o; \
    dte = (dtentry *)(&__dv->get_class) + N; \
    c = BUMP(core, o, __dv); \
    f = (obj(*)(core, ...))(dte->f); \
    return (*f)(c, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9); \
}

// To use full surrogate methods implemented in assembly,
// set the argument of "#if" to 0. Also see "full_surr_methods.S".

#include "config/vdefs/ASM_SURR_METHODS.h"
#if ASM_SURR_METHODS
#define FULL_SURR_METHOD(N) \
extern "C" obj full_surr_method_##N(obj o, obj arg1, obj arg2, obj arg3, \
	obj arg4, obj arg5, obj arg6, obj arg7, obj arg8, obj arg9);
#else
#define FULL_SURR_METHOD(N) \
obj full_surr_method_##N(obj o, obj arg1, obj arg2, obj arg3, obj arg4, \
                         obj arg5, obj arg6, obj arg7, obj arg8, obj arg9) \
{   obj s = o; core c; \
    DV __dv; \
    dtentry *dte; \
    obj (*f)(core, ...); \
    o = *SURR_FORWARD(s); \
    th_assert(o, "Surrogate points to null object"); \
    th_assert(!(o == s), "Surrogate pointing to itself"); \
    STATS(stats->surrogate_uses++) \
    __dv = *o; \
    dte = (dtentry *)(&__dv->get_class) + N; \
    c = BUMP(core, o, __dv); \
    f = (obj(*)(core, ...))(dte->f); \
    return (*f)(c, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9); \
}
#endif

#define UNSWIZZLED_METHOD(N) \
obj unswizzled_method_##N(obj o, obj arg1, obj arg2, obj arg3, obj arg4, \
                         obj arg5, obj arg6, obj arg7, obj arg8, obj arg9) \
{ \
    core c = BUMP(core, o, *o); \
    class_ cl = (class_)(*o)->t; \
    INCR_NUSED();  \
    o = OBJ_START(o); \
    th_assert(o, "unswizzled object is NIL"); \
    cache_prim_swizzle(o, c->xref.or, (OR_obj *) (((cacheval *)c)+1), cl);\
    dtentry *dte; \
    obj (*f)(core, ...); \
    dte = (dtentry *)(&(*o)->get_class) + N; \
    f = (obj(*)(core, ...))(dte->f); \
    return (*f)(c, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9); \
}

#undef ASM_METHODS_FILE
#include "special_methods.h"

dh_template get_dh_template(objtype t)
/*
  Returns the dh_template for type t. 
  Computes it, if not already computed.
*/
{
  bool saved = normal_heap;
  if (0 != t->dht) return t->dht;
  normal_heap = FALSE;
  bool saved_allow_meta = allow_meta;
  allow_meta = TRUE;
  
  objtype *sts = (objtype *)vec_items(t->supertypes_);
  int nsts = vec_length(t->supertypes_);

  /* (1) compute the size of dh */
  int dht_size = 0;
  objtype sct = 0;
  if (t->kind == CLASS_KIND) {
    class_ c = FORCE(t, class_, Class);
    if (vec_length(c->superclass)) {
      class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
      sct = class_as_objtype(sc);
      dht_size += vec_length(get_dh_template(class_as_objtype(sc)));
    }
  }
  if (t->kind == CLASS_INSTN_KIND) {
    class_instn ci = type_as_class_instn(objtype_as_type(t));
    class_ c = class_instn_as_class(ci);
    if (vec_length(c->superclass)) {
      class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
      sct = class_as_objtype(sc);
      dht_size += vec_length(get_dh_template(class_as_objtype(sc)));
    }
  }
  for (int i=0; i<nsts; i++) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dht_size += vec_length(get_dh_template(sts[i]));
  }
  if (0 == dht_size) dht_size = 1;

  /* (2) Create the dh template. */
  dh_template dht = make_vec_Any(dht_size, normal_heap);
  t->dht = dht;

  /* (3) fill up the dh */
  int dht_index = 0;
  int total_methods = 0;
  /* (3.a) copy the dh of each supertype, except that identical to sct */
  for (i=nsts-1; i>=0; i--) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dh_template st_dht = sts[i]->dht;
    int st_dht_size = vec_length(st_dht);
    dv_template *st_dvts = (dv_template *)vec_items(st_dht);
    for (int j=0; j<st_dht_size; j++) 
      vec_store(dht, dht_index++, (obj) st_dvts[j]);
    total_methods += vec_length(st_dvts[st_dht_size-1]);
  }
  /* (3.b) copy the dh of the superclass */
  // XXX it should be possible to avoid this in most cases!
  if (sct) {
    dh_template st_dht = sct->dht;
    int st_dht_size = vec_length(st_dht);
    dv_template *st_dvts = (dv_template *)vec_items(st_dht);
    for (int j=0; j<st_dht_size; j++) 
      vec_store(dht, dht_index++, (obj) st_dvts[j]);
    total_methods += vec_length(st_dvts[st_dht_size-1]);
  }
  th_assert ((dht_index == 0 && dht_size == 1) || (dht_index == dht_size), 
	     "computation of dht_size");

  /* (3.c) replace the core (last) dv of t by a concatenation of all methods
     in the core dv's of t's superclass and supertypes, 
     followed by t's local methods. */ 
  total_methods += vec_length(t->methods_);
  dv_template core = make_vec_Any(total_methods, normal_heap);
  vec_store(dht, dht_size-1, (obj) core);
  int core_index = 0;
  /* (3.c.1) concat the methods of each supertype, except that = sct */
  for (i=0; i<nsts; i++) {
    if (sct == sts[i]) continue; // XXX use type_equal here
    dh_template st_dht = sts[i]->dht;
    dv_template st_core = (dv_template)vec_fetch(st_dht, vec_length(st_dht)-1);
    any *st_methods = (any *)vec_items(st_core);
    int nst_methods = vec_length(st_core);
    for (int j=0; j<nst_methods; j++) 
      vec_store(core, core_index++, (obj) st_methods[j]);
  }
  /* (3.c.2) concat the methods of the superclass */
  if (sct) {
    dh_template st_dht = sct->dht;
    dv_template st_core = (dv_template)vec_fetch(st_dht, vec_length(st_dht)-1);
    any *st_methods = (any *)vec_items(st_core);
    int nst_methods = vec_length(st_core);
    for (int j=0; j<nst_methods; j++) 
      vec_store(core, core_index++, (obj) st_methods[j]);
  }
  /* (3.c.3) concat the local methods */
  any *local_methods = (any *)vec_items(t->methods_);
  int nlocal_methods = vec_length(t->methods_);
  for (int j=0; j<nlocal_methods; j++) 
    vec_store(core, core_index++, (obj) local_methods[j]);
  th_assert (core_index == total_methods, "computation of total_methods");

  normal_heap = saved;
  allow_meta = saved_allow_meta;
  return dht;
}


obj surr_get_address(obj s)
{
/*
   XXX This implementation needs to offset the result from cache_fetch,
       else cache_fetch needs to.
*/
  
    return cache_fetch(s, 0);
}

obj full_surr_get_address(obj s)
{
  obj forward = *SURR_FORWARD(s);
  /* Invoke the getAddress method on forward */
  return (*forward)->get_address(forward);
}


obj unswizzled_get_address(obj o)
{
  core c = BUMP(core, o, *o); \
  o = OBJ_START(o);
  cache_swizzle_object(c->xref.or, o);
  return (o);
}

bool is_unswizzled(obj o) {
  DV dv = *o;
  return (dv->get_address == unswizzled_get_address);
}

DV new_dv(int num_methods)
{
    DV ret = (struct dv_s *)malloc(sizeof(struct dv_s) + (num_methods-1) *
                                 sizeof(dtentry *));
    return ret;
}

DV *create_special_dh(objtype t, enum ObjKind objkind)
/* Creates a returns a dispatch header for objtype t and for the
   special kind of object indicated by objkind. */
{
  dh_template dht = get_dh_template(t);
  int dh_size = vec_length(dht);
  dv_template *dvts = (dv_template *)vec_items(dht);
  DV *dh = (DV *)malloc(dh_size*sizeof(DV));

  int boffset = 0;
  int offset = (dh_size-1)*sizeof(DV);
  int foffset = (dh_size)*sizeof(DV);

  for (int i=0; i<dh_size; i++) {
    dv_template dvt = dvts[i];
    int dv_size = vec_length(dvt);

    dv_size = 32;
    // XXX This is a quick fix so that surrogate 
    // dv's don't have to be updated on type refinement.
    // See thd note on Aug 4, 1995.
    
    DV dv = new_dv(dv_size);
    dh[i] = dv;
    dv->boffset = boffset-i;
    dv->offset = offset-i;
    dv->t = t;
    dv->c = 0;
    dtentry *des = (dtentry *)&dv->get_class;
    int j;
    switch (objkind){
    case EMPTY_SURR:
      dv->foffset = 0; 
      dv->get_address = surr_get_address; 
      for (j = 0; j < dv_size; j++) {
	des[j].f = surr_methods[j];
      }
      break;
    case FULL_SURR:
      dv->foffset = foffset-i;
      dv->get_address = full_surr_get_address; 
      for (j = 0; j < dv_size; j++) {
	des[j].f = full_surr_methods[j];
      }
      break;
    case UNSWIZZLED:
      dv->foffset = 1;
      dv->get_address = unswizzled_get_address; 
      for (j = 0; j < dv_size; j++) {
	des[j].f = unswizzled_methods[j];
      }
      break;
    };
  }
  return dh;
}  

int init_hdr_size(objtype t)
/*
   "hdr_size" computes the proper header size for this type, if
   it hasn't already been computed, and makes sure that the special
   dispatch header fields of "t" contain appropriate dispatch headers.

   Note that if any of the supertypes is actually the same as the
   superclass, its dispatch header is folded in with the superclass's.
   This is the purpose of the variable "sct".
*/
{
  if (t->hdr_size >= 0) return t->hdr_size;
  t->hdr_size = vec_length(get_dh_template(t));

  t->surr_DH = create_special_dh (t, EMPTY_SURR);
  t->full_surr_DH = create_special_dh (t, FULL_SURR);
  t->unswizzled_DH = create_special_dh (t, UNSWIZZLED);

  return t->hdr_size;
}

type get_surr_type(surr s)
{
    return objtype_as_type((*s)->t);
}

int hdr_offset(objtype t, objtype st)
{
    int offset = 0;
    int i;
    int num_supertypes = vec_length(t->supertypes_);
    if (st == t) { /* XXX use type_equal here */
        return 0;
    }
    if (isSubtype(objtype_as_type(t), objtype_as_type(st)) &&
	isSubtype(objtype_as_type(st), objtype_as_type(t))) {
      /* Must be "type_equal" */
      return 0;
    }
    if ((t->kind  == CLASS_INSTN_KIND) &&
	(st->kind == PCLASS_KIND)) {
        pclass pc1 = class_instn_get_pclass(
			type_as_class_instn(
		           objtype_as_type(t)));
	// Check that pc is the "same" as st.
	ptype pt2 = type_as_ptype(
		           objtype_as_type(st));
        pclass pc2 = ptype_as_pclass(pt2);
	if (pc1 == pc2) {
	  return 0;
	}
    }
    if (num_supertypes) {
	i = hdr_offset(UNPV(objtype, vec_fetch(t->supertypes_,
					    num_supertypes - 1)), st);
	if (!exc) return i;
	exc = EXC_NONE;
    }
    if (t->kind == CLASS_KIND) {
	class_ c = FORCE(t, class_, Class);
	if (vec_length(c->superclass)) {
	    class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
	    i = hdr_offset(class_as_objtype(sc), st);
	    if (!exc) return i;
	    exc = EXC_NONE;
	}
    }
    if (t->kind == CLASS_INSTN_KIND) {
	class_instn ci = type_as_class_instn(objtype_as_type(t));
	class_ c = class_instn_as_class(ci);
	if (vec_length(c->superclass)) {
	    class_ sc = UNPV(class_, vec_fetch(c->superclass, 0));
	    i = hdr_offset(class_as_objtype(sc), st);
	    if (!exc) return i;
	    exc = EXC_NONE;
	}
    }
    th_assert (num_supertypes < 2, "Complex type headers unimplemented");
    /* We would actually have to do some offset computations here;
       currently we only handle the trivial cases since that's all
       we expect to run into. */
    exc = &exc_not_possible;
}

int objtype_num_methods (objtype t) 
/* 
   Return the total number of methods for an object of type t,
   including those introduced by t's supertypes.
*/
{
  dh_template dht = get_dh_template(t);
  dv_template dvt = (dv_template)vec_fetch(dht, vec_length(dht)-1);
  return vec_length(dvt);
  // XXX this information should be memoized in t for efficiency
}

int direct_methods_size(objtype t)
/*
  Return the number of direct methods of "t".
  XXX The usage of this should be replaced by "objtype_num_methods"
  once the dh_template is used for all dispatch headers.
  */
{
    int size = 0;
    class_ c;
    if (t->kind == CLASS_KIND) {
	c = (class_)t;
	if (vec_length(c->superclass))
	  size = direct_methods_size
	    (class_as_objtype(UNPV(class_,
				vec_fetch(c->superclass, 0))));
    }
    if (t->kind == CLASS_INSTN_KIND) {
	class_instn ci = type_as_class_instn(objtype_as_type(t));
	c = class_instn_as_class(ci);
	if (vec_length(c->superclass))
	  size = direct_methods_size
	    (class_as_objtype(UNPV(class_,
				vec_fetch(c->superclass, 0))));
    }
    if (size == 0) {
        if (vec_length(t->supertypes_))
          size = direct_methods_size(UNPV(objtype,
                                          vec_fetch(t->supertypes_, 0)));
    }
    th_assert(!exc, "not implemented yet: complex type headers");
    return size + vec_length(t->methods_);
}
