/* Copyright Barbara Liskov, MIT 1996 */

#include "common/basic.h"
#include "common/or_num.h"
#include "common/th_assert.h"
#include "common/xref_set.h"
#include "common/mdebug.h"

#include "fe/cache/swiz.h"

#include "fe_config.h"
#include "obj.h"
#include "obj_class.h"
#include "transinfo.h"
#include "invalidation.h"
#include "except.h"

#define DEBUG_INVALIDATION FALSE

// This file handles the invalidation messages recevied by a frontend
// The Transinfo methods have also been put here for the sake of separating
// out the invalidation code.

extern "C" {
    bool cache_obj_invalidate (obj o);
}

or_inv_info* Transinfo::find_inv_info(OR_num or_num) {
    for (int i = 0; i < inv_info->size(); i++)
	if (inv_info->slot(i)->or_num == or_num)
	    return inv_info->slot(i);

    // Create new invalidation information
    or_inv_info *info = new or_inv_info;
    info->or_num = or_num;
    info->last_inv_num = 0;
    inv_info->append(info);
    return info;
}

bool Transinfo::invalidate_objects(OR_num or, const Oref* inv_orefs,
				   int setsize, ubits32 msg_start,
				   ubits32 msg_end, Msg_Wait_Type wait_type) {

    // We need not check if a fetched object is being invalidated.
    // The cache fetch code checks after its fetch request whether
    // the object is present in the cache, otherwise it sends
    // a another fetch request

    or_inv_info *inv_info = find_inv_info(or);
    th_assert(inv_info != 0, "Couldn't find invalid object set");
    th_assert(inv_info->last_inv_num == msg_start - 1,
	      "Wrong invalidation message number received");
    th_assert(msg_end >= msg_start, "Bad invalidation message number range");
    th_assert(setsize > 0, "Invalidation message without any orefs");
    inv_info->last_inv_num = msg_end;

    // Add the objects to the hash table for the read set
    int start_index = hashed_index + 1;
    int pers_size = pxrefs.size();
    for (int x = start_index; x < pers_size; x++) {
	read_set->insert(pxrefs[x]);
    }
    hashed_index = pers_size - 1;

    for (int i = 0; i < setsize; i++) {
	Xref xref;
	xref.or = or;
	xref.oref = inv_orefs[i];
	// If object has been read, this transaction has to abort
	// This is checked by first checking if the actual is marked read.
	// If not (or it is a surrogate), we lookup the shrunk_set

	// Note: Object may not be in swizzle table. So do not
	// call swiz_get_snap
	obj o = swiz_get_object(xref);
	// Important: If we are waiting for a commit reply, then the
	// stamps in the objects have been cleared. So even if a
	// transaction should be aborted, this code will not be able to
	// determine it. There are no correctness problems since the OR
	// will fail the transaction (the ack reaches the OR after the
	// commit message) 
	// XXX Needs to be fixed
	if (o && is_read(o) || read_set->contains(xref)) {
	    Abort_Transaction = TRUE;
	}
	// If object is present, shrink it
	if (o) {
	    cache_obj_invalidate(o);
#if DEBUG_INVALIDATION
	    core c = BUMP(core, o, *o);
	    fprintf(stderr, "Invalidated object (Modified = %d)  %d:%d.%d\n",
		    IS_WRITTEN_CORE(c), xref.or,
		    OREF_SEGMENT(xref.oref), OREF_INDEX(xref.oref));
#endif
	}
    }

    Abort_Transaction = Abort_Transaction || immediate_abort_on_invalidation;
    if (FEConf->debug_level > 0)
	fprintf (stderr, "Invalidation: Type = %d, Aborted = %d\n",
		 wait_type, Abort_Transaction);
    return Abort_Transaction;
}
