/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _OBJ_CLASS_H
#define _OBJ_CLASS_H

/*
    This header file defines the structure of an object and its
    dispatch tables. It also provides macros and procedures
    for manipulating objects and casting between various object
    representations.

    This file should be considered in conjunction with "runtime/obj.h",
    which contains some basic definitions that are used here.
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "config/vdefs/LRU.h"

#include "common/basic.h"
#include "common/obj_bitfield.h"
#include "common/xref.h"
#include "types/type_def.h"
#include "runtime/obj.h"
#include "runtime/surr.h"
#include "runtime/value.h"
#include "runtime/stats.h"
#include "runtime/stamps.h"
#include "cache/cache_fetch.h"
#include "client/handle_def.h"
#include "runtime/discover_fix.h"
#include <stddef.h>

/*
  "stddef.h" is included to support macros using "malloc", "free", and
  "sizeof".
*/

/*
   An "ofunc" is an implementation for an ordinary object method. The
   first argument is the method receiver.
*/
typedef void *(*ofunc)(obj, ...);

/*
   A "dtentry" represents an entry in the dispatch table. It may be
   either an ordinary method ("f") or an object ("eobj") that provides
   the implementation of a type extension within the code of this
   object's class.
*/
typedef union {
    ofunc f;
    obj eobj;
} dtentry;

/*
Casting an object to the type "obj" is always valid, and casting
it to "surr" is even more valid.

   A "DV" (a "struct dv_s *") is a dispatch vector.
   The "getAddress" and "getClass" methods
   are supported by "any", which is the universal supertype. Therefore,
   all dispatch vectors contain them. Any additional methods supported
   by a type are found in the "optional" section.

   The three "offset" fields tell how to find various parts of the
   object referencing this dispatch vector. These fields are offsets,
   in bytes, which can be added to the object pointer that was followed
   to find this vector. "boffset" contains the offset to the beginning
   of the object. "offset" contains the offset to the end of the dispatch
   header, where the class dispatch vector is located. "foffset" contains
   the offset to the first field. The "foffset" and "boffset" fields are
   provided for the use of the garbage collector; "offset" is used during
   method dispatch.

   The "t" field is used only in dispatch vectors pointed to by surrogates
   and unswizzled objects. For a surrogate, it contains the type expected for
   the object that will be fetched via that surrogate.  For an unswizzled
   object, it contains the class.  For other objects, "t" is zero. This fact
   is used in the garbage collector's test.  The "offset" field is used to
   find the xref in an empty surrogate.  In a full surrogate, "foffset" is the
   offset to the forwarding pointer.  

\hspace{1in}\begin{tabular}{c|c}
   Swizzled object & dv->t == 0 \\ \hline
   Surr/unswizzled & dv->t == type/class
\end{tabular}

   Empty surrogates, full surrogates, and unswizzled objects can be 
   distinguished from one another by their foffset fields. 

\hspace{1in}\begin{tabular}{c|c}
   Empty surrogate &  foffset = 0 \\ \hline
   Full surrogate &    foffset > 1\\ \hline
   Unswizzled object & foffset = 1
\end{tabular}
   
   For most classes, The field "c" contains the class of this object.
   The field may only be accessed if this is a real object, and not a
   surrogate or unswizzled object.
   If "c" does not contain the object's class, it contains 0.

   If "c" contains the object's class, then this dispatch vector is
   one of the DV's of that class. Unless the object is one of the
   primitive any wrappers: intAny, boolAny, charAny, realAny, nullAny,
   in which case "c" contains "int", "bool", etc., but the dispatch
   vector is owned by "intAny", "boolAny", etc.
*/

struct dv_s {
    int boffset;
    int offset;
    int foffset;
    objtype t;
    class_ c;
    obj (*get_address)(obj);
    class_ (*get_class)(obj);
    /* dtentry optional[0]; */
};

/*

A "core_s" is found at the core of an object. It represents an
object on which field accesses may be performed. For especially large
objects, the "bitfields" field may spill over into additional bitfield
fields, so the actual fields may not start at the point indicated
here. This only happens if "bitfields" matches "OBJ_BF_LONG_BF" (see
"common/obj_bitfield.h").

One bit of the stamp is used to indicate whether the object has been
read.  Another bit of the stamp is used to indicate whether the object
has been used recently. Another bit of stamp is used to indicate whether
the object has been written.
See "runtime/stamps.h" for more information about the
stamps. See also the macros "MARK_AS_USED", "UNMARK_AS_USED",
"FIXUPREAD", and "FIXUPWRITE", below.

The core_s also contains the handle assigned to the object (0 if none)
If the object is persistent, it contains an xref, else the xref field is 0
An "core_s" must be an integral number of words long.

*/

struct core_s {
    DV methods;
    Xref xref;
    int stamp;   
    handle handle_;
    int size;
    Obj_bitfield bitfields;
    /* fevalue fields[0]; */
};

/* Number of slots in the header apart from the dispatch header */
#define FE_obj_headers  3

#define OBJ_XREF(x) \
	(((core)((char *)(x) + ((*(x))->offset)))->xref)
/* Find the xref part of an object.  May be used as an lvalue. */

#define STD_FOFFSET sizeof(struct core_s)
/* For objects that are not surrogates and do not have long bitfields,
   this is the value of "foffset".
*/

/* XXX (above) 
   _We need to ensure that objects have the same layout at the OR and the
   FE.  Currently an FE object is one slot shorter.  We have to decide
   whether we will use that for another dispatch header slot or other 
   information._ */

#define cao(CORE) (&((CORE)->methods))
#define oac(OBJ) ((core)(void *)(OBJ))
/*
    A "core" is also an "obj". Use this macro to convert between them.
    The "oac" macro may only be used in situations where the "obj"
    is known to actually be an "core".
*/

/*
   "t BUMP(void *x)" returns a pointer to end of the dispatch header,
   where the class dispatch vector is located.
   
   Objects no longer need to be BUMPed before a call.
*/
#define BUMP(t,x,dv) ((t)((char *)x + (((DV)dv)->offset)))

/*
   "obj OBJ_START(obj)" returns a pointer to the beginning of the object:
   the first entry in the dispatch header.
*/
#define OBJ_START(x) ((obj)((char *)(x) + ((*(x))->boffset)))

/*
   "Obj_bitfield OBJ_BITFIELD(obj x)" returns the bitfield describing object
   "x".
*/
#define OBJ_BITFIELD(x) \
	 ((core)((char *)(x) + (((*(x))->offset))))->bitfields

/*
  "Obj_bitfield \*OBJ_LONG_BITFIELDS(obj x)" returns a pointer to additional
  bitfields describing object "x". This call only makes sense if the
  value of "OBJ_BITFIELD(x) \& OBJ_BF_LONG_BF" is  "OBJ_BF_LONG_BF".
*/
#define OBJ_LONG_BITFIELDS(x) \
	 (&((core)((char *)(x) + ((*(x))->offset)))->bitfields + 1)

/*
  "int OBJ_NUM_SLOTS(obj x)" returns the number of word-size slots
  starting at "OBJ_FIELDS(x)". These slots are described by the bitfield
  returned by "OBJ_BITFIELD".
*/
#define OBJ_NUM_SLOTS(x) \
	 ((core)((char *)(x) + ((*(x))->offset)))->size

/*
  "fevalue \*OBJ_FIELDS(x)" returns a pointer to the start of the fields of
  the object "x".
*/
#define OBJ_FIELDS(x) ((fevalue *)((char *)(x) + ((*(x))->foffset)))

/* int OBJ_SIZE_BYTE(x) returns the total size of the object in bytes */
#define OBJ_SIZE_BYTES(x) (((*(x))->foffset - (*(x))->boffset)\
			  + OBJ_NUM_SLOTS(x)*sizeof(fevalue))

extern void init_obj_hdr(obj ostart, class_ c);
/*{\sl
  Initialize the stamp (read/write bits), "inf", "size", and "bitfields"
  fields. Also, initialize the dispatch header. "ostart" must point to
  the beginning of the object.
}*/

extern void init_obj_hdr_prim(core ostart, int size, Obj_bitfield, DV methods);
/*{\sl
   A more primitive version of "init_obj_hdr" that can be used
   even before the class object has been initialized. This ability
   is needed to bootstrap the system. "ostart" must point to the beginning
   of the object.
}*/

extern obj get_address(obj o);
/*{\sl
   Invoke the "getAddress" method on "o", with the effect of
   fetching "o" if it is a surrogate. Equivalent to doing
   "o->methods->get_address()".
}*/

extern obj normal_get_address(obj o);
/*{\sl
   The standard "get_address" implementation for non-surrogate objects:
   the identity function. This function is installed in dispatch
   vectors.
}*/

extern class_ normal_get_class(obj o);
/*{\sl
   The standard "get_class" implementation for non-surrogate objects.
   This function is never called directly; it is installed in dispatch
   vectors.
}*/

extern class_ get_obj_class(obj o);
/*{\sl
    Return a class object that describes "o". This calls the "get_class"
    method of "o".
}*/

#define IMPL_GET_CLASS(CLASS)\
    class_ CLASS##_get_class(obj o) {\
        return CLASS;\
    }

extern bool same_obj(obj o1, obj o2);
/* {\sl
    Return whether they are the same object.
}*/
				   
/* The following macro can be used with a core object */
#define IS_PERSISTENT(x) (!IS_NULL_XREF((x)->xref))

/* These macros are for manipulating the read/write bits */

#define MARK_READ_CORE(c)\
              do { (c)->stamp = MARK_READ_PRIM((c)->stamp); } while (0)
#define UNMARK_READ_CORE(c)\
              do { (c)->stamp = UNMARK_READ_PRIM((c)->stamp); } while (0)
#define IS_READ_CORE(c) (IS_READ_PRIM((c)->stamp) != 0)

#define MARK_WRITTEN_CORE(c)\
	  do { (c)->stamp = MARK_WRITE_PRIM((c)->stamp); } while (0)
#define UNMARK_WRITTEN_CORE(c)\
	      do { (c)->stamp = UNMARK_WRITE_PRIM((c)->stamp); } while (0)
#define IS_WRITTEN_CORE(c) (IS_WRITE_PRIM((c)->stamp) != 0)


/* Whether GC has copied an object in fromspace or not */
#define MARK_COPIED_CORE(c)\
	  do { (c)->stamp = MARK_COPIED_PRIM((c)->stamp); } while (0)
#define UNMARK_COPIED_CORE(c)\
	      do { (c)->stamp = UNMARK_COPIED_PRIM((c)->stamp); } while (0)
#define IS_COPIED_CORE(c) (IS_COPIED_PRIM((c)->stamp) != 0)


#if LRU
#define MARK_AS_USED(x) (x)->stamp = MARK_USED_PRIM((x)->stamp);
#else
#define MARK_AS_USED(x) 
#endif
/*{\sl
   Mark object "x" as having been used recently by the current computation.
   To be called by "FIXUPREAD" only.
}*/

#if LRU
#define UNMARK_AS_USED(x) (x)->stamp = UNMARK_USED_PRIM((x)->stamp);
#else
#define UNMARK_AS_USED(x) th_fail("Cannot unmark unless LRU is being used");
#endif 
/*{\sl
   Unmark object "x" so that it does not appear to have been used recently.
   To be called by the LRU purging routines only.
}*/

#if LRU
#define IS_USED_CORE(x) (IS_USED_PRIM((x)->stamp != 0)
#else
#define IS_USED_CORE(x) (th_fail("Cannot use this without LRU", 0)
#endif


extern void mark_obj_as_read(core o);
/*{\sl
   Mark object "o" as being read by the current transaction.
   To be called by "FIXUPREAD" only.
}*/

extern core mark_obj_as_written(core o);
/*{\sl
   Mark object "o" as being written by the current transaction.
   To be called by "FIXUPWRITE" only.
}*/

#define FIXUPREAD(x) if (!IS_READ_CORE((x)))\
                        mark_obj_as_read(x); \
                     MARK_AS_USED(x)
/*{\sl
   Record the fact that object "x" has been read by the current
   transaction. Object "x" must already have been bumped.
}*/

#define FIXUPWRITE(x) if (!IS_WRITTEN_CORE((x)))\
                         mark_obj_as_written(x); \
                      MARK_AS_USED(x)
/*{\sl Record the fact that object "x" has been written by the current
   transaction. Object "x" has already been bumped.
}*/

extern bool is_persistent_object(obj o);
/* effects	Returns true iff object o is persistent. */


#ifdef __cplusplus
}
#endif

#endif /* _OBJ_CLASS_H */
