/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _SURR_H
#define _SURR_H

#ifdef __cplusplus
extern "C" {
#endif

/*
    \section{Surrogate}
    A "surr" is a surrogate for an object at the front end.

   The "t" field is used only in dispatch vectors pointed to by surrogates
   and unswizzled objects. For a surrogate, it contains the type expected for
   the object that will be fetched via that surrogate.  For an unswizzled
   object, it contains the class.  For other objects, "t" is zero. This fact
   is used in the garbage collector's test.  The "offset" field is used to
   find the xref in an empty surrogate.  In a full surrogate, "foffset" is the
   offset to the forwarding pointer.  
   Swizzled object -->        dv->t = 0
   Surr/unswizzled object --> dv->t <> 0

   Empty surrogates, full surrogates, and unswizzled objects can be 
   distinguished from one another by their foffset fields. 
   Empty surrogate -->   foffset = 0
   Full surrogate -->    foffset > 1
   Unswizzled object --> foffset = 1
   */

#include "runtime/surr_def.h"
#include "runtime/obj_class.h"

#include "config/vdefs/EDGE_MARKING.h"

/* The forwarding pointer (obj \*) of a full surrogate.  
   Requires: IS_FULL_SURR(x)
*/
#define SURR_FORWARD(x) ((obj *)((char *)(x) + ((*(x))->foffset)))

/* Get the type object of a surrogate */
#define SURR_TYPE(x) ((*(x))->t)

/* Is the object x a surrogate or unswizzled? */
#define IS_SPECIAL(x) ((*(x))->t != 0)

/* Is the object x actually a surrogate for another object? */
#define IS_SURR(x) (SURR_TYPE(x) != 0 && (*(x))->foffset != 1)

/* Is the object an unswizzled object? */
#define IS_UNSWIZZLED(x) ((*(x))->t != 0 && (*(x))->foffset == 1)

/* Is the pointer x marked to indicate that there is no object
   at the end? */
#define PMARK_MASK (0x1UL << ((sizeof(unsigned long) * byte_bits) - 1))

#define IS_MARKED_POINTER(x) (((long)(x) & PMARK_MASK)?TRUE:FALSE)
extern bool is_marked_pointer(obj x); /* for debugging */

#define MARK_POINTER(x) ((obj)((unsigned long)(x) | PMARK_MASK))
extern obj mark_pointer(obj x);  /* for debugging */

#define UNMARK_POINTER(x) ((obj)((unsigned long)(x) & ~(PMARK_MASK)))
extern obj unmark_pointer(obj x); /* for debugging */

#define ABSENT(x) (IS_MARKED_POINTER(x))
#define PRESENT(x) (!IS_MARKED_POINTER(x))

/* Get the type object as an "obj" */
#define SURR_TYPE_OBJ(x) ((obj)SURR_TYPE(x))

/* Is the object x a full surrogate? Requires IS_SURR(x) */
#ifdef NDEBUG
#define IS_FULL_SURR(x) (IS_FULL_SURR_DEFN(x))
#else 
#define IS_FULL_SURR(x) (assert(IS_SURR(x)),IS_FULL_SURR_DEFN(x))
#endif

/* Don't call this; used only for defining IS_FULL_SURR */
#define IS_FULL_SURR_DEFN(x) ((*(x))->foffset != 0)

/* Fill an empty surrogate.
   Requires: (IS_SURR(s) and !IS_FULL_SURR(s))
*/
extern void surr_fill(surr s, obj x);

/* How many 64-bit slots does a surrogate occupy? */
#define SURR_SLOTS 2   /* XXX this is wrong -- surrogates are variable size */

#ifdef __cplusplus
 }
#endif

#endif /* _SURR_H */
