/* Copyright Barbara Liskov, MIT 1996 */

/*
\subsection{Set of transactions}

A transaction set is implemented as a hash table from OR numbers to
transaction objects.  No other data is stored.

*/

#include "common/xref.h"
#include "common/mdebug.h"
#include "common/ros.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/or_set.h"
#include "runtime/trans_set.h"

implementOpenHashMap(TransSet, OR_num, Transaction *, ORNUM_HASH, ORNUM_EQUAL)

// \subsection{Public operations}

TransactionSet::TransactionSet(int _predict) {
    set = new TransSet;
    predict = _predict;
}

TransactionSet::~TransactionSet() {
    //  All the elements in the set are deleted in commit.cc
    delete set;
}

Ros *TransactionSet::get_ros(OR_num or_num) {
    Transaction *trans;

    if (or_num > 1)
	fprintf(stderr, "Warning (ROS): Or Number may be bad\n");
    if (!set->fetch(or_num, trans))
	trans = make_new_transaction(or_num);

    return trans->ros;
}

Mos *TransactionSet::get_mos(OR_num or_num) {
    Transaction *trans;

    if (or_num > 1)
	fprintf(stderr, "Warning (MOS): Or Number may be bad\n");
    if (!set->fetch(or_num, trans))
	trans = make_new_transaction(or_num);

    return trans->mos;
}

Nos *TransactionSet::get_nos(OR_num or_num) {
    Transaction *trans;

    if (or_num > 1)
	fprintf(stderr, "Warning (NOS): Or Number may be bad\n");
    if (!set->fetch(or_num, trans))
	trans = make_new_transaction(or_num);

    return trans->nos;
}

Transaction *TransactionSet::get_transaction(OR_num or_num) {
    Transaction *trans;

    if (or_num > 1)
	fprintf(stderr, "Warning (Trans): Or Number may be bad\n");
    if (set->fetch(or_num, trans))
	return trans;
    return 0;
}

void TransactionSet::get_participants(OR_set *or_set) {
    TransSet::Bindings bindings(set);

    for (; bindings.ok(); bindings.next())
	or_set->add(bindings.key());
}

// \subsection{Private operations}

Transaction *TransactionSet::make_new_transaction(OR_num or_num) {
    Transaction *trans = new Transaction;

    trans->ros = new Ros(predict);
    // This may be allocating too much space for multi-OR transactions
    // For single-OR transactions, it is taking extra space for MOS objects 
    trans->mos = new Mos; // Not using predict for MOS currently
    trans->nos = new Nos;
    trans->participants = 0;
    //  trans->tid is uninitialized

    set->store(or_num, trans);
    return trans;
}
