/* Copyright Barbara Liskov, MIT 1996 */

/*
\section{Set of transactions}

This abstraction is used at commit time to build up a set of transactions, one
transaction per participant.  The transaction for a participant contains only
those objects which reside at the participant, i.e. those objects with OR
numbers matching the particpant's OR number.

To use the set, retrieve the ros/mos/nos for the appropriate
participant and insert objects.  After all objects have been added,
retrieve the transaction for each participant.  

The transaction id and participant set for transactions created by
operations in this module are uninitialized and should be set after a
transaction is extracted from the set.  
*/

#ifndef _TRANS_SET_H
#define _TRANS_SET_H

#include "common/basic.h"
#include "common/bits.h"
#include "common/or_num.h"
#include "common/openhashmap.h"
#include "common/transaction.h"

class Ros;
class Mos;
class Nos;

declareOpenHashMap(TransSet, OR_num, Transaction *, ORNUM_HASH, ORNUM_EQUAL)

class TransactionSet {
public:    
    TransactionSet(int predict);
    // effects - Create an empty set of transactions.
    //           predict is a prediction of the totalsize of ROS+MOS

    ~TransactionSet();
    // effects - Delete space for set. 
    //           Members of the set are not themselves deleted.

    Ros *get_ros(OR_num or_num);
    // effects - Return ROS for given OR number in current transaction.
    //           Creates new transaction if participant doesn't appear in set.

    Mos *get_mos(OR_num or_num);
    // effects - Return ROS for given OR number in current transaction.
    //           Creates new transaction if participant doesn't appear in set.

    Nos *get_nos(OR_num or_num);
    // effects - Return ROS for given OR number in current transaction.
    //           Creates new transaction if participant doesn't appear in set.

    Transaction *get_transaction(OR_num or_num);
    // effects - Returns transaction for given OR number, or NULL if none
    //           exists.

    void get_participants(OR_set *or_set);
    // effects  - Adds OR numbers of all participants to or_set.
    // modifies - *or_set

private:
    TransSet *set;    // Set of transactions
    int predict;      // Predicted size of ROS + MOS

    Transaction *make_new_transaction(OR_num or_num);
    // effects - Make a new transaction with empty ROS, MOS and NOS,
    //           and add binding from or_num to new transaction.
    //           The Tid and the participant set are NULL.
};

#endif /* _TRANS_SET_H */
