/* Copyright Barbara Liskov, MIT 1996 */

#include <string.h>
#include "runtime/transinfo.h"
#include "runtime/obj_class.h"
#include "runtime/unswiz.h"
#include "runtime/except.h"

#include "fe_config.h"

#include "fe/cache/cache.h"
#include "fe/cache/swiz.h"

#include "common/basic.h"
#include "common/mdebug.h"
#include "common/Timer.h"
#include "common/ros.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/xref.h"
#include "common/sort.h"
#include "common/th_assert.h"
#include "common/transaction.h"
#include "common/oref_set.h"
#include "common/uids.h"
#include "common/xref_set.h"
#include "common/or_set.h"
#include "common/or_index.h"
#include "common/fail.h"

#include "config/vdefs/TRANS_KEEP_COPIES.h"

#define DEBUG_TRANS FALSE
#define DEBUG_SHRINK_OBJECTS FALSE

// Predicted sizes of volatile and persistent object sets and copy sets
#define PREDICT_CSIZE    10000
#define PREDICT_PSIZE    10000
#define PREDICT_VSIZE    10000

// Global variable used in the FE for calling various methods
Transinfo* Fe_Trans;

// Used by make transaction to sort the modified objects
struct modobj {
    Xref xref; // Xref of the modified object
    int index; // Index of the object in the persistent set
};

declareArray(modobjs,modobj)
implementArray(modobjs,modobj)

implementArray(cores,core)

implementOpenHashMap(NewObjTable, ubits64, New_Obj_Name, hash_int, comp_int)

implementArray(or_inv_array, or_inv_info *)

// We need to define comparison operators on "modobj".
inline int cmp(modobj m1, modobj m2) {
    // "or" is more significant than "oref".
    if (m1.xref.or != m2.xref.or)
	return (m1.xref.or - m2.xref.or);
    return OREF_CMP(m1.xref.oref, m2.xref.oref);
}
inline int operator <  (modobj m1, modobj m2) { return (cmp(m1,m2) <  0); }
inline int operator <= (modobj m1, modobj m2) { return (cmp(m1,m2) <= 0); }
inline int operator == (modobj m1, modobj m2) { return (cmp(m1,m2) == 0); }
inline int operator != (modobj m1, modobj m2) { return (cmp(m1,m2) != 0); }
inline int operator >= (modobj m1, modobj m2) { return (cmp(m1,m2) >= 0); }
inline int operator >  (modobj m1, modobj m2) { return (cmp(m1,m2) >  0); }

inline core fast_get_core_and_xref(obj o, Xref& xref) {
    // requires: o is either a surrogate or a core object
    // effects: Same as get_core_and_xref
    if (!IS_SURR(o)) {
	core c = oac(o); // BUMP being avoided here
	xref = c->xref;
	return c;
    }
    return cache_real_obj(o, &xref);
}

Transinfo::Transinfo(): persistent(PREDICT_PSIZE), modified(PREDICT_CSIZE),
    nonpersistent(PREDICT_VSIZE), copy_set(PREDICT_CSIZE) {
	
    inv_info = new or_inv_array(1);
    new_names = new NewObjTable(1);
    new_indices = new OR_Index;
    Abort_Transaction = FALSE;
    char *res = getenv("IMMEDIATE_ABORT");
    // Variable set to true (for debugging) if IMMEDIATE_ABORT has been set
    immediate_abort_on_invalidation = res != 0;
    gc_occurred = FALSE;
    hashed_index = -1;
    read_set = new XrefSet;
}

Transinfo::~Transinfo() {
    delete inv_info;
    delete new_names;
    delete new_indices;
}

void Transinfo::mark_object_read (core c) {
    th_assert(!IS_READ_CORE(c), "Tried to mark a marked object");
    MARK_READ_CORE(c);

    // Do not log non-persistents objects.  However, we set the
    // read bit in stamp for such objects to avoid multiple calls into this
    // procedure. We need to log volatile read objects also else they will
    // never be entered in the next transaction

#if DEBUG_TRANS
    Xref xref = c->xref;
    fprintf(stderr, "Read object %d:%d.%d\n", xref.or,
	    OREF_SEGMENT(xref.oref), OREF_INDEX(xref.oref));
#endif
    if (IS_PERSISTENT(c)) {
	persistent.append(c);
	pxrefs.append(c->xref);
	if (c->xref.or > 1)
	    fprintf(stderr, "Bad OR number = %d\n", c->xref.or);
    }
    else 
	nonpersistent.append(c);
}

core Transinfo::mark_object_written (core c) {
    th_assert(!IS_WRITTEN_CORE(c), "Tried to mark a marked object");

    MARK_WRITTEN_CORE(c);
    // Make a copy of the current object state for modified
    // objects so that it can be reverted to if the transaction aborts
    // XXX Not taking care of creators
    
#if !TRANS_KEEP_COPIES
    if (!IS_PERSISTENT(c)) {
#endif
	// Allocate space for the old copy in the heap
	// Find the size of the whole object and do a memcpy
	int tot_size = OBJ_SIZE_BYTES(cao(c));
	obj copy = cache_oalloc(tot_size);
	memcpy(copy, OBJ_START(cao(c)), tot_size);
	core copy_core = BUMP(core, copy, *copy);
	// The copy should not be marked as read/written
	UNMARK_READ_CORE(copy_core);
	UNMARK_WRITTEN_CORE(copy_core);
	copy_set.append(copy_core);
#if !TRANS_KEEP_COPIES
    } else
	copy_set.append(NULL);
#endif


    modified.append(c);
    if (IS_READ_CORE(c)) {
	// The object has already been entered
	return c;
    }
    mark_object_read(c);     // No blind writes.
    return c;
}

static inline void add_to_ros(Xref xref, int& last_ornum, Ros*& ros,
			      TransactionSet* tset) {
    // requires: last_ornum correspond to an invalid OR number or
    //           ros corresponds to the read object set being built
    //           for the OR last_ornum. tset corresponds to the transaction
    //           information being constructed for the current transaction
    // effects: Adds "xref" to the Ros of the OR, xref.or

    if (xref.or != last_ornum) {
	ros = tset->get_ros(xref.or);
	last_ornum = xref.or;
    }
    ros->add_obj(xref.oref);
}

static void lookup_core_and_xref(obj o, core& c, Xref& xref) {
    // requires: If o is non-NULL, o refers to a valid object/surrogate
    //           else xref refers to a valid xref
    // effects: Looks up the object in the swizzle table (if necessary) and
    //          fills out the information in xref and c (if object is present)

    c = NULL;
    if (o) {
	c = fast_get_core_and_xref(o, xref);
    } else {
	// The returned value is an xref. Check if we can get the object
	o = swiz_get_object(xref);
	if (o) {
	    c = BUMP(core, o, *o);
	    th_assert(XREF_EQUAL(xref, c->xref),
		      "Xref in object and RW set different");
	}
    }
}

bool Transinfo::make_transaction (TransactionSet* tset, OR_num *coord, 
				  cores* nobjs) {
    th_assert(new_names->size() == 0, "New Object table is not empty");
    bool need_to_send_to_or = FALSE;

    // All elements in the persistent set are unique because objects
    // that were shrunk and reread do not get added multiple times;
    // at fetch time, "mark_read_object_if_shrunk" marks the refetched object

    obj o;
    Ros *ros;
    int last_ornum = -1; // To optimize the get_ros calls

    modobjs mods(persistent.size()/5);
    // Set of modified objects that exist. We are predicting a modified
    // object size to be 20% of the touched objects

    RW_Iter iter(this, RW_ITER_PERSISTENT);
    Xref xref;
    while (iter.get(o, xref)) {
	th_assert(o != 0 || !IS_NULL_XREF(xref),
		  "Read/written object and its xref are NULL");
	need_to_send_to_or = TRUE;
	core c;
	th_assert(o || gc_occurred, "Object is NULL and GC did not occur");
	lookup_core_and_xref(o, c, xref);
	th_assert(xref.or, "Bad OR number received for read/written object");

	if (!c || !IS_WRITTEN_CORE(c)) {
	    // Object was not written
	    add_to_ros(xref, last_ornum, ros, tset);
	} else {
	    // We are breaking the iterator's abstraction here
	    modobj m;
	    m.xref = xref;
	    m.index = iter.index - 1 ;
	    mods.append(m);
	    UNMARK_WRITTEN_CORE(c);
	}
	if (c) {
	    th_assert(IS_PERSISTENT(c), "Non-persistent object received");
	    UNMARK_READ_CORE(c);
	}
    }

    // Generate the MOS now
    // Sort the mods array by increasing xref.
    int msize = mods.size();
    if (msize > 0)
	quick_sort(mods.as_pointer(), mods.size());

    last_ornum = -1;
    for (int i = 0; i < msize; i++) {
	Mos *mos;
	Xref xref = mods[i].xref;
	// The mods[i].index object in the persistent set may be NULL
	// In that case, get the object from the swizzle table
	core c = persistent[mods[i].index];
	if (!c) {
	    o = swiz_get_object(xref);
	    th_assert(o, "Modified object has been flushed out of the cache");
	    c = BUMP(core, o, *o);
	}
	if (last_ornum != xref.or) {
	    mos = tset->get_mos(xref.or);
	    last_ornum = xref.or;
	}
	Obj_Handle h = mos->add_object(xref.oref, c->size);
	unswizzle_object(tset, mos, h, xref.or, c, nobjs, new_names);
    }

    *coord = find_coordinator_and_sort_ros(tset); // Find coordinator.
    gc_occurred = FALSE;
    return need_to_send_to_or;
}

void Transinfo::sent_transaction(TransactionSet* tset, OR_set *participants) {
    // Objects will be coming back from coordinator in increasing OR number
    // order.  We've stored new objects in a linear array; we need to 
    // know the index of the first object at each participant in this array.
    
    new_indices->clear();
    int max_index = 0;
    OR_num or_num;

    OR_set::SortedElements gen(participants);
    while (gen.get(or_num)) {
	// Find number of new objects at this OR
	Nos *nos = tset->get_nos(or_num);
	th_assert(nos != 0, "Missing NOS in transaction set");
	
	new_indices->store(or_num, max_index);
	max_index += nos->count();
    }
}

void Transinfo::complete_transaction(TransEnv *tenv) {

    // Note: The relevant objects have been shrunk by the invalidation
    //        processing code.

    bool committed = tenv->committed;
    cores* nobjs = tenv->nobjs;
    Xrefs* xrefs = tenv->xrefs;
    Uids* uids = tenv->uids;

    hashed_index = -1;
    read_set->clear();
    if (!committed)
	reset_modified_objects();

    // Clear the transaction stamp information for all read/written objects.
    // Have to reset the copy index of all modified objects

    // If transaction committed, then the persistent objects must have been
    // unmarked, so do not need to iterate over them
    RW_Iter_Option opt = committed? RW_ITER_VOLATILE: RW_ITER_ALL;
    RW_Iter iter(this, opt);
    obj o;
    Xref xref;
    while (iter.get(o, xref)) {
	core c;
	th_assert(o || gc_occurred, "Object is NULL and GC did not occur");
	lookup_core_and_xref(o, c, xref);
	if (!c) continue;

	// Note that we do not unmark the USED bit of stamp --
	// only the shrinking mechanism does that.
	UNMARK_WRITTEN_CORE(c);
	UNMARK_READ_CORE(c);
    }
    trim_core_set(&persistent, PREDICT_PSIZE);
    trim_core_set(&nonpersistent, PREDICT_VSIZE);
    trim_core_set(&copy_set, PREDICT_CSIZE);
    trim_core_set(&modified, PREDICT_CSIZE);
    pxrefs.clear(); // We are not doing the same as for core_set

    // If the transaction committed, update the xref field
    // Note that this has to be called after the clearing of the
    // transaction information so that the xref information is not cleared
    if (committed) {
	update_newobjs(nobjs,xrefs, uids);
    }

    // Clear the new names table
    new_names->clear();
    Abort_Transaction = FALSE;
}

void Transinfo::abort_transaction(bool perform_longjmp) {
    TransEnv tenv;
    tenv.committed = FALSE;
    complete_transaction(&tenv);
    if (perform_longjmp)
	SIGNAL_ABORT();
}

// \subsection{Internal Methods}

void Transinfo::trim_core_set (cores* core_set, int predict_size) {
    int size = core_set->size();
    if (size > predict_size) {
	core_set->remove (size - predict_size);
	core_set->reclaim();
    }
    core_set->clear();
}

void Transinfo::reset_modified_objects() {
    // Transaction has aborted.
    // Revert the modified objects to their earlier state

    RW_Iter copy_iter(this, RW_ITER_COPY);
    RW_Iter modified_iter(this, RW_ITER_MODIFIED);
    obj o, copy;
    Xref copy_xref, mod_xref;
    while (copy_iter.get(copy, copy_xref) && modified_iter.get(o, mod_xref)) {
	th_assert(o, "Original object turned NULL");
	core c = fast_get_core_and_xref(o, mod_xref);
	if (!obj_in_heap(cao(c))) {
	    // If the object is not in the heap then it is a meta object, do
	    // not revert it. Note: if we are modifying c in the rest of the
	    // loop, we must do the same here also
	    continue;
	}
	core copy_core;
	if (!copy) {
	    copy_core = NULL;
#if TRANS_MODIFIED_COPIES
	    // Copy may have been reverted to NULL by GC
	    th_assert(gc_occurred, "Object copy is NULL without GC happening");
#endif
	}
	else
	    copy_core = fast_get_core_and_xref(copy, copy_xref);
	// If invalidation code has shrunk either the modified
	// version or original version, we cannot revert the object
	// If object is present and copy is not, then invalidate the object
	if (!copy_core && c) {
#if DEBUG_SHRINK_OBJECTS
	    fprintf(stderr, "GC has shrunk modified copy (Persistent = %d)"
		    "%d:%d.%d\n", IS_PERSISTENT(c), copy_xref.or, 
		    OREF_SEGMENT(copy_xref.oref), OREF_INDEX(copy_xref.oref));
#endif
	    cache_obj_invalidate(cao(c));
	}
	// Original object can be NULL due to invalidation
	if (!copy_core || !c) continue;
#if DEBUG_SHRINK_OBJECTS
	    fprintf(stderr, "Reverting object to modified copy (Persistent = %d)"
		    "%d:%d.%d\n", IS_PERSISTENT(c), copy_xref.or, 
		    OREF_SEGMENT(copy_xref.oref), OREF_INDEX(copy_xref.oref));
#endif

	// Objects should have same size/bitfields as before
	th_assert(c->size == copy_core->size, "Size of object changed"); 
	th_assert(c->bitfields == copy_core->bitfields,
		  "Bitfields of volatile object changed");
	memcpy(OBJ_START(cao(c)), OBJ_START(cao(copy_core)),
	       OBJ_SIZE_BYTES(cao(c)));
    }
}

void Transinfo::update_newobjs (cores* nobjs, Xrefs* xrefs, Uids* uids) {
    New_Obj_Name new_name;
    int size = nobjs->size();
    int base;
    int index;
    or_obj_descriptors *descs;

    th_assert(xrefs->size() == size, "wrong number of xrefs in commit reply");
    th_assert(uids->size()  == size, "wrong number of uids in commit reply");
    
    // Allocate descriptor for created objects for possible
    // later eviction.
    if (size) {
        descs = new or_obj_descriptors(size);
    }

    for (int i=0; i<size; i++) {
	core c = nobjs->slot(i);

	// Find temporary name of object
	if (!new_names->fetch((ubits64) c, new_name))
	    fail("New object missing from table");

	if (!new_indices->fetch(NEW_NAME_OR(new_name), base))
	    fail("OR missing from new object table");
	
	index = base + NEW_NAME_INDEX(new_name);

	c->xref = xrefs->slot(index);
	th_assert(!IS_NULL_XREF(c->xref),
		  "Xref of newly persistent object NULL");

	obj ostart = OBJ_START(cao(c));

        // Fill descriptor entry to allow posterior eviction
        // of this object.
	or_objdesc rec;
	rec.o =  c->xref.oref;
	rec.objsize = c->size;
        descs->append(rec);

        // Register new object in swizzle table.
	enter_nobj_info (ostart, xrefs->slot(index));
    }

    if (size) 
        subpage_fifo.insert(xrefs->slot(index).or, size, descs);
}

OR_num Transinfo::find_coordinator_and_sort_ros(TransactionSet *tset) {
    // Method of picking coordinator:
    // 1) Find OR which maximizes | MOS | + | ROS | for objects at the
    //    OR for this transaction.
    // 2) If no OR has any modified or new objects, 
    //    return 0 (transaction is read only).

    OR_num max_write_or = 0, or_num;
    int    write_objs = 0;

    OR_set ors;
    tset->get_participants(&ors);
    OR_set::Elements gen(&ors);

    while (gen.get(or_num)) {
	Transaction *trans = tset->get_transaction(or_num);
	trans->ros->sort();
	if (!max_write_or)
	    max_write_or = or_num;
	int msize = trans->mos->count();
	int count = trans->ros->size() + msize; 
	if (msize && count > write_objs) {
	    write_objs = count;
	    max_write_or = or_num;
	}
    }
    return max_write_or;
}

// \subsection{The RW Iterator}


void Transinfo::correct_non_persistent_objects() {
    int size = nonpersistent.size();
    for (int i = 0; i < size; i++) {
	obj forwarded = cache_forwarded(cao(nonpersistent[i]));
	if (forwarded == 0)
	    nonpersistent[i] = NULL;
	else
	    nonpersistent[i] = oac(forwarded);
    }
}

Transinfo::RW_Iter::RW_Iter(Transinfo* tinf, RW_Iter_Option opt) {
    index = 0;
    set_index = 0;
    transinfo = tinf;
    no_sets = 0;
    sets[0] = sets[1] = sets[2] = sets[3] = NULL;
    persistent_set = FALSE;
    int i, copy_size;
    switch (opt) {
    case RW_ITER_GC:
	sets[0] = &transinfo->modified;
	sets[1] = &transinfo->copy_set;
	// GC needs to contain the modified set so that the pointers can be
	// replaced appropriately
	// The non-persistent array will be scanned and corrected in
	// "correct_non_persistent_objects"
	no_sets += 2;
	transinfo->gc_occurred = TRUE;
	// Blank out the persistent copies in the copy array
	bzero((char *)transinfo->persistent.as_pointer(),
	      transinfo->persistent.size()*sizeof(core));
	copy_size = transinfo->copy_set.size();
	for (i = 0; i < copy_size; i++) {
	    core* rec = &transinfo->copy_set[i];
	    if (*rec && (IS_SURR(cao(*rec)) || IS_PERSISTENT(*rec)))
		*rec = NULL;
	}
	break;
    case RW_ITER_ALL_SETS:
	sets[0] = &transinfo->nonpersistent;
	sets[1] = &transinfo->persistent;
	sets[2] = &transinfo->copy_set;
	sets[3] = &transinfo->modified;
	no_sets += 4;
	break;
    case RW_ITER_ALL:
	sets[0] = &transinfo->nonpersistent;
	sets[1] = &transinfo->persistent;
	no_sets += 2;
	break;
    case RW_ITER_COPY:
	sets[0] = &transinfo->copy_set;
	no_sets++;
	break;
    case RW_ITER_MODIFIED:
	sets[0] = &transinfo->modified;
	no_sets++;
	break;
    case RW_ITER_PERSISTENT: 
	sets[0] = &transinfo->persistent;
	persistent_set = TRUE;
	no_sets++;
	break;
    case RW_ITER_VOLATILE: 
	sets[0] = &transinfo->nonpersistent;
	no_sets++;
	break;
    }
    // Note: sets[set_index] should not be NULL here
    th_assert(sets[set_index] && no_sets, "sets is NULL or no_sets is 0");
    core_set = sets[set_index]->as_pointer();
    size = sets[set_index]->size();
}

bool Transinfo::RW_Iter::get_from_next_set(obj& o, Xref& xref) {

    set_index++;
    if (set_index >= no_sets) {
	size = 0;
	return FALSE;
    }
    core_set = sets[set_index]->as_pointer();
    persistent_set = FALSE;
    if (core_set == transinfo->persistent.as_pointer())
	persistent_set = TRUE;
    size = sets[set_index]->size();
    index = 0;
    return get (o, xref);    // Note: Recursive call on get
}

void Transinfo::RW_Iter::replace(obj new_obj) {
    th_assert(new_obj != 0, "null object being put in r/w set");
    th_assert(index > 0, "replacing an object before fetching from r/w set");

    core c = BUMP(core, new_obj, *new_obj);
    core_set[index-1] = c;
}

void mark_obj_as_read (core c) {
    Fe_Trans->mark_object_read(c);
}

core mark_obj_as_written (core c) {
    return Fe_Trans->mark_object_written(c);
}

void free_transinfo_space () {
    delete Fe_Trans;
}

bool is_persistent_object (obj o) {
    if (IS_SURR(o))
	return TRUE;
    core c = BUMP(core, o, *o);
    return (IS_PERSISTENT(c));
}
