/* Copyright Barbara Liskov, MIT 1996 */

#ifndef _TRANSINFO_H
#define _TRANSINFO_H

#include "common/array.h"
#include "common/basic.h"
#include "common/bits.h"
#include "common/oref.h"
#include "common/or_num.h"
#include "common/openhashmap.h"
#include "common/modset.h"
#include "common/xrefs.h"
#include "runtime/trans_set.h"
#include "runtime/obj_class.h"
#include "runtime/invalidation.h"
#include "cache/cache.h"

class Transaction;
class Uids;
class NewObjTable;
class OR_Index;
class obj_copies;
class XrefSet; 

// Iterator options to yield the appropriate type of objects
typedef enum {
    RW_ITER_COPY,             // For the copies of objects
    RW_ITER_MODIFIED,         // for the corresponding modified objects
    RW_ITER_PERSISTENT,
    RW_ITER_VOLATILE,
    RW_ITER_ALL,    // Yield the persistent and non-persistent set
    RW_ITER_GC,     // With this option, references in copies of
    // objects are also yielded
    RW_ITER_ALL_SETS    // Yield the persistent and non-persistent set
} RW_Iter_Option;

// Number of sets that can be yielded in the RW_ITER
#define YIELD_SETS       4


// Hash table from core pointer to temporary object name, used during
// unswizzling.
declareOpenHashMap(NewObjTable, ubits64, New_Obj_Name, hash_int, comp_int)    


declareArray(cores, core)

// Per-OR invalidation information
typedef struct {
    OR_num or_num; 
    int last_inv_num;        // Last invalidation message number received by FE
} or_inv_info;

declareArray(or_inv_array, or_inv_info *)

// Structure used by the network code to receive a commit/abort reply
struct TransEnv {
    cores* nobjs; // Poinetrs to new objects created in this transaction
    Xrefs* xrefs;  // Place where the xrefs obtained must be stored
    Uids* uids;  // Place where the uids obtained must be stored
    bool read_only; // Indicates if transaction is readonly
    bool committed; // Whether the transaction committed or not
};

// This class has the methods for handling the transaction information at
// the FE. A global variable "Fe_Trans" is maintained. Handling of
// read/write sets, invalid object sets etc is done by this class.

class Transinfo {
public:
    Transinfo();
    // effects Default constructor. Creates an initialized object
    
    ~Transinfo();
    // effects Default destructor
    
    void mark_object_read(core c);
    // effects Marks the current object as read and enters it into the
    //         persistent or non-persistent object set (as is relevant)
    
    core mark_object_written(core c);
    // effects Marks the current object as written and enters it into the
    //         persistent or non-persistent object set (as is relevant)
    //         For non-persistent objects, allocates space and makes a copy
    //         of the current object state and stores it in the xref.
    
    bool must_abort();
    // effects: Returns TRUE if the current transaction must be aborted
    //          because of some invalidation message that was earlier received

    int predicted_size();
    // effects: Returns a prediction of the number of objects in the
    //          ROS and MOS (approximation)

    void correct_non_persistent_objects();
    // requires: This method is called just after the GC has taken place
    // effects:  Corrects the pointers in the nonpersistent set to point
    //           into the TOSPACE. If the object has not been copied, the
    //           corresponding entry is set to NULL


    bool make_transaction(TransactionSet* tset, OR_num *coord, 
			  cores* nobjs);
    // requires nobjs not be NULL
    // modifies	tset, coord
    // effects	Fills in the ros/mos/nos/coordinator components of tset.
    //          Guarantees that the Ros/Mos of all the participants have been
    //          placed in increasing order of Orefs (i.e. sorted)
    //		nobjs keeps track of new object ptrs.
    //          Sets *coord to OR number of coordinator.
    //		Returns true iff the transaction modified/read any
    //          persistent objs.

    void sent_transaction(TransactionSet* tset, OR_set *participants);
    // requires Given transaction has been sent, but response hasn't come back.
    //          participants is set of participants for transaction.
    // effects  Perform computation that must be done before commit succeeds,
    //          but which may be put off until after request has been sent.
    //          Currently computes new object offsets for multiple-OR 
    //          transactions.

    void abort_transaction(bool perform_longjmp);
    // effects aborts the current transaction. Resets the transaction state
    //         for the next transaction. If perform_longjmp is TRUE, this
    //         method makes a longjmp to the client interface code
    
    void complete_transaction(TransEnv* tenv);
    // requires to be called after the transaction result is known
    //          tenv->nobjs is a pointer to array that was returned by
    //          make_transaction (and has not been modified)
    // effects	tenv->committed refelcts the status of the transaction
    //          Clear the information maintained for this transaction
    //          (read/write sets, copies maintained 
    //          for volatile/persistent objects modified in this transaction).
    //          Based on the "tenv->committed" value, it updates the volatile
    //          objects (using tenv->xrefs/tenv->uids) or reverts their state.
    //          Space associated with nobjs is NOT freed
    //          Reads and writes of objects from this point on will be
    //          considered part of the next transaction.
    
    bool invalidate_objects(OR_num or, const Oref* inv_orefs, int setsize,
			    ubits32 msg_start, ubits32 msg_end,
			    Msg_Wait_Type wait_type);
    // effects: Invalidates the objects whose orefs are specified in
    //          "inv_orefs". Setsize is the size of the
    //          array inv_orefs. msg_start and msg_end indicate the
    //          invalidation message numbers to which these
    //          inv_orefs correspond. Note that this method does not
    //          abort the transaction but sets it up so that it can
    //          be aborted later (by calling abort_transaction_if_needed)
    //          wait_type is used for debugging purposes to determine where the
    //          call is from
    // returns: TRUE iff the transaction must be aborted
    
    // Class for iterating over the contents (persistent and non-persistent
    // objects) of the read/write set.
    
    class RW_Iter {
	//  requires initialize_transaction_info has been called
	
	//  Once created, an iterator must be used and destroyed before any 
	//  objects are added to the read/write set.
	//  The effect is undefined if an iterator method is called after
	//  such a change. The transinfo set should not be modified also

    public: 
	RW_Iter(Transinfo* transinfo, RW_Iter_Option opt);
	// effects Return an iterator over the read/write set in transinfo
	//         if opt is RW_ITER_PERSISTENT, yields only persistent objs
	//         if opt is RW_ITER_VOLATILE, yields only nonpersistent objs
	//         if opt is RW_ITER_ALL,  yields all objs
	//         if opt is RW_ITER_GC, yields all read/written objects
	//         and references in old copies of modified objects
	
	bool get(obj& o, Xref& xref);
	// modifies "o", xref
	// effects  Sets "o" to the next object in the appropriate
	//          read/write set and returns true.  Returns false if
	//          there are no more objects.  o can be a regular or a
	//          surrogate object (i.e. it could have been shrunk)
	//          or an Xref. If it is not an object (i.e. an xref),
	//          o is NULL and xref contains the object's xref
	//          Note: Exactly one of o or xref is valid (and xref is valid
	//          only if o is NULL)
	
	bool get_from_next_set(obj& o, Xref& xref);
	// requires: This method is called only by RW_Iter::get
	// effects:  Sets the iterator to proceed to the next set and then
	//           calls "get" to yield the next element from that set.
	//           The returned value and the value of "o", xref are set as
	//           specified in get.

	void replace(obj o);
	// requires An object has been extracted from the iterator.
	// effects  Replaces last extracted object with "o".

	int index;          // Index of next element to be yielded.
	// Made public so that Transinfo can access it in make_transaction

    private:
	Transinfo* transinfo;
	cores* sets[YIELD_SETS]; // Core sets from which elements have to be
	// yielded. The sets to be yielded are put from 0 onwards. If all
	// sets are not being yielded, the remaining entries are NULL
	int no_sets;        // No. of sets in "sets" that are non-null
	int set_index;      // Current set being yielded
	core* core_set;     // Current set being interated over
	int size;           // Size of core_set
	// Invariant: index is <= size of object_set, set_index < YIELD_SETS
	// Also core_set == sets[set_index] 
	bool persistent_set; // If the current set being yielded is the
	// persistent set
    };
    
private:
    // Representation

    friend RW_Iter;

    // Read/Write sets
    Xrefs pxrefs;           // Xref of persistent[i]
    cores persistent;       // Persistent objs read/written by transaction
    // persistent[i] may be NULL. In which case we use pxrefs[i] for ROS/MOS
    cores nonpersistent;    // Non-persistent objs written by transaction
    cores copy_set;         // Copies of modified objects
    cores modified;         // This is an array corresponding to copy_set
    // It keeps track of pointers to objects that are being modified.
    // Invariants: copy_set->size() == modified->size()
    //             copy_set->slot(i) is the original copy of the modified
    //             object modified->slot(i)

    // Individual elements of the above sets should be accessed only
    // using RW_iter

    XrefSet *read_set; // Subset of objects that have been read by the current
    // transaction. This hash table is present to speed up invalidations and
    // is constructed on receiving the first invalidation in the current trans
    int hashed_index; // Index into the persistent set till where we have
                          // constructed the hash table 
    

    // Keeps track of objects that have been shrunk at least once in the
    // current transaction. This set is used for checking read objects that
    // have been shrunk (and were not reread, i.e. the stamp bits are not
    // set or there is no copy).

    // Invalidation information received during lifetime of current trans
    or_inv_array *inv_info;   

    NewObjTable *new_names;  // Temporary names of new objects in
			     // transaction. It is cleared after a
			     // transaction's work has been completed
    OR_Index    *new_indices;  // Index of new objects for a particular OR
                               // into array of all new objects.

    bool Abort_Transaction;  // Boolean that is set by invalidation message
    // code in case the transaction has to be aborted.

    bool immediate_abort_on_invalidation;
    // When this variable is TRUE, the transaction aborts on receiving an
    // invalidation message

    bool gc_occurred; // Indicates whether GC occurred in the current trans
    // Used for debugging purposes only

    // Internal Methods

    void reset_modified_objects();
    // effects	Reset modified objects because the transaction aborted.
    //          

    void update_newobjs (cores* nobjs, Xrefs* xrefs, Uids* uids);
    // requires xrefs and uids are in increasing OR number order.  Within
    //          each OR number, xrefs and uids are in same order as they were
    //          sent to each participant.
    // effects	Transaction has committed. Enter nobjs into swizzle table and
    //		also update their Xref fields.

    void trim_core_set (cores* core_set, int predict_size);
    // requires core_set points to a valid array
    // effects Clears the core_set size to zero but keeps space allocated 
    //         to it for size at least min(core_set->size(), predict_size)

    OR_num find_coordinator_and_sort_ros(TransactionSet *tset);
    // requires Object sets of transactions in tset have been initialized.
    // effects  Selects and returns OR for given transaction set.  
    //          If transaction is read only, returns 0.
    //          Also sorts the Ros of each participant

    or_inv_info* find_inv_info(OR_num or_num);
    // effects  Return invalidation information for given OR.  Create new
    //          invalidation information if OR not present.
};

extern void free_transinfo_space ();
// effects    Frees the space associated with Fe_Trans

// Global variable used in te FE for calling various transinfo methods
extern Transinfo* Fe_Trans;


int inline Transinfo::predicted_size() {
    return persistent.size();
}

inline bool Transinfo::must_abort() {
    return Abort_Transaction;
}

inline bool Transinfo::RW_Iter::get(obj& o, Xref& xref) {

    o = NULL;
    SET_NULL_XREF(xref);
    if (index < size) {
	// At this point index < sets[set_index]->size()
	o = cao(core_set[index]);
	if (persistent_set && !o)
	    xref = transinfo->pxrefs[index];
	index++;
	return TRUE;
    }
    return get_from_next_set(o, xref);
}

inline core get_core_and_xref(obj o, Xref& xref) {
// effects: determines o's xref (which could be the null xref) and puts it
//          into "xref". Also looks up the swizzle table to check if the
//          actual object exists in the cache. If it is present, returns a
//          pointer to core object, else returns
// Note:    This function has the same functionality as cache_real_obj
//          It has been split to optimize the non-surrogate case
    if (!IS_SURR(o)) {
	core c = BUMP(core, o, *o);
	xref = c->xref;
	return c;
    }
    return cache_real_obj(o, &xref);
}

inline bool is_read (obj o) {
    // effects	Returns true iff object o has been read by current trans
    core c = BUMP(core, o, *o);
    return (IS_READ_CORE(c));
}

inline bool is_modified(obj o) {
    // effects	Returns true iff object o has been modified by current trans
    core c = BUMP(core, o, *o);
    return (IS_WRITTEN_CORE(c));
}

#endif /* _TRANSINFO_H */
