/* Copyright Barbara Liskov, MIT 1996 */

#include <string.h>

#include "config/vdefs/EDGE_MARKING.h"

#include "common/or_obj.h"
#include "common/mdebug.h"
#include "common/th_assert.h"
#include "common/objrefs.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/transaction.h"
#include "common/class_code.h"

/* fe includes */
#include "runtime/commit.h"
#include "runtime/obj_class.h"
#include "runtime/surr_xref.h"
#include "runtime/transinfo.h"
#include "runtime/surr.h"
#include "runtime/unswiz.h"

#include "cache/local_cacheval.h"

static class_code get_class_code(obj o) {
    // effects	Return class code for the class of "o".

    obj class_object = (obj) get_obj_class(o);
    Xref class_xref = (BUMP(core, class_object, *class_object))->xref;

    // This assertion will be wrong when we get user-defined
    // classes in the system.
    th_assert(class_xref.or == 0, "Unexpected OR for class");

    // XXX We need a real map from "class_xref" to "class_code".
    return OREF_INDEX(class_xref.oref);
}

int size_as_orobj(core o) {
    th_assert(!OBJ_BF_ISLONG(o->bitfields), "cannot handle long bitfields");
    return OR_obj_headers + o->size;
}

void unswizzle_object(TransactionSet* tset, Modset* mset, Obj_Handle handle,
		      OR_num or_num, core c1, cores* nobjs, NewObjTable *table)
{
    obj o1 = cao(c1);

    // Initialize the object prior to unswizzling the refs.
    OR_slot* or_rep = (OR_slot*) OBJ_FIELDS(o1);
    mset->init_object(handle, get_class_code(o1), c1->bitfields, c1->size, or_rep);

    // Now unswizzle all refs.
    Object_References gen = Object_References(c1->bitfields, c1->size);
    int slotno;

    // Iterate through all the slots of the object and unswizzle them
    // Add objects to NOS set if necessary
    while (gen.get(slotno)) {
	Xref x;
	obj o = (obj) or_rep[slotno].pointer;
	core c = get_core_and_xref(o, x);
	if (!IS_NULL_XREF(x)) {
	    mset->set_xref(handle, slotno, x);
	    continue;
	}

	// Handle a new object. c is now an actual object.
	New_Obj_Name new_name;

	// If we haven't yet encountered this object, make new name
	if (!table->fetch( (ubits64) c, new_name)) {
	    // Enter into new object set and record in list of new objects.

	    // XXX We place a new object at the OR of the first object
	    // we find that references it.  A better placement policy is
	    // needed.
	    Nos *nos = tset->get_nos(or_num);
	    Obj_Handle new_handle = nos->add_object(c->size);
	    nobjs->append(c);

	    // The temporary name of this new object is just the
	    // index into the list of new objects for this OR, plus the OR #.
	    new_name = NEW_NAME_MAKE(or_num, nos->count()-1);

	    // Store entry for this object's new name
	    table->store( (ubits64) c, new_name);
	    
	    // Recursively unswizzle the new object
	    unswizzle_object(tset, nos, new_handle, or_num, c, nobjs, table);
	}

	// New object is now in the NOS set and is swizzled.

	// Record in the tx, that this slot should be updated to
	// contain the xref of the new object once the xref has been 
	// assigned at the OR.
	mset->set_new_link(handle, slotno, new_name);
    }
}
