#include <assert.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/wait.h>
#include "client/shm_connect.h"

#include "config/vdefs/USE_SIGS.h"
#include "config/vdefs/NDEBUG.h"

extern "C" unsigned usleep(unsigned);

ShmServer *server;
int shmid;

#ifndef RW
#define RW 0
#endif

int const MAXBUF = 256; // must be a power of two

#if RW
#define RANDSIZE(j) (MAXBUF/(1 + (((j) ^ 0x5555) & (MAXBUF - 1))))
#else
#define RANDSIZE(j) j
#endif

void *get_shared_mem(size_t bytes)
{
    shmid = shmget(IPC_PRIVATE, bytes, IPC_CREAT | 0777);
    if (shmid < 0) {
	fprintf(stderr, "Couldn't get shared memory\n");
	exit(EXIT_FAILURE);
    }
	
    void *ret = shmat(shmid, 0, 0);
    if (-1 == (ptrdiff_t)ret) {
	fprintf(stderr, "Couldn't attach shared memory\n");
	exit(EXIT_FAILURE);
    }
    return ret;
}

void destroy_shared_mem(void *addr, int shmid)
{
    shmdt((char *)addr);
    shmctl(shmid, IPC_RMID, 0);
    fprintf(stderr, "Destroyed shared memory\n");
}

void catch_int(int dummy)
{
    destroy_shared_mem(server, shmid);
    exit(EXIT_FAILURE);
}
		     
void *operator new(size_t s, void *addr)
{
    return addr;
}

int wbuf[MAXBUF], rbuf[MAXBUF];

void main(int argc, char **argv)
{
    int bufsize = MAXBUF;
    if (argc == 1) {
	fprintf(stderr, "Usage: %s <iters> [sleeplen [bufsize]]\n", argv[0]);
	exit(EXIT_FAILURE);
    }
    if (argc > 3) bufsize = atoi(argv[3]);
    server = new (get_shared_mem(sizeof(ShmServer))) ShmServer(bufsize);
    int i;
    int n = atoi(argv[1]);
    for (i = 0; i < MAXBUF; i++) wbuf[i] = i;
    if (fork() != 0) {
	signal(SIGINT, catch_int);

	ShmClient *client = server->asClient();
	int j, i2;
	assert(client->looksOk());
	client->beClient();
	for (j = 0; j<n; j++)  {
	    i2 = client->full_get();
#if RW
#ifndef NDEBUG
	    if (i2 < 0 || i2 > MAXBUF) {
		fprintf(stderr, "Got batch of %d words!\n", i2);
		exit(EXIT_FAILURE);
	    }
#endif
	    client->read(rbuf, i2);
#ifndef NDEBUG
	    for (i = 0; i<i2; i++) assert(rbuf[i] == i);
#endif
#endif
	    i2 = RANDSIZE(j);
	    client->put(i2);
#if RW
	    client->write(wbuf, i2);
#endif
	}
	fprintf(stderr, "Received all values\n");

    } else {
	int j, i2;

	server->beServer();
	for (j = 0; j<n; j++)  {
	    i2 = RANDSIZE(-j);
	    server->put(i2);
#if RW
	    server->write(wbuf, i2);
#endif
	    i2 = server->full_get();
#if RW
#ifndef NDEBUG
	    if (i2 < 0 || i2 > MAXBUF) {
		fprintf(stderr, "Got batch of %d words!\n", i2);
		exit(EXIT_FAILURE);
	    }
#endif
	    server->read(rbuf, i2);
#ifndef NDEBUG
	    for (i = 0; i<i2; i++) assert(rbuf[i] == i);
#endif
#endif
	}

	shmdt((char *)server);
	exit(0);
    }
    int *dummy;
    (void)wait(dummy);
    destroy_shared_mem(server, shmid);
}
